function cnetwork = GetChirpletNetwork(C,cgraphparam,coarsestScale,fmin,fmax)
% GetChirpletNetwork -- fills in arc costs into the chirplet graph using
%                       the chirplet coefficients
%
%  Usage
%    cnetwork = GetChirpletNetwork(C,cgraphparam,coarsestScale,fmin,fmax)
%    cnetwork = GetChirpletNetwork(C,cgraphparam,coarsestScale,fmin)
%    cnetwork = GetChirpletNetwork(C,cgraphparam,coarsestScale)
%    cnetwork = GetChirpletNetwork(C,cgraphparam)
%    cnetwork = GetChirpletNetwork(C)
%  Inputs
%    C           Chirplet coefficient table as returned by ChirpletTransform.
%    cgraphparam Chirplet graph parameters as returned by GetChirpletGraphParam. 
%                If omitted or set to [], the default value as returned by 
%                GetChirpletGraphParam is used.
%    coarsestScale
%                to restrict the chirplet graph to scales
%                as coarse as coarsestScale and finer.
%                Cannot correspond to a coarser scale than is set in cgraphparam.
%                If omitted, the coarsest scale will be taken to be the one in cgraphparam. 
%                Mainly used in the case of the minimum cost to time ratio.
%    fmin        lowest frequency in the graph (0<=fmin<=fmax).
%                Corresponds to the frequency 2*pi*fmin/N. 
%                Overwrites what is set in cgraphparam. 
%                Default value is 0.
%    fmax        highest frequency in the graph (fmin<=fmax<= (no.samples)-1). 
%                Corresponds to the frequency 2*pi*maxfreq/N. 
%                Overwrites what is set in cgraphparam. 
%                Default value is (no.samples)-1. 
%  Outputs
%    cnetwork    a cell structure of size 1-by-5, where the cell elements
%                are G, nvertices, startnodes, endnodes, ord 
%                in this order and are described below:
%
%          G    Cell array of size (k,2) where k is the number of
%                nodes; cell (m,1) is a vector with all the nodes that
%                node m connects to and cell (m,2) is a vector with the 
%                corresponding arc costs
%
%           nvertices
%                number of nodes in the network.
%
%           startnodes  
%                List of nodes from where paths can start in the network.     
%                They must be the first nodes in the topological ordering. 
%
%           endnodes    
%                List of nodes from where paths can end in the network.
%                They must be the last nodes in the topological ordering.
%
%           ord	 Topological ordering of the nodes
%
%  Description
%    Preprocessing for the optimization routines in CalculateStatistic.
%    Fills in chirplet costs into the chirplet graph so if a new signal comes 
%    in this preprocessing has to be done again.
%
%  See Also
%    - ChirpletTransform
%    - GetChirpletGraphParam
%    - CalculateStatistic


% This function assumes that chirplets are the arcs of the
% network. The nodes in the network are the time-frequency endpoints
% of the chirplets.

if (nargin<2),
    % assuming default chirplet graph, i.e. all scales
    % allowed and default slope discretization.
    N = length(C)+1;
    cgraphparam = GetChirpletGraphParam(N);
elseif (isempty(cgraphparam)),
    % assuming default chirplet graph, i.e. all scales
    % allowed and default slope discretization.
    N = length(C)+1;
    cgraphparam = GetChirpletGraphParam(N);        
end

if (nargin < 5),
  fmax = cgraphparam{4}(2);
  if (nargin < 4),
    fmin = cgraphparam{4}(1);
    if (nargin<3),
      coarsestScale = cgraphparam{2}(1);
    else
      if (coarsestScale < cgraphparam{2}(1)),
        msg = strcat('The parameter coarsestScale cannot correspond',...
          ' to a scale coarser than what is set in cgraphparam');
        error(msg);
      end
    end
  else
    if (fmax < fmin | fmin < 0),
      msg = strcat('The parameter fmin cannot be bigger than the',...
          ' maximum allowed frequency or smaller than 0.');
      error(msg);
    end
  end
end


cnetwork = cell(1,5);
n = cgraphparam{1}(1);
J = cgraphparam{1}(2);
finestScale = cgraphparam{2}(2);


% ntfnodes depends upon the allowable scales
% PEND! Make the number of chirplets/cells be one of the parameters
%       returned by GetChirpletGraphParam. Avoids things specific to
%       the chirplet graph to be calculated at more than one place. 

nfreqs = fmax-fmin+1; % number of frequency indices

startfreqs = fmin:fmax; % the frequency offsets (starting frequencies)

if strcmp(cgraphparam{6},'COLOREDNOISEVARAMP'),
  % special handling needed for col. noise + var. amp.
  startfreqsIndices = 1:(fmax-fmin+1); % the frequency offsets (starting frequencies)
else
  startfreqsIndices = startfreqs+1;
end


slopeindices = cell(nfreqs,1);

%nfreqs = N; % number of frequency indices
ntfnodes = nfreqs*2^finestScale; % number of left time points is 2^finestScale                
cnetwork{1} = cell(ntfnodes,2);


for s = finestScale:-1:coarsestScale,
  % Calculate the changes in frequency. A time interval at scale s
  % is of length 2^(J-s)
  dfreq = (cgraphparam{3}{s-cgraphparam{2}(1)+1})*2^(J-s); 
  Dfreq = repmat(dfreq,length(startfreqs),1);
  M = repmat(startfreqs',1,length(cgraphparam{3}{s-cgraphparam{2}(1)+1}));
  endfreqs = round(M+Dfreq);	% Calculate all the end frequencies and round to 
                                % get the frequency as an integer
  if (fmax == n-1),
    % the frequencies have period N, want a value
    % between 0 and N-1.
    endfreqs = mod(endfreqs,n);
    for m=1:nfreqs,
      % allow all slope indices
      slopeindices{m} = 1:length(endfreqs(m,:));
    end
  else                          
    % we will only use chirplets with end-frequency between the allowed range   
    for m=1:nfreqs,
      slopeindices{m} = find(endfreqs(m,:) <= fmax  & endfreqs(m,:) >= fmin );
    end
  end
  
  % loop over time indices and find the connecting chirplets
  for b = 0:(2^s-1),
    c = C{node(s,b)};
    leftoffset = b*2^(finestScale-s)*nfreqs + 1-fmin;
    rightoffset = (b+1)*2^(finestScale-s)*nfreqs + 1-fmin;
    leftindex = startfreqs + leftoffset;
    rightindex = endfreqs + rightoffset;
    
    for m = 1:nfreqs,      
       cnetwork{1}{leftindex(m),1} =... 
                  [cnetwork{1}{leftindex(m),1} rightindex(m,slopeindices{m})];
       cnetwork{1}{leftindex(m),2} =... 
         [cnetwork{1}{leftindex(m),2} -abs(c(startfreqsIndices(m),slopeindices{m})).^2];
    end
  end
end

% list the start nodes and end nodes
cnetwork{3} = 1:nfreqs; % the topological ordering of startnodes
cnetwork{4} = ntfnodes+(1:nfreqs); % the topological ordering of endnodes
cnetwork{2} = ntfnodes+nfreqs; % number of nodes in the graph

% The topological ordering (the nodes where numbered that way)
cnetwork{5} = 1:(ntfnodes+nfreqs);


% $RCSfile: GetChirpletNetwork.m,v $
% $Date: 2006/09/05 19:09:45 $
% $Revision: 1.8 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2005
