function [cost,thepath] = ShortestPathCell(G,n,alpha,ord,startnodes,endnodes)        
% ShortestPathCell -- 	solves the shortest path problem for an acyclic graph, starting from given
% 			start nodes to given endnodes. Uses a cell array data structure for the network.
%  Usage
%    [cost,thepath] = ShortestPathCell(G,n,alpha,order,startnodes,endnodes)
%  Inputs
%    G          a cell array of size (k,2) where k is the number of nodes in the graph
%               for which arcs go out from. cell (m,1) is a vector with all the nodes 
%               that node m connects to and cell (m,2) is a vector with the corresponding
%               arc costs
%    n          total number of nodes in graph.
%    alpha      a constant to update the costs on the arcs.
%    ord        a vector with the topological ordering of the nodes in G
%    startnodes	list of nodes from where paths can start in the acyclic network.
%               NOTE: They must be the first nodes in the topological ordering.
%    endnodes   list of nodes from where paths can end in the acyclic network.
%               NOTE: They must be the last nodes in the topological ordering.
%  Outputs
%    cost       The cost of the shortest path.
%    thepath    The shortest path, starting from one of the startnodes and ending at one
%               of the endnodes.
%  Description
%    Returns the shortest path (list of nodes) and the cost of it. The path starts from one of the
%    given startnodes and ends at one of the given endnodes.
%    Uses distance labels to solve the shortest path problem for an acyclic graph.
%    This is an inner function in ChirpLab.
%    It is recommended to use CalculateStatistic instead.
%
%  See Also
%    - CalculateStatistic

%   PEND: Use default values if startnodes and endnodes are not provided; i.e. the first and 
%   last nodes in the topologocial ordered list. That is not really needed for the chirplet
%   graph anyways.

nG = size(G,1);  % returns the number of starting nodes in G
if nargin < 4
  % Topological order was not provided. Assume that the
  % nodes in G appear in topological order.
  ord = 1:nG;
end

% Call mex function to find the distance labels and the list of
% predecessor nodes for the shortest path.
[d,pred] = ShortestPathCellStub(G(:,1),G(:,2),n,alpha,ord,startnodes,nG);

% Now we figure out which of the endnodes has the lowest distance label. That is going to be
% the end node of our shortest path. To find the shortest path we just have to start at
% the endnode and track the predecessor nodes.
[cost,ind] = min(d(endnodes));
thepath = pred2sp( pred(1:endnodes(ind)) );


% $RCSfile: ShortestPathCell.m,v $
% $Date: 2006/05/01 17:48:32 $
% $Revision: 1.3 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2004
