function [alpha,lpaths,paths] = minctratiocells(G,alphamax,n,order,startnodes,endnodes)        
% MINCTRATIOCELLS --	solves the minimum cost-time ratio problem for an acyclic graph 
%			with unit times.
%  Usage
%    [alpha,lpaths,paths] =
%              minctratiocells(G,alphamax,alphamax,n,order,startnodes,endnodes)
%  Inputs
%    G          a cell array of size (k,2) where k is the number of nodes in the graph
%               for which arcs go out from. cell (m,1) is a vector with all the nodes
%               that node m connects to and cell (m,2) is a vector with the corresponding
%               arc costs
%    alphamax   a known upper bound on the minimum cost-time ratio
%    n          number of nodes in graph
%    order      a vector with the topological ordering of the nodes in G
%    startnodes list of nodes from where paths can start in the acyclic network.
%               NOTE: They must be the first nodes in the topological ordering.
%    endnodes   list of nodes from where paths can end in the acyclic network.
%               NOTE: They must be the last nodes in the topological ordering.
%  Outputs
%    alpha      a vector with the sequence of "thresholds" that lead to the minimum value
%    lpaths     the entries in this vector are the lengths of the optimal paths that give
%		        the values in alpha (in the same order as they appear in alpha).
%    paths      a vector with the paths corresponding to the entries in alpha and lengths.
%               They appear in the same order as in alpha, one after another, and can be
%               retrieved using 'lpaths'.
%
%  Description
%    Solves the minimum cost-time ratio problem for an acyclic graph. 
%    The arcs have unit time lengths.
%    This is an inner function in ChirpLab.
%    It is recommended to use CalculateStatistic instead.
%
%  See Also:
%    - CalculateStatistic

% We'll pass alphamax to spoptimum so we can define new costs for the arcs
% and solve the shortest path problem for these costs.

[cost, p] = ShortestPathCell(G,n,alphamax,order,startnodes,endnodes);
% 'cost' is the cost of this path, p is the list of nodes on path
alpha = alphamax;
lpath = length(p) - 1;
lpaths = lpath; pp = p;

while (cost < 0 && abs(cost) > 1e-03),
  % we have not found the optimum value, but we have a better upper
  % bound 
  alphamax = cost/lpath + alphamax;
  [cost, p] = ShortestPathCell(G,n,alphamax,order,startnodes, ...
                               endnodes);
  
  lpath = length(p) - 1;
  alpha = [alpha alphamax];
  lpaths = [lpaths lpath];
  pp = [pp p];
end
% When we exit the loop, the modified cost is ~ 0 and we hold 
% the optimal value

npaths = length(alpha);
paths = cell(1,npaths);
a = 1;
for k = 1:npaths,
  b = a + lpaths(k);
  paths{k} = pp(a:b);
  a = b+1;
end


% $RCSfile: minctratiocells.m,v $
% $Date: 2006/05/01 17:48:32 $
% $Revision: 1.4 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2004
