function [yhat,CV] = SplineSmoother(y,t,alpha)
% SplineSmoother -- a spline smoother
%            
%  Usage
%    [yhat,CV] = SplineSmoother(y,t,alpha);
%  Inputs
%    y       vector of noisy samples
%    t       vector of time indices where the data is sampled
%    alpha   vector of smoothing parameters for which to calculate 
%            the cross-validation score function CV(alpha) (see Chapter 5 in Thesis)
%  Outputs
%    yhat    the estimate of the signal in the noise using spline smoothing with
%            the entry in alpha which minimizes CV(alpha)
%    CV      the cross-validation score function evaluated at the entries in alpha

n = length(t);

if (length(y)~=n),
    error('The length of y and t has to be equal!');
end

y = reshape(y,n,1);
t = reshape(t,n,1);

% step 1:
Qty = SetupStep1(y,t);

% step 2:
[Q,R] = SetupStep2(t);

QTQ = (Q.')*Q;

CV = zeros(1,length(alpha));
for k=1:length(alpha),
  B = R + alpha(k)*QTQ;
  
  % step 3:
  gamma = B\Qty;

  % step 4:
  yhat = y - alpha(k)*Q*gamma;

  % find the cross-valdiation score
  A = speye(n,n) - alpha(k)*Q*inv(B)*(Q.');
  
  emse = (y-yhat).^2;
  denom = (1-diag(A)).^2;

  CV(k) = sum( emse./denom  )/n;
end

% find the minimum CV and return the corresponding estimate
[bogus,ind] = min(CV);
B = R + alpha(ind)*QTQ;

% step 3:
gamma = B\Qty;

% step 4:
yhat = y - alpha(ind)*Q*gamma;


% $RCSfile: SplineSmoother.m,v $
% $Date: 2008/05/19 20:05:11 $
% $Revision: 1.1 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2007

