function [estPower, pmin, ind] = EstimateDetRate(nullT,signalT,thresh,whichIndArray,rejectForBig)
% EstimateDetRate -- estimate detection rate
%
%  Usage
%    [estPower, pmin, ind] = EstimateDetRate(nullT,signalT,thresh,whichIndArray,rejectForBig)
%    [estPower, pmin, ind] = EstimateDetRate(nullT,signalT,thresh,whichIndArray)
%    [estPower, pmin, ind] = EstimateDetRate(nullT,signalT,thresh)
%  Inputs
%    nullT          a matrix with realizations under the null distribution. 
%                   Each sample of the random vector corresponds to one row.
%    signalT        a matrix with realizations for a noisy signal.
%                   Each sample of the random vector corresponds to one row.
%    thresh         a vector of threshold values to use for comparing the minimum
%                   p-values
%    whichIndArray  choose which coordinates in the random vector to restrict to
%                   and base the decision on.
%                   Has to be the same indices as were used for getting the
%                   threshold in the parameter 'thresh' from the null distribution.
%                   If omitted, use all coordinates.
%    rejectForBig   if set to 1, big values are "unlikely" under the null
%                   if set to 0, small values are "unlikely" under the null
%                   If omitted the value will be set to 1.
%  Outputs
%    estPower       the estimated power (i.e. detection rate), 
%                   a vector of same length as thresh. estPower(k) corresponds to thresh(k)
%    pmin           a vector with estimation of minimum p-values for the realizations in signalT
%    ind            the indices for which gave minimum p-values in pmin.
%                   For the k-th realization, whichIndArray(ind(k)) is the
%                   the coordinate in signal(k,:) which gave the minimum p-value.
%
%  Description
%    Given samples of a multivariate test statistic under H0, the null 
%    hypothesis, and samples of the same test statistic for a given
%    signal with additive noise drawn from H0, this function calculates the 
%    estimated detection rate for given thresholds. Decision is based on the
%    minimum p-value.

[nSamples, noCoordinates] = size(signalT);

if (nargin < 5),
  rejectForBig = 1;
  if (nargin < 4),
    % use all coordinates by default
    whichIndArray = 1:noCoordinates;
    if (nargin < 3),
      errormsg = 'EstimateDetRate: The parameters nullT, signalT and thresh have to be provided';
	  error(errormsg);
    end
  end
end

nCoordinates = length(whichIndArray);

pvalues = zeros(nSamples,nCoordinates);

[pmin,ind] = EstimateMinPvalues(nullT,signalT,whichIndArray,rejectForBig);

estPower = zeros(1,length(thresh));
for k=1:length(thresh),
  estPower(k) = sum(pmin<=thresh(k))/length(pmin);
end

% $RCSfile: EstimateDetRate.m,v $
% $Date: 2006/05/01 17:52:55 $
% $Revision: 1.2 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2006
