function Cnorm = CTNormsDataStream(S,param)
% CTNormsDataStream -- Calculates x'Sigma^(-1)x, for chirplets x
%                          given a noise spectrum.
%  Usage
%    Cnorm = CTNormsDataStream(S,param);
%  Inputs
%    S       spectrum of noise process
%    param   chirplet graph parameters as returned by
%            GetChirpletGraphParam
%  Outputs
%    Cnorm  a cell structure storing x'Sigma^(-1)x for all
%           chirplets for a signal of length N=2^J (given by the chirplet graph param).
%           
% Description
%   Calculates x'Sigma^(-1)x for all the chirplets x in a chirplet graph.
%   Theses coefficients are stored in a J-by-1 cell structure, where
%   cell s corresponds to scale s and is an nx-by-n matrix (nx is the
%   number of slopes allowed at this scale, equal to 2^(J-s) ). The
%   entries in the matrix are as follows:
%        -each column corresponds to one slope
%        -each entry in a column corresponds to an offset

if nargin < 2,
  errormsg = 'CTNormsDataStream: Chirplet graph parameters have to be provided!';
  error(errormsg);
end

n = param{1}(1);
J = param{1}(2);
coarsestScale = param{2}(1);
finestScale = param{2}(2);

D = 1./S; % inverse of noise spectrum
nD = length(D); % if the length of the spectrum is larger than the length of the signal, zero-pad the fft
df = nD/n;

Cnorm = cell(1,J);

for s = finestScale:-1:coarsestScale,,
  nx = 2^(J-s);
  t = 0:nx-1;

  slopes = param{3}{s-coarsestScale+1};
  nSlopes = length(slopes);

  % Modulation followed by zero-padding
  phi = pi*slopes.'*t.^2/n; 
  X = exp(i * phi);  
  % NOTE: Each column in X corresponds to one slope.
  Xfft = abs(fft(X.',nD)/sqrt(nD)).^2;
  XfftFlip = flipud(Xfft);

  % Do a circular convolution with D and Xfft flipped
  Dfft = fft(D');
  c = repmat(Dfft,1,nSlopes).*fft(XfftFlip);
  c = ifft(c);
  Cnorm{s+1} = circshift(sqrt(abs(c((1:n)*df,:))),1); % have to do a shift to index right

  % taking the absolute value to
  % force the value to be real and positive in case
  % of some numerical errors.
end


% $RCSfile: CTNormsDataStream.m,v $
% $Date: 2008/05/25 20:07:51 $
% $Revision: 1.2 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2007
