% A demo for the Two-Stage BP Test code in ChirplabDataStream
% NOTE: ChirpLab needs to be in the Matlab path (run ChirpPath.m in the ChirpLab package)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CONFIGURATION AND INITIALIZATION
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

Fs = 2048;
freqmin = 20;

% CONFIGURATIONS FOR THE STEP I STATISTIC
% Fs: sampling frequency  in H; freqmin: minimum frequency in Hz
J = 8;
N = 2^J;
sldf = 8;     % slope discretization factor
fmin = floor(freqmin/Fs*N);    % minimum discrete frequency in chirplet graph
fmax = N/2-1; % maximum frequency in chirplet graph
csc = 1;      % coarsest scale in chirplet graph
fsc = 1;      % finest scale in chirplet graph
graphParamStepI = GetChirpletGraphParam(N,csc,fsc,sldf,[],fmin,fmax,'COLOREDNOISE');
sbase = csc;      % scale index for the base time interval in search
smin = 7;    % scale index for the smallest time interval to consider, length 2^smin
smax = 9;   % scale index for the largest time interval to consider, length 2^smax;

% SETTINGS FOR THE NOISE
M = 2^16;
S = MakeLIGO1Psd(M,Fs);
kappa = 1e+20;
P = kappa^2*Fs*S;
D = 1./P;

% calculate chirp norms
tic;
CnormStepI = CTNormsDataStream(P,graphParamStepI);
runt = toc;
disp(sprintf('Time to calculate norms for Step I: %0.4f',runt));

% Generate norms and graph parameters for Step II statistic
fname = 'graphParamAndNormStepII';
[graphParamExtInt, CnormExtInt, maxLength] = InitGraphParam(P,fname);

% Uncomment the line below and comment the line above if the configuration
% and norms have been saved to the file
%load(fname);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% generate signal to detect
load('Data/BANKHH-2048@2048');
bankIndex = 5;
nf = length(template{bankIndex,bankIndex}.f);
slength = nf; % signal length
h = template{bankIndex,bankIndex}.h;
nh = length(h);
tmp = (1:slength) + nh-slength;
h = h(tmp).';

% normalize h
hnormalized = h/spw_norm(h,P);
ind = (1:slength) + 10000;
startInd = ind(1);
sig = zeros(1,M);
sig(ind) = hnormalized;


% generate data
randn('state',0);
noise = MakeRealNoise2(P);
slevel = 13;
y = slevel*sig + noise;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Step I: Mono-scale statistic
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% do mono-scale chirplet transform
yy = ifft(D.*fft(y));
overwhitened = 1;
C = CTDataStream(yy,graphParamStepI,P,CnormStepI,overwhitened);

% find BP on all dyadic intervals of lengths 2^s, where s=smin,...,smax
% time-support of chirplets is 2^(graphParam{1}(1)-sbase) 
T = MonoScaleBP(C{1},M,smin,smax,sbase,graphParamStepI);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Find promising intervals and extend them
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
numTagged = [50 30 12];

taggedIntervals = TagIntervals2(T,smin,smax,numTagged);

% Extend promising intervals and calculate BP on each of them
lmax = 1; % extension-level
[TT, intervalExtensions] = ExtIntAndFindBP(yy,overwhitened,P,lmax,smin,smax,...
                                 taggedIntervals,graphParamExtInt,CnormExtInt,maxLength);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Visualization
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% plot tagged intervals
ShowTaggedIntervals(taggedIntervals,smin,smax);

% plot tagged intervals and the true support of the unknown signal
ShowTaggedIntervals(taggedIntervals,smin,smax);
hold on;
plot([ind(1) ind(end)],[smin-1 smin-1],'-ro');          
axis([ind(1)-2^smax ind(end)+2^smax ylim]);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Check the maximum overlap of tagged intervals with the signal support
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
[maxOverlap, maxOverlapDyadInd] = MaxOverlapTaggedIntervals(ind,taggedIntervals,smin,smax)
