function intervalExtensions = ExtendIntervals(ny,taggedInt,smin,smax,lmax)
% ExtendIntervals -- extends promising dyadic intervals
%  Usage
%    intervalExtensions = ExtendIntervals(ny,taggedInt,smin,smax,lmax)
% Inputs
%   ny        length of data
%   taggedInt list of promising intervals
%   smin      dyadic length of shortest interval
%   smax      dyadic length of longest interval
%   lmax      extension level
% Output
%   intervalExtensions  list of extended intervals

intervalExtensions = cell(1,length(taggedInt));

for s=smin:smax,
  baseIntervals = taggedInt{s-smin+1};
  nBaseIntervals = length(baseIntervals);
  intervalExtensions{s-smin+1} = cell(2,nBaseIntervals);

  % loop over base intervals
  for b=1:nBaseIntervals,
    seedDyadInd = baseIntervals(b);

    % check if this interval is at either end of data
    atRightEnd = 0;
    atLeftEnd = (seedDyadInd==0);
    if( (seedDyadInd+1)*2^s+1 > ny),
      atRightEnd = 1;
    end

    doubleToTheRight = 0;   
    if (s~=smax),
      if (bitget(seedDyadInd,1)==1 & atRightEnd~=1),
        % we only double the length of base intervals to the right
        % if the resulting interval is not a dyadic interval of
        % the next scale
        doubleToTheRight = 1;
      end
    end

    % Extend interval:
    offsets = 2^(s-lmax)*(0:(2^lmax-1));
    dataInd = seedDyadInd*2^s + 1; % data stream index of left endpoint
    leftIndices = [];
    rightIndices = [];
    if(atLeftEnd),
      leftIndices = dataInd;
    else
      leftIndices = dataInd-offsets;
    end
    if (atRightEnd==0),
      rightIndices = dataInd+2^s + offsets-1;
    else
      rightIndices = dataInd+2^s-1;
    end
    if (doubleToTheRight),
      tmp = dataInd+2*2^s + offsets-1;
      rightIndices = [rightIndices tmp];
    end
    
    intervalExtensions{s-smin+1}{1,b} = leftIndices;
    intervalExtensions{s-smin+1}{2,b} = rightIndices;

  end
end

% $RCSfile: ExtendIntervals.m,v $
% $Date: 2007/05/17 04:14:21 $
% $Revision: 1.1.1.1 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2007
