function [maximum, indAtMax] = FindMaximumStats(T)
% FindMaximumStats -- returns the maximum BP statistic for
%                     each type of interval extended and chirplet path length
% Usage
%   [maximum, indAtMax] = FindMaximumStats(T)
% Input
%   T         collection of BP statistic as returned from ExtIntAndFindBP
% Outputs
%   maximum   a cell structure storing the maximum value of the BP statistics 
%             for each base interval length, interval extension type and chirplet path length
%   indAtMax  the index in T where the maximum was achieved
%
% Example: 
%   Assume the base intervals have lengths 128, 256, and 512, max. extension level 1
%   and maximum number of chirplets in a path equal to 8. Then maximum is a cell
%   structure of size 3x8, since there are 3 base interval sizes and 8 types of interval
%   extensions. We number the types of base intervals with 1,2,3, and the types of
%   extensions by 1,2,...,8. Types of extended intervals can thus be described by a
%   pair (s,m), where s corresponds to the type of base interval and m to the type of extension.
%   Then maximum{s,m}(k) stores the maximum value of the BP statistic of chirplet paths 
%   of length (at most) k, over all (s,m) intervals.
% 
% NOTE: This function is tailored for the Two-Stage BP test with a maximum
%   level of extension equal to 1 (i.e., the parameter lmax in ExtInAndFindBP is 1)

nScales = length(T);


maximum = cell(nScales,8);
indAtMax = cell(nScales,8);

for s=1:nScales,
  T2atScale = T{s};
  nPromInt = length(T2atScale);

%  maxX1Y1 = zeros(length(chirpPathLengths),1);
%  maxX1Y2 = zeros(length(chirpPathLengths),1);
%  maxX1Y3 = zeros(length(chirpPathLengths),1);
%  maxX1Y4 = zeros(length(chirpPathLengths),1);
%  maxX2Y1 = zeros(length(chirpPathLengths),1);
%  maxX2Y2 = zeros(length(chirpPathLengths),1);
%  maxX2Y3 = zeros(length(chirpPathLengths),1);
%  maxX2Y4 = zeros(length(chirpPathLengths),1);

% chirpPathLengths = [5 8];

  chirpPathLengths = 1:size(T2atScale{1}{1},1);
  
  for m=1:8,
    maximum{s,m} = zeros(length(chirpPathLengths),1);
    indAtMax{s,m} = zeros(length(chirpPathLengths),1);
  end

  maxX1Y1 = zeros(length(chirpPathLengths),1);
  maxX1Y2 = zeros(length(chirpPathLengths),1);
  maxX1Y3 = zeros(length(chirpPathLengths),1);
  maxX1Y4 = zeros(length(chirpPathLengths),1);
  maxX2Y1 = zeros(length(chirpPathLengths),1);
  maxX2Y2 = zeros(length(chirpPathLengths),1);
  maxX2Y3 = zeros(length(chirpPathLengths),1);
  maxX2Y4 = zeros(length(chirpPathLengths),1);

  indmaxX1Y1 = zeros(length(chirpPathLengths),1);
  indmaxX1Y2 = zeros(length(chirpPathLengths),1);
  indmaxX1Y3 = zeros(length(chirpPathLengths),1);
  indmaxX1Y4 = zeros(length(chirpPathLengths),1);
  indmaxX2Y1 = zeros(length(chirpPathLengths),1);
  indmaxX2Y2 = zeros(length(chirpPathLengths),1);
  indmaxX2Y3 = zeros(length(chirpPathLengths),1);
  indmaxX2Y4 = zeros(length(chirpPathLengths),1);


  for k=1:nPromInt,
    extIntStats = T2atScale{k};
  
      %  Interval extension for extension level l=1:
      %  |----|--------|----|
      %  |----|--------|----|----|----|
      %  x2   x1       y1   y2   y3   y4
      %  extIntStats{1}(:,1) % x1-y1 base interval
      %  extIntStats{1}(:,2) % x1-y2
      %  extIntStats{2}(:,1) % x2-y1
      %  extIntStats{2}(:,2) % x2-y2
      %  In case interval was doubled to the right:
      %  extIntStats{1}(:,3) % x1-y3
      %  extIntStats{1}(:,4) % x1-y4
      %  extIntStats{2}(:,3) % x2-y3
      %  extIntStats{2}(:,4) % x2-y4
      %
      %  x1-y2 and x2-y1 are "equivalent", i.e. the topology of the graph is the same  
      %  x2-y3 and x1-y4 are "equivalent"
      %
      %  If x1 is at the beginning of data, [x1,y1] was not extended to the left
      %  If y1 is at the end of data, [x1,y1] was not extended to the right
      
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % Handle BP starting at x1:

    tmp = abs(extIntStats{1}(chirpPathLengths,1)); % x1-y1 base interval
    ind = (tmp > maxX1Y1); 
    maxX1Y1(ind) = tmp(ind);
    indmaxX1Y1(ind) = k;

    % check whether y1 is at the end of data
    if (size(extIntStats{1},2)~=1),
      tmp = abs(extIntStats{1}(chirpPathLengths,2)); % x1-y2 base interval
      ind = (tmp > maxX1Y2); 
      maxX1Y2(ind) = tmp(ind);
      indmaxX1Y2(ind) = k;
    end
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % Handle BP starting at x2:

    % check whether x1 is at the beginning of data
    if(length(extIntStats)==2),
      % we're not at the beginning of data, so x2 falls within data sequence
      tmp = abs(extIntStats{2}(chirpPathLengths,1)); % x2-y1 base interval
      ind = (tmp > maxX2Y1); 
      maxX2Y1(ind) = tmp(ind);
      indmaxX2Y1(ind) = k;
      
      % check whether y1 is at the end of data
      if (size(extIntStats{2},2)~=1),
        tmp = abs(extIntStats{2}(chirpPathLengths,2)); % x2-y2 base interval
        ind = (tmp > maxX2Y2); 
        maxX2Y2(ind) = tmp(ind);
        indmaxX2Y2(ind) = k;
      end
    end
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


    % Check whether this tagged interval was doubled to the right,
    % Note: If an interval was doubled to the right, y3 and y4 will
    %       will always fall within the data sequence
    %       Also, x2 will also fall within the data sequence 
    if(size(extIntStats{1},2)==4), 
      % interval was doubled to the right
      tmp = abs(extIntStats{1}(chirpPathLengths,3)); % x1-y3 base interval
      ind = (tmp > maxX1Y3); 
      maxX1Y3(ind) = tmp(ind);
      indmaxX1Y3(ind) = k;

      tmp = abs(extIntStats{1}(chirpPathLengths,4)); % x1-y4 base interval
      ind = (tmp > maxX1Y4); 
      maxX1Y4(ind) = tmp(ind);
      indmaxX1Y4(ind) = k;

      tmp = abs(extIntStats{2}(chirpPathLengths,3)); % x2-y3 base interval
      ind = (tmp > maxX2Y3); 
      maxX2Y3(ind) = tmp(ind);
      indmaxX2Y3(ind) = k;

      tmp = abs(extIntStats{2}(chirpPathLengths,4)); % x2-y4 base interval
      ind = (tmp > maxX2Y4); 
      maxX2Y4(ind) = tmp(ind);
      indmaxX2Y4(ind) = k;

    end % if (size(extInt....

  end % for k (tagged intervals at scale s)

  maximum{s,1} = maxX1Y1;      
  maximum{s,2} = maxX1Y2;
  maximum{s,3} = maxX1Y3;
  maximum{s,4} = maxX1Y4;
  maximum{s,5} = maxX2Y1;
  maximum{s,6} = maxX2Y2;
  maximum{s,7} = maxX2Y3;
  maximum{s,8} = maxX2Y4;
    
  indAtMax{s,1} = indmaxX1Y1;
  indAtMax{s,2} = indmaxX1Y2;
  indAtMax{s,3} = indmaxX1Y3;
  indAtMax{s,4} = indmaxX1Y4;
  indAtMax{s,5} = indmaxX2Y1;
  indAtMax{s,6} = indmaxX2Y2;
  indAtMax{s,7} = indmaxX2Y3;
  indAtMax{s,8} = indmaxX2Y4;

end % for s (loop over scales)

% $RCSfile: FindMaximumStats.m,v $
% $Date: 2008/05/25 17:21:24 $
% $Revision: 1.2 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2007
