function [graphParamExtInt, CnormExtInt, maxLength] = InitGraphParam(P,fname)
% InitGraphParam -- a utility script for configuring chirplet graph parameters
% and precalculating chirplet norms to use for extended intervals
%  Input
%    P      noise spectrum
%    fname  file name for saving the graph parameters and norms. Optional.
%  Output
%    graphParamExtInt  graph parameters 
%    CnormExtInt       chirplet norms 
%    maxLength         maximum number of chirplets to use in the BP algorithm

smin = 7;
smax = 10;
nScales = smax-smin+1;
graphParamExtInt = cell(1,nScales);
CnormExtInt = cell(1,nScales);
maxLength = zeros(1,nScales);

% Different chirplet graph parameters are used for different
% lengths of base intervals that are extended
for s=smin:smax,
  sindex=s-smin+1;

  steepSlopeParam = 1;
  J = s;
  % CONFIGURATIONS FOR THE STATISTICS
  Fs = 2048;
  N = 2^J;
  sldf = 8;     % slope discretization factor
  fmin = floor(40/Fs*N);    % minimum frequency in chirplet graph
  fmax = N/2-1; % maximum frequency in chirplet graph
  csc = 0;      % coarsest scale in chirplet graph
  if (J==10),
    fsc = 6;      % finest scale in chirplet graph
    ml = 16;
  elseif(J==9),
    fsc = 5;      % finest scale in chirplet graph
    ml = 16;
  elseif(J==8),
    fsc = 4;      % finest scale in chirplet graph
    ml = 8;
  elseif(J==7),
    fsc = 3;
    ml = 8;
  end
  gp = GetChirpletGraphParam(N,csc,fsc,sldf,[],fmin,fmax,'COLOREDNOISE');

  if (steepSlopeParam>0),
    % use steeper slopes at finest scales
    gp{3}{fsc+1} = [-0.5:sldf*2^((fsc)-J):8];
    if (steepSlopeParam>1),
      gp{3}{fsc} = [-0.5:sldf*2^(fsc-1-J):8];
    end
  end

  graphParamExtInt{sindex} = gp;
  maxLength(sindex) = ml;

  % calculate chirp norms
  CnormExtInt{sindex} = CTNormsDataStream(P,graphParamExtInt{sindex});
end

if (nargin > 1),
  % save graph parameters and norm
  save(fname,'graphParamExtInt','CnormExtInt','maxLength')
end

