function [maxOverlap, ind] = MaxOverlapTaggedIntervals(supportInd,taggedInt,minScale,maxScale)
% MaxOverlapTaggedIntervals -- finds the maximum normalized overlap of a signal support
% with a list of tagged intervals
%                       
%  Usage
%   [maxOverlap, ind] = MaxOverlapTaggedIntervals(supportInd,taggedInt,minScale,maxScale)
%  Input
%    supportInd  indices of the signal support in the data sequence 
%                (indices of the data sequence start counting from 1, i.e. Matlab convention)
%    taggedInt   list of intervals tagged as promising as returned by TaggedIntervals
%    minScale	 dyadic length of shortest interval considered
%    maxScale    dyadic length of longest interval considered
%  Output
%    maxOverlap  vector of maximum normalized overlaps.
%                maxOverlap(m) is the maximum overlap for tagged intervals at scale 2^(-(m+minScale-1))
%                equal to zero if no tagged interval overlaps with the support 
%    ind         vector of dyadic indices of interval achieving the maximum overlap,
%                ind(m) corresponds to scale 2^(-(m+minScale-1)) 
%                equal to zero if no tagged interval overlaps with the support 
%
%    Maximum normalized overlap is the ratio:
%       #( supp(signals) \cap supp(tagged interval) )
%      -----------------------------------------------------
%      sqrt( # supp(signals) * # supp(tagged interval) )

supportLength = length(supportInd);

nScales = maxScale-minScale+1;

maxOverlap = zeros(1,nScales);
ind = zeros(1,nScales);
% loop over scales
for m=1:nScales,
  s = minScale+m-1;
  intervalLength = 2^s;

  nIntervals = length(taggedInt{m});
  % check each tagged interval at scale 2^(-s)
  for n=1:nIntervals,
    k = taggedInt{m}(n);
    % dyadic interval is indexed by (s,k)
    lPt = k*2^s+1;
    rPt = lPt+2^s;

    % check the overlap   
    tmp = sum(supportInd <= rPt & supportInd >= lPt);
    tmpOverlap = tmp/sqrt(supportLength*intervalLength);
    if (maxOverlap(m) < tmpOverlap),
      maxOverlap(m) =  tmpOverlap;
      ind(m) = k;
    end
  end
end


% $RCSfile: MaxOverlapTaggedIntervals.m,v $
% $Date: 2007/05/19 20:16:10 $
% $Revision: 1.2 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2007
