function T = MonoScaleBP(C,ny,smin,smax,s,graphParam)
% MonoScaleBP -- Calculates the monoscale BP on dyadic intervals
%  Usage
%    T = MonoScaleBP(C,ny,smin,smax,s,graphParam)
%  Input
%    C		chirplet coefficients as returned by CTDataStream
%    ny		length of data stream
%    smin	dyadic length of shortest dyadic interval to consider
%    smax	dyadic of longest dyadic interval to consider
%    s		scale index to use in the chirplet graph defined by graphParam
%    graphParam	chirplet graph parameters. Have to be the same as used for
%		calculating the chirplet coefficients in C. 
%		Also has to include the scale index s

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% chirplet graph parameters
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
n = graphParam{1}(1);    
J = graphParam{1}(2); 
csc = graphParam{2}(1);  % scale index for coarsest scale (needed for indexing
                    % in slope cell array
minFreq = graphParam{4}(1);
maxFreq = graphParam{4}(2);
nx = n/2^s;  % number of points in time interval, sets the time scale

% get the slopes
slopes = graphParam{3}{s-csc+1};
nSlopes = length(slopes);

% Precalculate connection rules for nodes
nFreq = maxFreq-minFreq+1;
dfreq = slopes*2^(J-s);
endFreqInd = cell(1,nFreq);
endSlopeInd = cell(1,nFreq);
for k=1:nFreq,
  tmp = minFreq+(k-1)+dfreq;
  endSlopeInd{k} = find(tmp <= maxFreq  & tmp >= minFreq );
  endFreqInd{k} = tmp(endSlopeInd{k})-minFreq+1;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% calculate statistics
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% min and max lengths of intervals (in samples)
lmin = 2^smin;
lmax = 2^smax;

% check-points
cpts = 2.^(smin:smax)/nx;

% number of left endpoints to consider in the
% whole data stream
nStartInd = ny/lmin;

% initializing structure used for storing the
% values of the test statistics
%T = zeros(nStartInd,length(cpts));
T = cell(1,length(cpts));
%for k=1:length(cpts),
%  T{k} = zeros(1,nStartInd/2^(k-1));
%end

% loop over left endpoints
for k = 1:nStartInd,
  startInd = (k-1)*lmin+1;

  % check what is the maximum power of two that divides k-1
  tmp = bitget(k-1,1:(smax-smin+1));
  maxPow2 = find(tmp==1,1,'first')-1;
  if( isempty(maxPow2)),
    % all the bits in k-1 are zeros
    maxPow2 = smax-smin;
  end
  allowableScales = smin:(maxPow2+smin);
  intLength = 2^(allowableScales(end));
  endInd = min(startInd+intLength-1,ny);
% endInd = min(startInd+lmax-1,ny);
  ind = startInd:endInd;

  ccStartInd = (k-1)*lmin/nx+1; % the starting time-index in the chirplet coef table

  nData = endInd-startInd+1;
  
  % nData has to be an integer multiple of nx
  nIntervals = nData/nx; % number of time intervals
  startNodeDistance = zeros(nFreq,1);

  % no. check-points included
  noCptsIncluded = maxPow2+1;
  optimalDistances = zeros(1,noCptsIncluded);

  startInterval = 1;
  for ind=1:noCptsIncluded,
    endInterval = cpts(ind);
    % update distance labels up to the next check point
    for b = startInterval:endInterval,
      % set distance labels of end-nodes to 0 since we
      % haven't visited them yet
      endNodeDistance = zeros(nFreq,1); 
      slopeIndOffset = (ccStartInd-1)*nSlopes+(b-1)*nSlopes; % need this to index the chirplet coefficient table
      % loop over frequencies and update distance labels
      for m=1:nFreq,
        % calculate tentative distances        
        tmpCost = startNodeDistance(m)+C(m,slopeIndOffset+endSlopeInd{m});        
        % check whether any of the tentative distances
        % are better than the current distances
        tmpInd = (tmpCost' > endNodeDistance(endFreqInd{m}));
        % update cost
        endNodeDistance(endFreqInd{m}(tmpInd)) = tmpCost(tmpInd);
      end
      startNodeDistance = endNodeDistance;
    end
    % save optimal distance at current check point
%    optimalDistances(ind) = max(endNodeDistance); % normalizing with the length of interval
    T{ind} = [T{ind} max(endNodeDistance)]; % normalizing with the length of interval
    startInterval = endInterval+1;
  end
%  T(k,1:noCptsIncluded) = optimalDistances;
end


% $RCSfile: MonoScaleBP.m,v $
% $Date: 2007/05/17 04:14:21 $
% $Revision: 1.1.1.1 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2007
