function C = CTDataStream(sig,param,S,Cnorm,overWhitened)
% CTDataStream -- Chirplet Transform for handling data stream
%  Usage
%    C = CTDataStream(sig,param,S,Cnorm);
%  Input
%    sig     data stream to apply a chirplet transform to.
%            It is possible to enter the "overwhitened"
%            data stream, i.e. the vector inv(Sigma)*y, where
%            Sigma is the covariance of the noise.
%    param   chirplet graph parameters as returned by
%            GetChirpletGraphParam
%    S       The spectrum of the noise process.
%    Cnorm   Normalizing constants for the chirplets in the transform
%            as returned by CTNormsDataStream, calculated using the
%            same chirplet graph parameters as in param and the 
%            noise spectrum S
%    overWhitened  set to 1 if the data stream in sig has been overwhitened
%            if omitted, the noise in sig is assumed to be colored
%            with noise spectrum S
%  Outputs
%    C       chirplet coefficients, stored in a cell array.
%

if nargin < 5,
  overWhitened = 0;
end

if nargin < 4,
  errormsg = 'CTDataStream: Chirplet norms have to be provided!';
  error(errormsg);
elseif nargin < 3,
  errormsg = 'CTDataStream: Spectrum and chirplet norm parameters have to be provided!';
  error(errormsg);
elseif nargin < 2,
  errormsg = 'CTDataStreamGraph parameters have to be provided!';
  error(errormsg);
end

% ensure that sig is a row vector and change
% it if it isn't
signalLength = length(sig);
x = reshape(sig,1,signalLength);

n = param{1}(1);
J = param{1}(2);
coarsestScale = param{2}(1);
finestScale = param{2}(2);
fmin = param{4}(1);
fmax = param{4}(2);
freqIndices = (fmin:fmax)+1;
nFreq = length(freqIndices);

nCells = finestScale-coarsestScale+1; 

C = cell(1,nCells);

% Weight data by inverse of noise spectrum and multiply by mask

if (overWhitened==0)
  % the noise is assumed to be circular stationary
  D = 1./S;
  y = ifft(D.*fft(x).*param{7});
else
  y = x;
end

for s = coarsestScale:finestScale,
  slopes = param{3}{s-coarsestScale+1};
  nSlopes = length(slopes);
  nx = 2^(J-s);
  t = 0:(nx-1);
  % number of intervals at scale index s
  nIntervals = floor(signalLength/nx);
  costs = zeros(nFreq,nIntervals*nSlopes);
  for b = 0:(nIntervals-1),
    ix = ( (b*nx):((b+1)*nx-1) ) +1;
    % Modulation followed by zero-padding
    X = repmat(y(ix),nSlopes,1) .* exp(-i * pi*slopes.' * t.^2/n);
%    X = [X.'; zeros(n - length(ix),nSlopes)];
%    c = fft(X); % Operates along columns
    X = X.';
    c = fft(X,n); % Operates along columns
    ind = (1:nSlopes) + b*nSlopes;
    tmp = c(freqIndices,:)./Cnorm{1,s+1}(freqIndices,:); % normalizing
    costs(:,ind) = abs(tmp).^2;
  end
  C{s-coarsestScale+1} = costs;
end;


% $RCSfile: CTDataStream.m,v $
% $Date: 2007/05/25 19:03:22 $
% $Revision: 1.2 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2007
