/**
 * GraphParam.hpp -- A wrapper utility for chirplet graph parameters.
 *
 * THIS IS PART OF CHIRPLAB
 */
   
/**
 * $RCSfile: GraphParam.hpp,v $
 * $Date: 2007/05/17 04:14:21 $
 * $Revision: 1.1.1.1 $
 * Copyright (c) Hannes Helgason, California Institute of Technology, 2006
 */

#ifndef _GRAPHPARAM2_HPP_
#define _GRAPHPARAM2_HPP_

#include "mex.h"
#include "Base2Utilities.hpp"
#include <exception>
#include <string>
#include <vector>
using namespace std;

// constants for chirplet XT type
const int PLAINXT = 1;
const int VARAMPXT = 2;
const int COLOREDNOISEXT = 3;
const int COLOREDNOISEVARAMPXT = 4;

class GraphParam {
    
    // Parameters determining the topology of the graph:
    int _n;                 // length of signal under consideration
    int _j;                 // dyadic length of signal
    int _size;              // number of nodes in graph
    int _cs;                // coarsest allowable scale        
    int _fs;                // finest allowable scale
    int _fmin;              // smallest allowable frequency
    int _fmax;              // greatest allowable frequency
    double* _slopeRange;    // slope range
    vector< vector<double> > _slopes; // slopes per scale,
                                     // the k-th entry corresponds to the
                                     // degree for the k-th coarsest scale

    vector< vector<double> > _dfreq; // precomputed values of the 
                                     // difference in start and end frequency 
                                     // per scale, calculated from _slopes
    
    // Parameters to set the chirplet transform:
    int _xtType;            // parameter setting the transfom
    
    vector<int> _ampDegrees;// in the case of varying amplitude type XT
                            // this is a vector of degrees of the
                            // polynomial fit, the k-th entry corresponds
                            // to the degree for the k-th coarsest scale
 


    public:   
        // Contructors

        // Constructor using chirplet graph parameters as returned by 
        // GetChirpletGraph.m
        GraphParam(const mxArray *matlabGraphParam) {
            const mxArray *cellElementPtr;
            double* cellValues;
            int nElements;
            int nSlopes;
            const mxArray *slopeCellPtr; 
            char *input_buf;
            string xtTypeStr;
            
            // read in signal length and dyadic length
            cellElementPtr = mxGetCell(matlabGraphParam,0);
            cellValues = mxGetPr(cellElementPtr);
            _n = (int) cellValues[0];
            _j = (int) cellValues[1];
            
            // read in coarsest and finest scale
            cellElementPtr = mxGetCell(matlabGraphParam,1);
            cellValues = mxGetPr(cellElementPtr);
            _cs = (int) cellValues[0];
            _fs = (int) cellValues[1];
            
            // Given the list of slopes and possible start frequencies,
            // precompute the difference in start and end frequency 
            // Recall that the slopes are stored from the coarsest scale
            // to the finest scale.
            cellElementPtr = mxGetCell(matlabGraphParam,2);
            nElements = mxGetNumberOfElements(cellElementPtr);
            if (nElements == _fs-_cs+1) {
                _slopes.resize(nElements);
                _dfreq.resize(nElements);
                int currentScale;
                for (int k=0; k<nElements; k++) {
                    currentScale = _cs+k;
                    slopeCellPtr = mxGetCell(cellElementPtr,k);
                    nSlopes = mxGetNumberOfElements(slopeCellPtr);
                    _slopes[k].resize(nSlopes);
                    cellValues = mxGetPr(slopeCellPtr);
                    _dfreq[k].resize(nSlopes);
                    for (int m=0; m<nSlopes; m++) {
                        _slopes[k][m] = cellValues[m];
                        _dfreq[k][m] = _slopes[k][m]*((double)pow2(_j-currentScale));
                    }  
                }
            } else {
                mexErrMsgTxt("Error in Chirplet Graph Parameters: Slopes need to be provided for all allowed scales.");
            }

            // read in maximum and minimum frequency
            cellElementPtr = mxGetCell(matlabGraphParam,3);
            cellValues = mxGetPr(cellElementPtr);
            _fmin = (int) cellValues[0];
            _fmax = (int) cellValues[1];
            
            // read in degrees for amplitude fit
            cellElementPtr = mxGetCell(matlabGraphParam,4);
            nElements = mxGetNumberOfElements(cellElementPtr);
            if (nElements > 0) {
                _ampDegrees.resize(nElements);
                cellValues = mxGetPr(cellElementPtr);
                for (int k=0; k<nElements; k++) {
                    _ampDegrees[k] = (int) cellValues[k];
                }
            }
            
            // read in the transormation type
            cellElementPtr = mxGetCell(matlabGraphParam,5);
            if (mxIsChar(cellElementPtr) == 1) {
                // it's a string
                input_buf = mxArrayToString(cellElementPtr);
                xtTypeStr = string(input_buf);
                mxFree(input_buf);
                if(xtTypeStr=="PLAIN") {
                    _xtType = PLAINXT;
                } else if(xtTypeStr=="VARAMP") {
                    _xtType = VARAMPXT;
                } else if(xtTypeStr=="COLOREDNOISE") {
                    _xtType = COLOREDNOISEXT;
                } else if(xtTypeStr=="COLOREDNOISEVARAMP") {
                    _xtType = COLOREDNOISEVARAMPXT;
                }
            } else if (mxIsDouble(cellElementPtr) == 1) {
                // NOTE!! HERE WE ARE ASSUMING THE MEX AND MATLAB CODE
                // BOTH USE THE SAME INTEGERS FOR REPRESENTING THE XT TYPE
                _xtType = (int) mxGetScalar(cellElementPtr);
            }// if
            
        }//GraphParam constructor
        
        // methods to access graph parameters
        int n() const { return _n; }
        int j() const { return _j; }
        int cs() const { return _cs; }                
        int fs() const { return _fs; }
        int fmin() const { return _fmin; }
        int fmax() const { return _fmax; }
        int xtType() const {return _xtType; }

        // get the number of slopes at a certain scale
        int nslopes(int scale) const {
            if(scale>=_cs & scale<=_fs) {
                return(_slopes[scale-_cs].size());
            } else {
                return(0);
            }
        }

        // return the slope given a scale and slope index
        double slope(int scale, int slInd) const {
            try {
                return(_slopes[scale-_cs][slInd]);
            } catch (exception const& e) {
                // the caller is trying to access out-of-bounds elements
                mexErrMsgTxt("Error while trying to access out-of-bounds elements in the list of slopes.");
            }
        }

        // return the frequency change given a scale and slope index
        double dfreq(int scale, int slInd) const {
            try {
                return(_dfreq[scale-_cs][slInd]);
            } catch (exception const& e) {
                // the caller is trying to access out-of-bounds elements
                mexErrMsgTxt("Error while trying to access out-of-bounds elements in the list of slopes.");
            }
        }

        // no. frequency indices
        int numFrequencies() const {
            return(_fmax-_fmin+1);
        }

        // total number of time indices in [0,1)
        int numTimeIndices() const {
            return(pow2(_fs));
        }

        // no. nodes where chirplets can eminate from
        int numTimeFreqNodes() const {
            return(numFrequencies()*numTimeIndices());
        }

        // total no. of nodes in the graph
        int numNodes() const {
            return(numTimeFreqNodes() + numFrequencies());
        }

        // no. nodes where chirplet paths can start from
        int numStartNodes() const {
            return(numFrequencies());
        }

        // returns the coarsest allowable scale given a time index k
        // corresponding to the time t=k/2^_fs
        int coarsestAllowableScale(int k) const {
            int cas;
            if (k!=0) {
                cas = _fs-maxPower2(k);
                if (cas<_cs) {
                    // we don't want to allow a scale that is coarser
                    // than the absolute coarsest scale set for the graph
                    cas = _cs;
                }
            } else {
                // for t=0 we allow all scales
                cas = _cs;
            }
            return(cas);
        }
};

#endif

