#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "atomic.h"
#include "constantunits.h"
#include "programming.h"

#include "errors.h"
#include "interface.h"
#include "coordinates.h"
#include "cartesian.h"

double length(int i,int j,double *q)
{
	double x=q[i*3+0]-q[j*3+0];
	double y=q[i*3+1]-q[j*3+1];
	double z=q[i*3+2]-q[j*3+2];
	return(sqrt(x*x+y*y+z*z));
}

double vibration(int i,int j,double *u2)
{
	double x2=u2[i*3+0]+u2[j*3+0];
	double y2=u2[i*3+1]+u2[j*3+1];
	double z2=u2[i*3+2]+u2[j*3+2];
	return(sqrt(x2+y2+z2));
}


// The following function calculates the rotational constants of the molecule and returns the number of rotational degrees of freedom

int rotational(CARTESIAN *f)
{
	int i,j,k,n=3;
	char JOBZ='V',UPLO='U';
	int INFO=0;
	double B[3]={0},e[3]={0},I[6]={0},p[9]={0},WORK[9]={0};

	for(i=0;i<f->na;i++)
		for(j=0;j<3;j++)
		{
			for(k=0;k<=j;k++)
				I[uindex(k,j)]-=f->m[i*3+k]*f->cc[i*3+j]*f->cc[i*3+k];	// CONFIRM
			for(k=0;k<3;k++)
				I[uindex(j,j)]+=f->m[i*3+k]*f->cc[i*3+k]*f->cc[i*3+k];
		}
	if(1)
		for(j=0;j<3;j++)
		{
			for(k=0;k<=j;k++)
				fprintf(stderr," %+le ",I[uindex(j,k)]);
			fprintf(stderr,"\n");
		}
	fDSPEV(JOBZ,UPLO,3,I,e,p,3,WORK,&INFO);		// e holds the eigenvalues, p holds the eigenvectors, and I is diagonalized
	if(e[0]<0.1*(AMU*Bohr*Bohr)) n=2;
	for(j=3-n;j<3;j++)
		B[j]=h2_8pi2/(e[j]);			// in energy units
	fprintf(stderr," Moment of Intertia   = %lf %lf %lf (amu bohr^2)\n",e[0]/(AMU*Bohr*Bohr),e[1]/(AMU*Bohr*Bohr),e[2]/(AMU*Bohr*Bohr));
	fprintf(stderr," Rotational Constants = %lf %lf %lf (/cm)\n",B[0]/hc/PerCentimeter,B[1]/hc/PerCentimeter,B[2]/hc/PerCentimeter);
	fprintf(stderr," Rotational Constants = %lf %lf %lf (GHz)\n",B[0]/PlanckConstant/GigaHerz,B[1]/PlanckConstant/GigaHerz,B[2]/PlanckConstant/GigaHerz);
	f->nv=f->nc-n-3;
	return(n);					// return the number of nonzero rotational constants
}


// The following function calculates the center-of-mass, the mass-weighted cartesian coordinates and fills them into the structure CARTESIAN

int massweighting(CARTESIAN *f)
{
	int i,j,k,n=6;
	double w=0,m,com[3]={0};
	double rg=0;
	for(i=0;i<f->na;i++)
	{
		w+=(m=atomicmass(f->an[i]));
		for(j=0;j<3;j++)
		{
			k=i*3+j;
			f->m[k]=m;
			f->rm[k]=1/sqrt(m);
			com[j]+=m*f->cc[k];
		}
	}
	for(j=0;j<3;j++)
		com[j]/=w;
	for(i=0;i<f->na;i++)
		for(j=0;j<3;j++)
		{
			f->cc[i*3+j]-=com[j];
			rg+=f->cc[i*3+j]*f->cc[i*3+j];
		}
	rg=sqrt(rg/w);
	fprintf(stderr," (mass-weighted) radius of gyration is %lf A\n",rg/Angstrom);
	for(j=0;j<f->nc;j++)
	{
		f->mc[j]=f->cc[j]/f->rm[j];
		f->mg[j]=f->cg[j]*f->rm[j];
		for(i=0;i<=j;i++)
		{
			k=uindex(i,j);
			f->mf[k]=f->cf[k]*(f->rm[i]*f->rm[j]);
		}
	}
	n=3+rotational(f);
	fprintf(stderr," cartesians are mass-weighted\n");
	return(n);					// return the number of external (vibrational and rotational) degrees of freedom
}

int cartesianamplitudes(int nx,THERMAL *t,CARTESIAN *f,double *eigenvalue,double *eigenvector)
{
	int i,j,k,n;
	double Ti,r,l1,l2,l2sum,frequency,wavenumber;
	double *bt,*v;
	if(NULL==(bt=(double *)calloc(f->nc,sizeof(double)))) 
		ued3error("lack of memory");
	if(NULL==(v=(double *)calloc(f->nc,sizeof(double)))) 
		ued3error("lack of memory");
	for(j=1;j<f->na;j++)
		for(i=0;i<j;i++)
		{
			l2sum=0;
			fDCLEAR(f->nc,bt,1);
			r=distance(i,j,f->cc,bt);
			for(n=nx;n<f->nc;n++)
			{
				Ti=t->Tv[n-nx];
				frequency=sqrt(fabs(eigenvalue[n]))*(0.5*M_1_PI);	// in Hz
				wavenumber=frequency/(SpeedLight);	// in 1/cm
				l2=h_8pi2c/(wavenumber*tanh(hc_2k*wavenumber/Ti));	// in amu A^2
				l1=sqrt(l2);	// in sqrt(amu) A
				for(k=0;k<f->nc;k++)
					v[k]=f->rm[k]*l1*eigenvector[rindex(k,n,f->nc,f->nc)];
				l1=fDDOT(f->nc,bt,1,v,1);
				l2sum+=l1*l1;
				fprintf(stderr,"%5d%5d%5d l= %10.5lf A from %10.8lf sqrt(amu) A of %10.4lf /cm\n",i+1,j+1,n+1,l1/Angstrom,sqrt(l2)/SqrtAMUAngstrom,wavenumber/PerCentimeter);
			}
			// fprintf(stdout," %4d %4d      r= %10.5lf l= %10.8lf\n",i+1,j+1,r,sqrt(l2sum));
			//fprintf(stdout," %2s%-4d %2s%-4d    r= %15.8lf    l= %10.8lf\n"
			//,tablesymbol[f->an[i]],i+1,tablesymbol[f->an[j]],j+1
			//,"",i+1,"",j+1
			char buffer[3];
			atomicsymbol(f->an[i],buffer);
			fprintf(stdout," %2s%-4d",buffer,i+1);
			atomicsymbol(f->an[j],buffer);
			fprintf(stdout," %2s%-4d",buffer,j+1);
			fprintf(stdout," r= %15.8lf",r/Angstrom);
			fprintf(stdout," l= %10.8lf",sqrt(l2sum)/Angstrom);
			fprintf(stdout,"\n");
			fprintf(stderr,"\n");

		}
	free(v);
	free(bt);
	return(f->na);
}


int massweightedcartesian(THERMAL *t,CARTESIAN *f)
{
	int i,j,nx=6;
	char JOBZ='V',UPLO='U'; // JOBZ='N';
	int INFO=0,LWORK=3*f->nc;
	double *eigenvalue=NULL,*eigenvector=NULL,*WORK=NULL;
	double Ti,rg,l2,frequency,wavenumber;

	rg=fDDOT(f->nc,f->cg,1,f->cg,1);
	fprintf(stderr," rms cartesian gradient is %le hartree/bohr^2\n",sqrt(rg/f->nc)/(Hartree/Bohr/Bohr));
	nx=massweighting(f);
	if(NULL==(eigenvalue=(double *)calloc(f->nc,sizeof(double)))) 
		ued3error("lack of memory");
	if(NULL==(eigenvector=(double *)calloc(f->nc*f->nc,sizeof(double)))) 
		ued3error("lack of memory");
	if(NULL==(WORK=(double *)calloc(LWORK,sizeof(double)))) 
		ued3error("lack of memory");
	fprintf(stderr," memory for matrix diagonalization is allocated\n");

	fDSPEV(JOBZ,UPLO,f->nc,f->mf,eigenvalue,eigenvector,f->nc,WORK,&INFO);
	fprintf(stderr," mass weighted force constant is diagonalized\n");
	for(i=0;i<f->nc;i++)
	{
		for(j=0;j<f->nc;j++)
			fprintf(stderr,"%6.3lf",eigenvector[rindex(j,i,f->nc,f->nc)]);
		fprintf(stderr,"\n");
		for(j=0;j<f->nc;j++)
			fprintf(stderr,"%6.3lf",eigenvector[rindex(j,i,f->nc,f->nc)]*f->rm[j]);
		fprintf(stderr,"\n");
		fprintf(stderr,"\n");
	}
	for(i=0;i<nx;i++)
	{
		frequency=sqrt(fabs(eigenvalue[i]))*(0.5*M_1_PI);
		wavenumber=frequency/(SpeedLight);
		fprintf(stderr,"%5d: %12.3lf GHz %12.4lf /cm from %16.7le\n"
			,i+1,frequency/GigaHerz,wavenumber/PerCentimeter,eigenvalue[i]);
	}
	for(i=nx;i<f->nc;i++)
	{
		Ti=t->Tv[i-nx];
		frequency=sqrt(fabs(eigenvalue[i]))*(0.5*M_1_PI);
		wavenumber=frequency/(SpeedLight);
		fprintf(stderr,"%5d: %12.3lf GHz %12.4lf /cm from %16.7le  "
			,i+1,frequency/GigaHerz,wavenumber/PerCentimeter,eigenvalue[i]);
		l2=h_8pi2c/(wavenumber*tanh(hc_2k*wavenumber/Ti));
		fprintf(stderr,"l= %lf sqrt(amu) A at %lf K",sqrt(l2)/SqrtAMUAngstrom,Ti/Kelvin);
		fprintf(stderr,"\n");
	}
	fprintf(stderr,"\n");

	cartesianamplitudes(nx,t,f,eigenvalue,eigenvector);

	free(WORK);
	free(eigenvector);
	free(eigenvalue);
	return(f->na);
}

