#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "errors.h"
#include "intrinsic.h"
#include "coordinates.h"


// calculate the distance vector/angle/dihedral angle, q, and returns its length/value, r, 
// and assign B matrix elements (for the current column), btc.
// For a recipe to calculate the B matrix entries refer to (S. Califano, Vibrational States, 1976, pp.84)

double calculatebtcolumn(int type,int *ijkl,double *q,double *btc) // B^T in C storage
{
	double r=0;
	switch(type)
	{
		case BOND_STRETCHING:
			r=distance(ijkl[0],ijkl[1],q,btc);		
			break;
		case BOND_RECIPROCAL:
			r=reciprocal(ijkl[0],ijkl[1],q,btc);
			break;
		case BOND_RECIPROCAL2:
			r=reciprocal2(ijkl[0],ijkl[1],q,btc);
			break;
		case BOND_RECIPROCAL6:
			r=reciprocal6(ijkl[0],ijkl[1],q,btc);
			break;
		case BENDING_ANGLE:
			r=bending(ijkl[0],ijkl[1],ijkl[2],q,btc);
			break;
		case DIHEDRAL_ANGLE:
			r=dihedral(ijkl[0],ijkl[1],ijkl[2],ijkl[3],q,btc);
			break;
		case DIHEDRAL_360:
			r=dihedral360(ijkl[0],ijkl[1],ijkl[2],ijkl[3],q,btc);
			break;
		case BENDING2_ANGLE:
			r=bending2(ijkl[0],ijkl[1],ijkl[2],ijkl[3],q,btc);
			break;
		case TANGENT_DIHEDRAL2:
			r=tangent2dihedral(ijkl[0],ijkl[1],ijkl[2],ijkl[3],q,btc);
			break;
		case TANGENT_PI_DIHEDRAL2:
			r=tangentpi2dihedral(ijkl[0],ijkl[1],ijkl[2],ijkl[3],q,btc);
			break;
		default:
			ued3error("unknown coordinate");
			break;
	}
	return(r);
}

double calculateallinearity(int type,int *ijkl,double *q) 
{
	double r=0,nl=0;
	double qij[3]={0};
	switch(type)
	{
		case BOND_STRETCHING:
			nl=0;
			break;
		case BOND_RECIPROCAL:
			r=displacement(ijkl[0],ijkl[1],q,qij);
			nl=2/(r*r*r);
			break;
		case BOND_RECIPROCAL2:
			r=displacement(ijkl[0],ijkl[1],q,qij);
			nl=6/(r*r*r*r);
			break;
		case BOND_RECIPROCAL6:
			r=displacement(ijkl[0],ijkl[1],q,qij);
			nl=42/(r*r*r*r*r*r*r*r);
			break;
		case BENDING_ANGLE:
			nl=0;
			break;
		case DIHEDRAL_ANGLE:
			nl=0;
			break;
		case DIHEDRAL_360:
			nl=0;
			break;
		case BENDING2_ANGLE:
			nl=0;
			break;
		case TANGENT_DIHEDRAL2:
			nl=0;	// ???
			break;
		case TANGENT_PI_DIHEDRAL2:
			nl=0;	// ???
			break;
		default:
			ued3error("unknown coordinate");
			break;
	}
//	fprintf(stderr,"nl= %le\n",nl);
	return(nl);
}



double calculatenonlinearity(int type,int *ijkl,double *q) 
{
	double r=0,lc=0;
	double qij[3]={0};
	switch(type)
	{
		case BOND_STRETCHING:
			lc=0;
			break;
		case BOND_RECIPROCAL:
			r=displacement(ijkl[0],ijkl[1],q,qij);
			lc=2*r;
			break;
		case BOND_RECIPROCAL2:
			r=displacement(ijkl[0],ijkl[1],q,qij);
			lc=3*r*r/2;
			break;
		case BOND_RECIPROCAL6:
			r=displacement(ijkl[0],ijkl[1],q,qij);
			lc=7*r*r*r*r*r*r/6;
			break;
		case BENDING_ANGLE:
			lc=0;
			break;
		case DIHEDRAL_ANGLE:
			lc=0;
			break;
		case DIHEDRAL_360:
			lc=0;
			break;
		case BENDING2_ANGLE:
			lc=0;
			break;
		case TANGENT_DIHEDRAL2:
			lc=0;	// ???
			break;
		case TANGENT_PI_DIHEDRAL2:
			lc=0;	// ???
			break;
		default:
			ued3error("unknown coordinate");
			break;
	}
	return(lc);
}


// This function subtracts a multiple of 2*PI from the value d

double dihedraldifference(double d)
{
	int n=(int)floor(d/(2*M_PI)+0.5);
	double dd=d-n*2*M_PI;
//	fprintf(stderr,"dihedraldifference= %+lf %+lf\n",dd*180/M_PI,d*180/M_PI);
	return(dd);
}


// The following function calculates the dot product of two 3-vectors

double ddot3(double *qi,double *qj)
{
	return(qi[0]*qj[0]+qi[1]*qj[1]+qi[2]*qj[2]);
}


// The following function calculates the cross product of vectors qi and qj and multiplies the result by f to get the vector q

void ftimes3(double f,double *qi,double *qj,double *q)
{
	q[0]=f*(qi[1]*qj[2]-qj[1]*qi[2]);
	q[1]=f*(qi[2]*qj[0]-qj[2]*qi[0]);
	q[2]=f*(qi[0]*qj[1]-qj[0]*qi[1]);
}


// The following function calculates a difference 3-vector and returns its length
// The difference is between vectors i and j 

double displacement(int i,int j,double *q,double *qij)
{
	int i3=i*3, j3=j*3, n;
	for(n=0;n<3;n++)
		qij[n]=q[i3+n]-q[j3+n];
	return(sqrt(ddot3(qij,qij)));
}


// The following function returns the length of a difference 3-vector (between i and j)
// The function also assings the entries for the B matrix (S. Califano, Vibrational States, 1976, pp.84)

double distance(int i,int j,double *q,double *b)
{
	int i3=i*3, j3=j*3, n;
	double qij[3]={0};
	double rij=displacement(i,j,q,qij);
	double sij=1/rij;
	for(n=0;n<3;n++)
		b[j3+n]=-(b[i3+n]=sij*qij[n]);			// entries for the B matrix
	return(rij);
}

double reciprocal(int i,int j,double *q,double *b)
{
	int i3=i*3, j3=j*3, n;
	double qij[3]={0};
	double zij=1/displacement(i,j,q,qij);
	double zij3=zij*zij*zij;
	for(n=0;n<3;n++)
		b[j3+n]=-(b[i3+n]=-zij3*qij[n]);
	return(zij);
}

double reciprocal2(int i,int j,double *q,double *b)
{
	int i3=i*3, j3=j*3, n;
	double qij[3]={0};
	double zij=1/displacement(i,j,q,qij);
	double zij2=zij*zij;
	double zij4=zij2*zij2;
	for(n=0;n<3;n++)
		b[j3+n]=-(b[i3+n]=-2*zij4*qij[n]);
	return(zij2);
}

double reciprocal6(int i,int j,double *q,double *b)
{
	int i3=i*3, j3=j*3, n;
	double qij[3]={0};
	double zij=1/displacement(i,j,q,qij);
	double zij2=zij*zij;
	double zij4=zij2*zij2;
	double zij6=zij2*zij4;
	double zij8=zij4*zij4;
	for(n=0;n<3;n++)
		b[j3+n]=-(b[i3+n]=-6*zij8*qij[n]);
	return(zij6);
}

double inrange(double x)
{
	if(x>1) x=1;
	if(x<-1) x=-1;
	return(x);
}


// The following function returns the angle between two 3-vectors qij and qjk
// The function also assings the entries for the B matrix
	
double bending(int i,int j,int k,double *q,double *b) // p140, eqn A2.2 in JMSp v160 pp117 
{
	int i3=i*3,j3=j*3,k3=k*3,n;
	double qij[3]={0},qjk[3]={0};
	double rij=displacement(i,j,q,qij);
	double rjk=displacement(j,k,q,qjk);
	double sij=1/rij, sjk=1/rjk;
	double cosbijk=inrange(-sij*sjk*ddot3(qij,qjk)); 
	double bijk=acos(cosbijk);
	// ea = ejk X eji / sin bijk = eij X ejk / sin bijk = dij x djk / rij rjk sin bijk
	double f=sij*sjk/sin(bijk);
	double ea[3]={0};
	ftimes3(-f,qij,qjk,ea);				// calculate unit vector normal to the plane of bending motion
	
	// assign the entries for the B matrix
	ftimes3(sij*sij,qij,ea,b+i3);			// bi = ea X eji / rij = eij X ea / rij	
	ftimes3(sjk*sjk,qjk,ea,b+k3);			// bk = ejk X ea / rjk		
	for(n=0;n<3;n++)
		b[j3+n]=-(b[i3+n]+b[k3+n]);		
	return(bijk);
}

double dihedral(int i,int j,int k,int l,double *q,double *b)
{
	int i3=i*3,j3=j*3,k3=k*3,l3=l*3,n;
	double qij[3]={0},qjk[3]={0},qkl[3]={0};
	double rij=displacement(i,j,q,qij);
	double rjk=displacement(j,k,q,qjk);
	double rkl=displacement(k,l,q,qkl);
	double sij=1/rij, sjk=1/rjk, skl=1/rkl;
	double cosbijk=inrange(-sij*sjk*ddot3(qij,qjk));
	double cosbjkl=inrange(-sjk*skl*ddot3(qjk,qkl));
	double ej[3]={0},ek[3]={0},ejk[3]={0};
	// ftimes3(-sij*sij*sjk/(1-cosbijk*cosbijk),qij,qjk,b+i3);
	// ftimes3(-skl*skl*sjk/(1-cosbjkl*cosbjkl),qkl,qjk,b+l3);
	ftimes3(sij*sjk,qij,qjk,ej);	// |ej| = rij rjk sin(ijk) sij sjk = sin(ijk)
	ftimes3(sjk*skl,qjk,qkl,ek);	// |ek| = rjk rkl sin(jkl) skl sjk = sin(jkl)
	double rj=sqrt(ddot3(ej,ej));
	double rk=sqrt(ddot3(ek,ek));
	for(n=0;n<3;n++)
	{
		b[i3+n]=-sij/(1-cosbijk*cosbijk)*ej[n];
		b[l3+n]=+skl/(1-cosbjkl*cosbjkl)*ek[n];
		b[j3+n]=-sjk*((rjk-rij*cosbijk)*b[i3+n]+rkl*cosbjkl*b[l3+n]);
		b[k3+n]=-sjk*((rjk-rkl*cosbjkl)*b[l3+n]+rij*cosbijk*b[i3+n]);
	}
	double cosdijkl=inrange(ddot3(ej,ek)/(rj*rk));
	double dijkl=acos(cosdijkl);
	ftimes3(1.,ej,ek,ejk);
	if(ddot3(ejk,qjk)>0) dijkl*=-1;
	return(dijkl);
}

double dihedral360(int i,int j,int k,int l,double *q,double *b)
{
	double dijkl=dihedral(i,j,k,l,q,b);
	if(dijkl<0) dijkl+=2*M_PI;
	return dijkl;
}

double bending2(int i,int j,int k,int l,double *q,double *b) // p140 in JMSp v160 pp117 : wrong
{
	int i3=i*3,j3=j*3,k3=k*3,l3=l*3,n;
	double qij[3]={0},qjk[3]={0},qjl[3]={0};
	double rij=displacement(i,j,q,qij);
	double rjk=displacement(j,k,q,qjk);
	double rjl=displacement(j,l,q,qjl);
	double sij=1/rij, sjk=1/rjk,sjl=1/rjl;
	double cosbijk=-sij*sjk*ddot3(qij,qjk); 
	if(cosbijk>1) cosbijk=1;
	if(cosbijk<-1) cosbijk=-1;
	double cosbijl=-sij*sjl*ddot3(qij,qjl); 
	if(cosbijl>1) cosbijl=1;
	if(cosbijl<-1) cosbijl=-1;
	double bijk=acos(cosbijk);
	double bijl=acos(cosbijl);
	// ea = ejk X eji / sin bijk = eij X ejk / sin = dij x djk / rij rjk sin
	double f=sij*sjk/sin(bijk);
	double g=sij*sjl/sin(bijl);
	double ea[3]={0};
	double eb[3]={0};
	ftimes3(-f,qij,qjk,ea);
	ftimes3(-g,qij,qjl,eb);
	ftimes3(sij*sij,qij,ea,b+i3); // bi = ea X eji / rij = eij X ea / rij
	ftimes3(sjk*sjk,qjk,ea,b+k3); // bk = ejk X ea / rjk
	ftimes3(sij*sij,qij,eb,ea);
	for(n=0;n<3;n++)
		b[i3+n]-=ea[n];
	ftimes3(-sjl*sjl,qjl,eb,b+l3); 
	for(n=0;n<3;n++)
		b[j3+n]=-(b[i3+n]+b[k3+n]+b[l3+n]);

	return(bijk-bijl);
}

double tangent2dihedral(int i,int j,int k,int l,double *q,double *b)
{
	int i3=i*3,j3=j*3,k3=k*3,l3=l*3,n;
	double r=0.5*dihedral(i,j,k,l,q,b); // dr/dx = dr/dy dy/dx 
	double z=tan(r); // dr/dy = s c^-1 ' = 1 + s^2 c^-2 = 1 + cot^2 = c^-2
	double cosr=cos(r);
	double f=0.5/(cosr*cosr);
	for(n=0;n<3;n++)
	{
		b[i3+n]*=f;
		b[l3+n]*=f;
		b[j3+n]*=f;
		b[k3+n]*=f;
	}
	return(z);
}

double tangentpi2dihedral(int i,int j,int k,int l,double *q,double *b)
{
	int i3=i*3,j3=j*3,k3=k*3,l3=l*3,n;
	double r=0.5*dihedral(i,j,k,l,q,b)-M_PI_2; // dr/dx = dr/dy dy/dx 
	double z=tan(r); // dr/dy = s c^-1 ' = 1 + s^2 c^-2 = 1 + cot^2 = c^-2
	double cosr=cos(r);
	double f=0.5/(cosr*cosr);
	for(n=0;n<3;n++)
	{
		b[i3+n]*=f;
		b[l3+n]*=f;
		b[j3+n]*=f;
		b[k3+n]*=f;
	}
	return(z);
}

