#include <stdio.h>
#include <stdlib.h>

#include "programming.h"
#include "constantunits.h"

#include "errors.h"
#include "cartesian.h"


// The following function assigns all the rotational and vibrational temperatures, which are either given in a listfile or as part of the input.

int assigntemperature(int n,THERMAL *T,char *sr,char *sv)
{
	FILE *fp;
	int i;
	double Tc=298;
	if('-'==sr[0])
	{
		if(NULL==(fp=fopen(sr+1,"rt")))
			ued3error(sr+1);
		else
		{
			for(i=0;i<3;i++)
			{
				if(1>fscanf(fp,"%lf",T->Tr+i)) break;
				else if(T->Tr[i]<=0) T->Tr[i]=1E-10;
				fprintf(stderr,"Tr[%d]= %lf\n",i+1,T->Tr[i]);
			}
			fclose(fp);
			if(i!=3) fprintf(stderr,"# read %d rotational temperatures\n",i);
		}
	}
	else
	{
		Tc=atof(sr);
		if(Tc<=0) Tc=1E-10;
		for(i=0;i<3;i++) T->Tr[i]=Tc;
	}
	for(i=0;i<3;i++) T->Tr[i]*=InputUnitTemperature;
	
	if('-'==sv[0])
	{
		if(NULL==(fp=fopen(sv+1,"rt")))
			ued3error(sv+1);
		else
		{
			for(i=0;i<n;i++)
			{
				if(1>fscanf(fp,"%lf",T->Tv+i)) break;
				else if(T->Tv[i]<=0) T->Tv[i]=1E-10;
				fprintf(stderr,"T[%d]= %lf\n",i+1,T->Tv[i]);
			}
			fclose(fp);
			if(i!=n-6) fprintf(stderr,"# read %d temperatures when 3N-6 = %d\n",i,n-6);
		}
	}
	else
	{
		Tc=atof(sv);
		if(Tc<=0) Tc=1E-10;
		for(i=0;i<n;i++) T->Tv[i]=Tc;
	}
	for(i=0;i<n;i++) T->Tv[i]*=InputUnitTemperature;
	return(i);
}

// The following function allocates the memory for the 1 dimensional arrays needed by the CARTESIAN structure defined in programming.h

int memorycartesianallocate(CARTESIAN *f)
{
	f->nc=3*f->na;
	f->nf=f->nc*(f->nc+1)/2;
	fprintf(stderr,"na= %d nc= %d nf= %d\n",f->na,f->nc,f->nf);
	if(NULL==(f->an=(int *)calloc(f->na,sizeof(int)))) ued3error("memory");
	if(NULL==(f->m =(double *)calloc(f->nc,sizeof(double)))) ued3error("memory");
	if(NULL==(f->rm=(double *)calloc(f->nc,sizeof(double)))) ued3error("memory");
	if(NULL==(f->cc=(double *)calloc(f->nc,sizeof(double)))) ued3error("memory");
	if(NULL==(f->cg=(double *)calloc(f->nc,sizeof(double)))) ued3error("memory");
	if(NULL==(f->cf=(double *)calloc(f->nf,sizeof(double)))) ued3error("memory");
	if(NULL==(f->ff=(double *)calloc(f->nc*f->nc,sizeof(double)))) ued3error("memory");
	if(NULL==(f->mc=(double *)calloc(f->nc,sizeof(double)))) ued3error("memory");
	if(NULL==(f->mg=(double *)calloc(f->nc,sizeof(double)))) ued3error("memory");
	if(NULL==(f->mf=(double *)calloc(f->nf,sizeof(double)))) ued3error("memory");
	fprintf(stderr," memory for cartesian force constants of %d are allocated\n",f->nc);
	return(f->nc);
}


// Free memory held by the CARTESIAN structure

int memorycartesianfree(CARTESIAN *f)   
{
	free(f->mf);
	free(f->mg);
	free(f->mc);
	free(f->ff);
	free(f->cf);
	free(f->cg);
	free(f->cc);
	free(f->rm);
	free(f->m);
	free(f->an);
	fprintf(stderr," memory for cartesian force constants are freed\n");
	return(f->nc);
}


// The following function allocates the memory for the CARTESIAN structure defined in programming.h

int memoryinternalallocate(INTERNAL *h)
{
	int nr2=h->nr*h->nr;
	h->nc=3*h->na;
	h->nf=h->nr*(h->nr+1)/2;
	h->freeze=-1;
	fprintf(stderr,"na= %d nr= %d nc= %d nf= %d\n",h->na,h->nr,h->nc,h->nf);
	if(NULL==(h->z   =(ZMT *)calloc(h->nr,sizeof(ZMT)))) ued3error("memory");
	if(NULL==(h->ai  =(NUCLEAR *)calloc(h->na,sizeof(NUCLEAR)))) ued3error("memory");
	if(NULL==(h->an  =(int *)calloc(h->na,sizeof(int)))) ued3error("memory");
	if(NULL==(h->cc  =(double *)calloc(h->nc,sizeof(ZMT)))) ued3error("memory");
	if(NULL==(h->m   =(double *)calloc(h->nc,sizeof(double)))) ued3error("memory");
	if(NULL==(h->mm  =(double *)calloc(h->nc*(h->nc+1)/2,sizeof(double)))) ued3error("memory");
	if(NULL==(h->mi  =(double *)calloc(h->nc*(h->nc+1)/2,sizeof(double)))) ued3error("memory");
	if(NULL==(h->mibt=(double *)calloc(h->nr*h->nc,sizeof(double)))) ued3error("memory");
	if(NULL==(h->b   =(double *)calloc(h->nc*h->nr,sizeof(double)))) ued3error("memory");
	if(NULL==(h->bt  =(double *)calloc(h->nr*h->nc,sizeof(double)))) ued3error("memory");
	if(NULL==(h->bi  =(double *)calloc(h->nr*h->nc,sizeof(double)))) ued3error("memory");
	if(NULL==(h->bit =(double *)calloc(h->nc*h->nr,sizeof(double)))) ued3error("memory");
	if(NULL==(h->bi2 =(double *)calloc(h->nr*h->nr*h->nc,sizeof(double)))) ued3error("memory");
	// if(NULL==(h->c   =(double *)calloc(h->nr*h->nc*h->nc,sizeof(double)))) ued3error("memory");
	if(NULL==(h->ct  =(double *)calloc(h->nc*h->nc*h->nr,sizeof(double)))) ued3error("memory");
	if(NULL==(h->lc  =(double *)calloc(h->nr,sizeof(double)))) ued3error("memory");
//	if(NULL==(h->nl  =(double *)calloc(h->nr,sizeof(double)))) ued3error("memory");

	if(NULL==(h->g   =(double *)calloc(h->nr*h->nr,sizeof(double)))) ued3error("memory");
	if(NULL==(h->gs  =(double *)calloc(h->nr*h->nr,sizeof(double)))) ued3error("memory");
	if(NULL==(h->gq  =(double *)calloc(h->nr*h->nr,sizeof(double)))) ued3error("memory");
	// if(NULL==(h->gt  =(double *)calloc(h->nf,sizeof(double)))) ued3error("memory");
	if(NULL==(h->gi  =(double *)calloc(h->nr*h->nr,sizeof(double)))) ued3error("memory");
	if(NULL==(h->gi2 =(double *)calloc(h->nr*h->nr*h->nr,sizeof(double)))) ued3error("memory");
	if(NULL==(h->r   =(double *)calloc(h->nr,sizeof(double)))) ued3error("memory");
	if(NULL==(h->gr  =(double *)calloc(h->nr,sizeof(double)))) ued3error("memory");
	if(NULL==(h->fr  =(double *)calloc(h->nr*h->nr,sizeof(double)))) ued3error("memory");
	if(NULL==(h->fi  =(double *)calloc(h->nr*h->nr,sizeof(double)))) ued3error("memory");
	if(NULL==(h->mr  =(double *)calloc(h->nr,sizeof(double)))) ued3error("memory");
	if(NULL==(h->mg  =(double *)calloc(h->nr,sizeof(double)))) ued3error("memory");
	if(NULL==(h->mf  =(double *)calloc(h->nr*h->nr,sizeof(double)))) ued3error("memory");
	if(NULL==(h->uf  =(double *)calloc(h->nr*h->nr,sizeof(double)))) ued3error("memory");

	if(NULL==(h->ev  =(double *)calloc(h->nr,sizeof(double)))) ued3error("memory");
	if(NULL==(h->vq  =(double *)calloc(h->nr*h->nr,sizeof(double)))) ued3error("memory");
	if(NULL==(h->vr  =(double *)calloc(h->nr*h->nr,sizeof(double)))) ued3error("memory");
	if(NULL==(h->vx  =(double *)calloc(h->nc*h->nr,sizeof(double)))) ued3error("memory");

	if(NULL==(h->fvt =(double *)calloc(h->nr*h->nr,sizeof(double)))) ued3error("memory");
	if(NULL==(h->frt =(double *)calloc(3*h->nr,sizeof(double)))) ued3error("memory");
	fprintf(stderr," memory for internal force constants of %d are allocated\n",h->nr);
	return(h->nr);
}


// Free memory held by the INTERNAL structure

int memoryinternalfree(INTERNAL *h)
{
	free(h->frt);
	free(h->fvt);

	free(h->vx);
	free(h->vr);
	free(h->vq);
	free(h->ev);

	free(h->uf);
	free(h->mf);
	free(h->mg);
	free(h->mr);
	free(h->fi);
	free(h->fr);
	free(h->gr);
	free(h->r);
	free(h->gi2);
	free(h->gi);
	// free(h->gt);
	free(h->gq);
	free(h->gs);
	free(h->g);
//	free(h->nl);
	free(h->lc);
	free(h->ct);
	// free(h->c);
	free(h->bi2);
	free(h->bit);
	free(h->bi);
	free(h->bt);
	free(h->b);
	free(h->mibt);
	free(h->mi);
	free(h->mm);
	free(h->m);
	free(h->cc);
	free(h->an);
	free(h->ai);
	free(h->z);
	fprintf(stderr," memory for internal force constants are freed\n");
	return(h->nr);
}

