#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "atomic.h"
#include "constantunits.h"
#include "intrinsic.h"
#include "programming.h"

#include "errors.h"
#include "outputs.h"
#include "interface.h"
#include "coordinates.h"
#include "cartesian.h"
#include "calculate.h"
//#include "transformation.h"

#include "internal.h"

int externalrotational(int detail,INTERNAL *h)	// per unit energy of rotation
{
	int i,j,k,n=3;
	char JOBZ='V',UPLO='U';
	int INFO=0;
	double v[3]={0},f[3]={0},e[3]={0},I[6]={0},p[9]={0},WORK[9]={0};
	double *fx=NULL,*dr=NULL;
	if(NULL==(fx=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(dr=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");

	for(i=0;i<h->na;i++)
		for(j=0;j<3;j++)
		{
			for(k=0;k<=j;k++)
				I[uindex(k,j)]-=h->m[i*3+k]*h->cc[i*3+j]*h->cc[i*3+k];
			for(k=0;k<3;k++)
				I[uindex(j,j)]+=h->m[i*3+k]*h->cc[i*3+k]*h->cc[i*3+k];
		}
	if(detail) printtriangularmatrix(3,3,I);
	fDSPEV(JOBZ,UPLO,3,I,e,p,3,WORK,&INFO);
	if(detail) printrectangularmatrix(3,3,p);
	if(e[0]<0.1*(AMU*Bohr*Bohr)) n=2;
	// <Er> = kT = 0.5*I*w^2
	// w = sqrt(2*Er/I)
	fDCLEAR(3*h->nr,h->frt,1);
	for(j=3-n;j<3;j++)
	{
		fDCLEAR(h->nc,fx,1);
		for(i=0;i<h->na;i++)
		{
			ftimes3(1,p+rindex(0,j,3,3),h->cc+i*3,v);
			ftimes3(1,p+rindex(0,j,3,3),v,f);
			fDAXPY(3,-2/e[j]*h->m[i*3],f,1,fx+i*3,1);	// per unit energy
		}
		if(detail) formattedvector(h->nc,fx,"fx");
		fDGEMV('T',h->nc,h->nr,1.0,h->bi,h->nc,fx,1,0.0,h->frt+rindex(0,j,h->nr,3),1);
		if(detail) formattedvector(h->nr,h->frt+rindex(0,j,h->nr,3),"fr");
		if(detail>1)
		{
			fDGEMV('N',h->nr,h->nr,1.0,h->fi,h->nr,h->frt+rindex(0,j,h->nr,3),1,0.0,dr,1);
			formattedvector(h->nr,dr,"drt");
		//	fprintf(stderr," & this might be off by 50%%\n");
		}
	}

	free(dr);
	free(fx);
	h->nv=h->nc-n-3;
	return(n);
}


// this function performs the massweighting of the Gradient and the Hessian in internal coordinates
// Need Reference

int massweightinginternal(int detail,INTERNAL *h)
{
	int i,j,k,l,n=h->nr;
	double e,*uf=NULL;

	if(NULL==(uf=(double*)calloc(n*n,sizeof(double)))) ued3error("lack of memory");
	fDGEMV('N',n,n,1.0,h->gs,n,h->gr,1,0.0,h->mg,1);		// Massweighted Gradient(internal) = G^(1/2)*Gradient(internal)
	fDGEMM('N','N',n,n,n,1.0,h->g,n,h->fr,n,0.0,h->mf,n);		// Massweighted Hessian(internal) = G*Hessian(internal)

	if(detail) fprintf(stderr," uG= \n");
	if(detail) printrectangularmatrix(n,1,h->mg);
	if(detail) fprintf(stderr,"uuH= \n");
	if(detail) printrectangularmatrix(n,n,h->mf);

	fDGEMM('N','N',n,n,n,1.0,h->fr,n,h->gs,n,0.0,uf,n);		// uf = Hessian(internal)*G^(1/2)
	fDGEMM('T','N',n,n,n,1.0,h->gs,n,uf,n,0.0,h->mf,n);		// Massweighted Hessian(internal) = G^(1/2)*Hessian(internal)*G^(1/2)
	if(detail) fprintf(stderr,"uHu= \n");
	if(detail) printrectangularmatrix(n,n,h->mf);

	for(i=0;i<n;i++)						// explicit calculation of Massweighted Hessian(internal)
	{
		for(j=0;j<n;j++)
		{
			e=0;
			for(k=0;k<n;k++)
				for(l=0;l<n;l++)
					e+=h->gs[rindex(i,k,n,n)]*h->fr[rindex(k,l,n,n)]*h->gs[rindex(l,j,n,n)];
			h->mf[rindex(i,j,n,n)]=e;
		}
	}
	if(detail) fprintf(stderr,"uHu= \n");
	if(detail) printrectangularmatrix(n,n,h->mf);
	for(j=0;j<n;j++)						// symmetrice mf
		for(i=0;i<=j;i++)
			h->uf[uindex(i,j)]=0.5*(h->mf[rindex(i,j,n,n)]+h->mf[rindex(j,i,n,n)]);
	if(detail) fprintf(stderr,"uHu= \n");
	if(detail) printtriangularmatrix(n,n,h->uf);
	
	free(uf);
	return(h->nr);
}


// this function calculates the eigenvalues and the eigenvectors of the force constant matrix (Hessian) 
// in mass-weighted internal coordinates, internal coordinates, and cartesian coordinates

int normalmode(int detail,INTERNAL *h)
{
	int nr=0,INFO=0,LWORK=3*h->nr;
	double *WORK=NULL;
	nr=massweightinginternal(0,h);
	if(NULL==(WORK=(double *)calloc(LWORK,sizeof(double)))) ued3error("lack of memory");
	fprintf(stderr," memory for matrix diagonalization is allocated\n");
	fDSPEV('V','U',h->nr,h->uf,h->ev,h->vq,h->nr,WORK,&INFO);			// calculate eigenvalues and eigenvectors 
											// in mass-weighted internal coordinates		
	fDGEMM('N','N',h->nr,h->nr,h->nr,1.0,h->gs,h->nr,h->vq,h->nr,0.0,h->vr,h->nr);	// EigenvectorMatrix(internal) = G^(1/2)*EigenvectorMatrix
	fDGEMM('N','N',h->nc,h->nr,h->nr,1.0,h->bi,h->nc,h->vr,h->nr,0.0,h->vx,h->nc);	// EigenvectorMatrix(cartesian) = B^(-1)*G^(1/2)*EigenvectorMatrix
	fprintf(stderr," mass weighted force constant is diagonalized\n");
	free(WORK);
	return(nr);
}


// The square function

double dsqr(double x)
{
	return(x*x);
}


// The sinc function

double sinlxlx(double lx)
{
	if(lx==0) return 1;
	else return(sinl((long double)lx)/((long double)lx));
}


double evaluatelh(double lm,double a)
{
	return(lm*M_SQRT2/sqrt(1+sqrt(1+6*a*a*lm*lm)));
}

double estimatea(double dr,double lm)
{
	// lm^2 = lh^2 + 1.5 a^2 lh^4
	// dr  = 1.5 a lh^2 + 1.5 a lm^2 = 3 a lh^2 + 2.25 a^3 lh^6
	double a=0,b=0,lm2=lm*lm,lh2=dsqr(evaluatelh(lm,a));
	int i;
	for(i=0;i<16;i++)
	{
		lh2=dsqr(evaluatelh(lm,a));
		a=dr/(1.5*(lh2+lm2));
		fprintf(stderr,"%5d a= %lf lh2= %lf lm2=%lf\n",i,a,lh2,lm2);
		if(fabs(a-b)<1.0E-8/Angstrom) break; else b=a;
	}
	fprintf(stderr,"\n");
	return(a);
}


double displacemolecule_finite(int nr,int nc,ZMT *z,double *r,double *w,double *x)
{
	int detail=0;
	int i,j,n=6+nr+nc;
	double step=1,rd2=0,rd1=0,*ri,*rd,*xd,*bi;
	double rlimit=dsqr(SMALLNUMBER*Angstrom);
	double dlimit=dsqr(SMALLDIFFERENCE*Angstrom);
	double slimit=dsqr(0.1*Angstrom);
	if(NULL==(ri=(double *)calloc(nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(rd=(double *)calloc(nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(xd=(double *)calloc(nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(bi=(double *)calloc(nc*nr,sizeof(double)))) ued3error("lack of memory");

	for(i=0;i<n;i++)
	{
		instantaneous(nc,nr,z,x,w,ri,bi);
		for(j=0;j<nr;j++) 
			switch(z[j].type)
			{
			case DIHEDRAL_ANGLE:
			case DIHEDRAL_360:
				rd[j]=dihedraldifference(r[j]-ri[j]);
				break;
			default:
				rd[j]=r[j]-ri[j];
				break;
			}
		fDGEMV('N',nc,nr,1.0,bi,nc,rd,1,0.0,xd,1);
		rd2=fDDOT(nc,xd,1,xd,1);
		if(rd2<rlimit) break;
		if(rd2<slimit) step=1; else { step=sqrt(slimit/rd2); i--; }
		for(j=0;j<nc;j++) x[j]+=step*xd[j];
		if(fabs(rd1-rd2)<dlimit)
		{
			if(rd2>rlimit)
				fprintf(stderr," # not improving at i= %d rd2= %le #\n",i,rd2);
			break;
		}
		else rd1=rd2;
		if(detail)
		{
			fprintf(stderr," rd2= %le \n",rd2);
			if(detail>1) formattedvector(nc,xd,"xd");
			if(detail>2) formattedvector(nr,rd,"rd");
		}
		if(detail)
		{
			if(detail>3) formattedvector(nc,x,"x");
		}
	}
	if(i==n) fprintf(stderr," # not improved after i= %d rd2= %le #\n",i,rd2);

	free(bi);
	free(xd);
	free(rd);
	free(ri);
	return(rd2);
}


// This function takes the displaced (due to vibration) internal coordinates, calculates the corresponding cartesian diplacement,
// then constructs a series of k cartesian steps to reach the internal coordinate point r until the stepsize falls below a threshold value.
// Once this convergence is achieved, the vector x holds the displaced, converged, cartesian coordinates.
// The return value is the convergence indicator

double displacemolecule(int detail,int nr,int nc,ZMT *z,double *rt,double *w,double *x)
{
//	detail=0;
	int i,j,k=0,n=6+nr+nc;
	double step=1,rd2=0,rd1=0,*r,*ri,*rd,*xd,*bi;
	double rlimit=dsqr(SMALLNUMBER*Angstrom);
	double dlimit=dsqr(SMALLDIFFERENCE*Angstrom);
	double slimit=dsqr(0.02*Angstrom);
	double lambda=1,sd=0.2;
	if(NULL==(r =(double *)calloc(nr,sizeof(double)))) ued3error("lack of memory");		// displaced internal coordinates 
	if(NULL==(ri=(double *)calloc(nr,sizeof(double)))) ued3error("lack of memory");		// the initial values of the internal coordinates
	if(NULL==(rd=(double *)calloc(nr,sizeof(double)))) ued3error("lack of memory");		// difference in internal coordinates 
												// due to cartesian displacement
	if(NULL==(xd=(double *)calloc(nc,sizeof(double)))) ued3error("lack of memory");		// displacement in cartesian coordinates
	if(NULL==(bi=(double *)calloc(nc*nr,sizeof(double)))) ued3error("lack of memory");	// inverse of B Matrix

	for(j=0;j<nr;j++) 					// for all redundant internal coordinates, assign the r values
								// corresponding to the "jumping-to"-point
		switch(z[j].type)
		{
		case DIHEDRAL_ANGLE:
			r[j]=dihedraldifference(rt[j]);
			break;
		case DIHEDRAL_360:
			r[j]=dihedraldifference(rt[j]-M_PI)+M_PI;
			break;
		case BENDING_ANGLE:
			if((rt[j]>M_PI)||(rt[j]<0))
				fprintf(stderr," #### invalid bending angle a[%d]= %lf\n",j+1,rt[j]);
			if(rt[j]>M_PI) r[j]=2*M_PI-rt[j];
			if(rt[j]<0) r[j]=-rt[j];
			r[j]=dihedraldifference(rt[j]-M_PI_2)+M_PI_2;
			break;
		case BENDING2_ANGLE:
			r[j]=dihedraldifference(rt[j]);
			break;
		default:
			r[j]=rt[j];
			break;
		}
	for(i=0;i<n;i++)
	{
		for(k=0;k<16;k++)		// This loop constructs a series of k cartesian steps to reach the internal coordinate point r
		{
			instantaneous(nc,nr,z,x,w,ri,bi);		// this function is run to assign ri and bi at the updated cartesian coordinates x
			for(j=0;j<nr;j++) 
				switch(z[j].type)
				{
				case DIHEDRAL_ANGLE:
				case DIHEDRAL_360:
					rd[j]=dihedraldifference(r[j]-ri[j]);
				//	if(rd[j]>sd) rd[j]=sd;		// this makes converge slower
				//	if(rd[j]<-sd) rd[j]=-sd;	// this makes converge slower
					break;
				default:
					rd[j]=r[j]-ri[j];		// rd is the displacement value from previous position, ri
					break;
				}
			fDGEMV('N',nc,nr,1.0,bi,nc,rd,1,0.0,xd,1);	// xd := bi*rd , xd are are the cartesian displacements
			rd2=fDDOT(nc,xd,1,xd,1);			// compute the dot product of the xd vector
			if(rd2<rd1) lambda/=2; else lambda*=2;		// depending on the result increase or decrease lambda
			if(lambda>8) lambda=8;	
			if(rd2<slimit) step=1; else step=sqrt(slimit/rd2); 
			if(detail>1) fprintf(stderr," step= %lf lambda= %lf\n",step,lambda);
			for(j=0;j<nc;j++) x[j]+=step/(1+lambda)*xd[j];	// x is a further displaced cartesian coordinate along the direction of xd
			if(rd2<slimit) break;				// if the rd2 is below the threshold value, then break the loop
		}
		if(rd2<rlimit) break;					// if the rd2 is below the threshold value, then break the loop
		if(fabs(rd1-rd2)<dlimit)
		{
			if(rd2>rlimit)
				fprintf(stderr," # not improving at i= %d rd2= %le #\n",i,rd2);
			break;
		}
		else rd1=rd2;
		if(detail)
		{
			fprintf(stderr," rd2= %le i= %d\n",rd2,i);
			if(detail>3) formattedvector(nr,ri,"ri");
			if(detail>2) formattedvector(nr,rd,"rd");
			if(detail>1) formattedvector(nc,xd,"xd");
			if(detail>4) formattedvector(nc,x,"x");
		}
	}
	if(i==n) fprintf(stderr," # not improved after i= %d rd2= %le #\n",i,rd2);

	free(bi);
	free(xd);
	free(rd);
	free(ri);
	free(r);
	return(rd2);
}

double refinemolecule(int detail,int nr,int nc,ZMT *z,double *gqr,double *w,double *x)
{	// this is worse
	detail=0;
	int i,j,k=0,n=6+nr+nc;
	double step=1,rd2=0,rd1=0,*gr,*ri,*rd,*xd;
	double *bt,*mb,*g,*gi,*gs,*gq,*bi;
	double rlimit=dsqr(SMALLNUMBER*Angstrom);
	double dlimit=dsqr(SMALLDIFFERENCE*Angstrom);
	double slimit=dsqr(1.0*Angstrom);
	if(NULL==(gr=(double *)calloc(nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(ri=(double *)calloc(nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(rd=(double *)calloc(nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(xd=(double *)calloc(nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(bt=(double *)calloc(nc*nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(mb=(double *)calloc(nc*nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(g =(double *)calloc(nr*nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(gi=(double *)calloc(nr*nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(gs=(double *)calloc(nr*nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(gq=(double *)calloc(nr*nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(bi=(double *)calloc(nc*nr,sizeof(double)))) ued3error("lack of memory");

	for(i=0;i<n;i++)
	{
		calculatebtmatrix(nr,nc,z,ri,x,bt);
		calculatemibtmatrix(nc,nr,bt,w,mb);
		calculategtmatrix(nc,nr,bt,mb,g);
		calculategsmatrix(nr,g,gi,gs,gq);
		calculatebimatrix(nc,nr,mb,gi,bi);
		fDGEMV('N',nr,nr,1.0,gq,nr,ri,1,0.0,gr,1);
		fDAXPY(nr,-1.0,gqr,1,gr,1);
		fDGEMV('N',nr,nr,-1.0,gs,nr,gr,1,0.0,rd,1);
		for(j=0;j<nr;j++) 
			switch(z[j].type)
			{
			case DIHEDRAL_ANGLE:
			case DIHEDRAL_360:
				rd[j]=dihedraldifference(rd[j]);
				break;
			default:
				break;
			}
		fDGEMV('N',nc,nr,1.0,bi,nc,rd,1,0.0,xd,1);
		rd2=fDDOT(nc,xd,1,xd,1);
		if(rd2<rlimit) break;
		if(rd2<slimit) { step=1; k=0;} else { step=sqrt(slimit/rd2); i--; k++;}
		if(k>16) { k=1; i++; }
		for(j=0;j<nc;j++) x[j]+=step*xd[j];
		if(fabs(rd1-rd2)<dlimit)
		{
			if(rd2>rlimit)
				fprintf(stderr," # not improving at i= %d rd2= %le #\n",i,rd2);
			break;
		}
		else rd1=rd2;
		if(detail)
		{
			fprintf(stderr," rd2= %le i= %d\n",rd2,i);
			if(detail>4) formattedvector(nr,gqr,"gqr");
			if(detail>3) formattedvector(nr,ri,"ri");
			if(detail>3) formattedvector(nr,gr,"gr");
			if(detail>2) formattedvector(nr,rd,"rd");
			if(detail>1) formattedvector(nc,xd,"xd");
		}
		if(detail)
		{
			if(detail>3) formattedvector(nc,x,"x");
		}
	}
	if(i==n) fprintf(stderr," # not improved after i= %d rd2= %le #\n",i,rd2);

	free(bi);
	free(gq);
	free(gs);
	free(gi);
	free(g );
	free(mb);
	free(bt);
	free(xd);
	free(rd);
	free(ri);
	free(gr);
	return(rd2);
}


// This function excites the molecule (along its normal modes of vibration) by an amount corresponding to the mean amplitude of vibration 
// and calculates a first and second order approximation of the molecular structure due to this motion in internal coordinates.
// On return, the vector x holds the displaced cartesian coordinates in either the positive or negative direction of l

double excitemolecule(int detail,int nr,int nc,ZMT *z,double l,double *v,double *r0,double *w,double *x)
{
	int j;
	double t=1,lv2=0,*rd=NULL;
	if(NULL==(rd=(double *)calloc(nr,sizeof(double)))) ued3error("lack of memory");
	for(j=0;j<nr;j++)				// for all redundant internal coordinates
	{
		switch(z[j].type)
		{
		case DIHEDRAL_ANGLE:
		case DIHEDRAL_360:
		//	t*=sinlxlx(l*v[j]);
			lv2+=(l*v[j])*(l*v[j]);
			break;
		default:
			break;
		}
	}
	t=sinlxlx(sqrt(lv2));				// empirical (by STP) displacement scalefactor 
	for(j=0;j<nr;j++)
	{
		switch(z[j].type)
		{
		case DIHEDRAL_ANGLE:
		case DIHEDRAL_360:
			rd[j]=r0[j]+(l*v[j]);			// displacement of internal coordinate due to torsion
			rd[j]+=0.75*z[j].a*dsqr(l*v[j]);	// second order correction, z[j].a=0 by default
			break;
		default:
			rd[j]=r0[j]+(l*v[j])*t;			// displacement of internal coordinate due to vibration
			rd[j]+=0.75*z[j].a*dsqr(l*v[j]*t);	// second order correction, z[j].a=0 by default
			break;
		}
	}
//	fprintf(stderr," # torsional periodic correction, t= %le l*t= %+lf\n",t,l*t);
	t = displacemolecule(0,nr,nc,z,rd,w,x);
	if(t>dsqr(SMALLNUMBER*Angstrom))
	{
		fprintf(stderr," # not converged, trying again\n");
		formattedvector(nr,rd,"rt");
		t = displacemolecule(detail,nr,nc,z,rd,w,x);
	}
	free(rd);
	return(t);
}

void localcentrifugal_bt(int detail,int i,INTERNAL *h,double *frt)
{
	if(detail)
		fprintf(stderr," & evaluating fictitious force by curvature in B matrix\n");
	int k;
	double *bv=NULL,*ar=NULL,*lc=NULL;
	if(NULL==(bv=(double *)calloc(h->nc*h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(ar=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(lc=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	formattedvector(h->nr,h->vq+rindex(0,i,h->nr,h->nr),"vq");
	formattedvector(h->nr,h->vr+rindex(0,i,h->nr,h->nr),"vr");
	formattedvector(h->nc,h->vx+rindex(0,i,h->nc,h->nr),"vx");
	// per unit mass-weighted momentum
	for(k=0;k<h->nr;k++)
		fDGEMV('T',h->nc,h->nc,M_SQRT2,h->ct+cindex(0,0,k,h->nc,h->nc,h->nr),h->nc
			,h->vx+rindex(0,i,h->nc,h->nr),1
			,0.0,bv+rindex(0,k,h->nc,h->nr),1);
	fDGEMV('T',h->nc,h->nr,M_SQRT2,bv,h->nc,h->vx+rindex(0,i,h->nc,h->nr),1,0.0,ar,1);// per unit energy
	formattedvector(h->nr,ar,"ar");
	// nonlinearity elimination
//	formattedvector(h->nr,h->lc,"lcc");
	for(k=0;k<h->nr;k++)
		lc[k]=h->lc[k]*2*h->vr[rindex(k,i,h->nr,h->nr)]*h->vr[rindex(k,i,h->nr,h->nr)];
	formattedvector(h->nr,lc,"lc");
	for(k=0;k<h->nr;k++) ar[k]-=lc[k];
	formattedvector(h->nr,ar,"arl");
	fDGEMV('N',h->nr,h->nr,1.0,h->gi,h->nr,ar,1,0.0,frt,1);	// CHANGE TO INSTANTANEOUS
	formattedvector(h->nr,frt,"fr");
	free(lc);
	free(ar);
	free(bv);
	return;
}

void localcentrifugal_bi(int detail,int i,INTERNAL *h,double *frt)
{
	if(detail)
		fprintf(stderr," & evaluating fictitious force by curvature in B^-1 matrix\n");
	int k;
	double *bv=NULL,*ax=NULL,*ar=NULL,*lc=NULL;
	if(NULL==(bv=(double *)calloc(h->nc*h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(ax=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(ar=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(lc=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	formattedvector(h->nr,h->vq+rindex(0,i,h->nr,h->nr),"vq");
	formattedvector(h->nr,h->vr+rindex(0,i,h->nr,h->nr),"vr");
	formattedvector(h->nc,h->vx+rindex(0,i,h->nc,h->nr),"vx");
	// per unit mass-weighted momentum
	for(k=0;k<h->nc;k++)
		fDGEMV('T',h->nr,h->nr,M_SQRT2,h->bi2+cindex(0,0,k,h->nr,h->nr,h->nc),h->nr
			,h->vr+rindex(0,i,h->nr,h->nr),1
			,0.0,bv+rindex(0,k,h->nr,h->nc),1);
	fDGEMV('T',h->nr,h->nc,M_SQRT2,bv,h->nr,h->vr+rindex(0,i,h->nr,h->nr),1,0.0,ax,1);// per unit energy
	fDGEMV('T',h->nc,h->nr,-1.0,h->bt,h->nc,ax,1,0.0,ar,1);
	formattedvector(h->nr,ar,"ar");
	// nonlinearity elimination
//	formattedvector(h->nr,h->lc,"lcc");
	for(k=0;k<h->nr;k++)
		lc[k]=h->lc[k]*2*h->vr[rindex(k,i,h->nr,h->nr)]*h->vr[rindex(k,i,h->nr,h->nr)];
	formattedvector(h->nr,lc,"lc");
	for(k=0;k<h->nr;k++) ar[k]-=lc[k];
	formattedvector(h->nr,ar,"arl");
//p	fDGEMV('N',h->nc,h->nr,1.0,h->bi,h->nc,ar,1,0.0,bt,1);
//p	fDGEMV('T',h->nc,h->nr,1.0,h->bt,h->nc,bt,1,0.0,ar,1);
//p	formattedvector(h->nr,ar,"arl");
	fDGEMV('N',h->nr,h->nr,1.0,h->gi,h->nr,ar,1,0.0,frt,1);	// CHANGE TO INSTANTANEOUS
	formattedvector(h->nr,frt,"fr");
	free(lc);
	free(ar);
	free(ax);
	free(bv);
	return;
}

// m v^2 = u^2 = 2 T where u is mass-weighted velocity
// vq is a unit vector, and vq^2 = 1
// the kinetic energy of each unit vector is 1/2 eu
// the actual velocity should be u = sqrt(2T)vq
void localcentrifugal_gi(int detail,int i,INTERNAL *h,double *frt)
{
	if(detail) 
		fprintf(stderr," & evaluating fictitious force by curvature in G^-1 matrix\n");
	int j,k;
	double sf,*bv=NULL,*bu=NULL,*ar=NULL,*lc=NULL;
	if(NULL==(bv=(double *)calloc(h->nr*h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(bu=(double *)calloc(h->nr*h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(ar=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(lc=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	formattedvector(h->nr,h->vq+rindex(0,i,h->nr,h->nr),"vq");
	formattedvector(h->nr,h->vr+rindex(0,i,h->nr,h->nr),"vr");
	formattedvector(h->nc,h->vx+rindex(0,i,h->nc,h->nr),"vx");
	fDCLEAR(h->nr,frt,1);
	// multiplying to depth
	for(j=0;j<h->nr;j++)
		for(k=0;k<h->nr;k++)
			bu[rindex(k,j,h->nr,h->nr)]
				=M_SQRT2*fDDOT(h->nr,h->gi2+cindex(k,j,0,h->nr,h->nr,h->nc),h->nr*h->nr
					    ,h->vr+rindex(0,i,h->nr,h->nr),1);
	fDGEMV('N',h->nr,h->nr,0.5*M_SQRT2,bu,h->nr,h->vr+rindex(0,i,h->nr,h->nr),1,1.0,frt,1);// major
	if(detail>1) formattedvector(h->nr,frt,"fl");
	for(k=0;k<h->nr;k++)
		fDGEMV('T',h->nr,h->nr,M_SQRT2,h->gi2+cindex(0,0,k,h->nr,h->nr,h->nc),h->nr
			,h->vr+rindex(0,i,h->nr,h->nr),1
			,0.0,bv+rindex(0,k,h->nr,h->nr),1);
			// store row first = multiplying vector transpose = v^T T
	fDGEMV('N',h->nr,h->nr,-M_SQRT2,bv,h->nr,h->vr+rindex(0,i,h->nr,h->nr),1,1.0,frt,1);// minor
	// per unit kinetic energy
	formattedvector(h->nr,frt,"fr");
	fDGEMV('N',h->nr,h->nr,1.0,h->gs,h->nr,frt,1,0.0,ar,1);	// borrowing ar to display
	formattedvector(h->nr,ar,"fq");


	fDGEMV('N',h->nr,h->nr,1.0,h->g,h->nr,frt,1,0.0,ar,1);	// CHANGE TO INSTANTANEOUS ?
	formattedvector(h->nr,ar,"ar");

//	sf=fDDOT(h->nr,ar,1,h->vr+rindex(0,i,h->nr,h->nr),1)/fDDOT(h->nr,h->vr+rindex(0,i,h->nr,h->nr),1,h->vr+rindex(0,i,h->nr,h->nr),1);
//	fDAXPY(h->nr,-sf,h->vr+rindex(0,i,h->nr,h->nr),1,ar,1);	// TESTING
//	fprintf(stderr," self acceleration = %le\n",sf);
//	formattedvector(h->nr,ar,"as");

	// nonlinearity elimination
//	formattedvector(h->nr,h->lc,"lcc");
	for(k=0;k<h->nr;k++)
		lc[k]=h->lc[k]*2*h->vr[rindex(k,i,h->nr,h->nr)]*h->vr[rindex(k,i,h->nr,h->nr)];
	formattedvector(h->nr,lc,"lc");
	for(k=0;k<h->nr;k++) ar[k]-=lc[k];
	formattedvector(h->nr,ar,"arl");
	fDGEMV('N',h->nr,h->nr,1.0,h->gi,h->nr,ar,1,0.0,frt,1);	// CHANGE TO INSTANTANEOUS
	formattedvector(h->nr,frt,"frt");

	free(lc);
	free(ar);
	free(bu);
	free(bv);
	return;
}

void localcentrifugal_gil(int detail,int i,INTERNAL *h,double *frt)
{
	if(detail) 
		fprintf(stderr," & evaluating fictitious force by first order curvature in G^-1 matrix\n");
	int j,k,l;
	double *bv=NULL,*bu=NULL,*ar=NULL,*lc=NULL;
//	if(NULL==(bv=(double *)calloc(h->nr*h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(bu=(double *)calloc(h->nr*h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(ar=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(lc=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	formattedvector(h->nr,h->vq+rindex(0,i,h->nr,h->nr),"vq");
	formattedvector(h->nr,h->vr+rindex(0,i,h->nr,h->nr),"vr");
	formattedvector(h->nc,h->vx+rindex(0,i,h->nc,h->nr),"vx");

	fDCLEAR(h->nr,frt,1);
	for(j=0;j<h->nr;j++)
		for(k=0;k<h->nr;k++)
			bu[rindex(k,j,h->nr,h->nr)]
				=M_SQRT2*fDDOT(h->nr,h->gi2+cindex(k,j,0,h->nr,h->nr,h->nc),h->nr*h->nr
					    ,h->vr+rindex(0,i,h->nr,h->nr),1);
	fDGEMV('N',h->nr,h->nr,0.5*M_SQRT2,bu,h->nr,h->vr+rindex(0,i,h->nr,h->nr),1,1.0,frt,1);
	// per unit kinetic energy
	formattedvector(h->nr,frt,"fr");
	fDGEMV('N',h->nr,h->nr,1.0,h->g,h->nr,frt,1,0.0,ar,1);	// CHANGE TO INSTANTANEOUS
	formattedvector(h->nr,ar,"ar");
	// nonlinearity elimination
//	formattedvector(h->nr,h->lc,"lcc");
	if(0)	// TESTING // TESTING // TESTING //
	{
		for(k=0;k<h->nr;k++)
			lc[k]=h->lc[k]*2*h->vr[rindex(k,i,h->nr,h->nr)]*h->vr[rindex(k,i,h->nr,h->nr)];
		formattedvector(h->nr,lc,"lc");
		for(k=0;k<h->nr;k++) ar[k]-=lc[k];
		formattedvector(h->nr,ar,"arl");
	}
	fDGEMV('N',h->nr,h->nr,1.0,h->gi,h->nr,ar,1,0.0,frt,1);	// CHANGE TO INSTANTANEOUS
	formattedvector(h->nr,frt,"fr");

	free(lc);
	free(ar);
	free(bu);
//	free(bv);
	return;
}

void localcentrifugal(int detail,int i,INTERNAL *h,double *frt)
{
	switch(h->curvature)
	{
	case CURVATURE_DBTDX:
		localcentrifugal_bt(detail,i,h,frt);
		break;
	case CURVATURE_DBIDR:
		localcentrifugal_bi(detail,i,h,frt);
		break;
	case CURVATURE_DGIDR:
		localcentrifugal_gi(detail,i,h,frt);
		break;
	default:
	case CURVATURE_DGIDR_LINEAR:
		localcentrifugal_gil(detail,i,h,frt);
		break;
	}
	return;
}

void vibrationalcentrifugal(int detail,INTERNAL *h)
{
	int i,nz=h->nr-h->nv;
	double frequency,wavenumber;
	fDCLEAR(h->nr*h->nr,h->fvt,1);
	for(i=nz;i<h->nr;i++)
	{
		frequency=sqrt(fabs(h->ev[i]))*(0.5*M_1_PI);
		wavenumber=frequency/(SpeedLight);
		fprintf(stderr,"%5d: %12.4lf /cm \n",i+1,wavenumber/PerCentimeter);

		localcentrifugal(detail,i,h,h->fvt+rindex(0,i,h->nr,h->nr));
	}
	fprintf(stderr,"\n");
}

void decomposecentrifugalforce(int detail,INTERNAL *h)	// per square of mass-weighted momentum
{
	int i,j,k,nz=h->nr-h->nv;
	double *bv=NULL,*bu=NULL,*fr=NULL,*vq=NULL,*vr=NULL;
	if(NULL==(bv=(double *)calloc(h->nr*h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(bu=(double *)calloc(h->nr*h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(fr=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(vq=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(vr=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");

	for(i=0;i<h->nr;i++)
	{
		fDCLEAR(h->nr,vq,1); vq[i]=1;
		fDGEMM('N','N',h->nr,h->nr,h->nr,1.0,h->gs,h->nr,vq,h->nr,0.0,vr,h->nr);
		fDCLEAR(h->nr,vr,1); vr[i]=1;
	//	fDGEMM('N','N',h->nc,h->nr,h->nr,1.0,h->bi,h->nc,vr,h->nr,0.0,vx,h->nc);
		formattedvector(h->nr,vq,"VQ");
		formattedvector(h->nr,vr,"VR");
		// per unit mass-weighted momentum
		for(k=0;k<h->nr;k++)
			fDGEMV('T',h->nr,h->nr,1.0,h->gi2+cindex(0,0,k,h->nr,h->nr,h->nc),h->nr
				,vr,1
				,0.0,bv+rindex(0,k,h->nr,h->nr),1);	// store row first
		fDGEMV('N',h->nr,h->nr,-1.0,bv,h->nr,vr,1,0.0,fr,1);
		// multiplying to depth
		for(j=0;j<h->nr;j++)
			for(k=0;k<h->nr;k++)
				bu[rindex(k,j,h->nr,h->nr)]
					=fDDOT(h->nr,h->gi2+cindex(k,j,0,h->nr,h->nr,h->nc),h->nr*h->nr
					    	,vr,1);
		fDGEMV('N',h->nr,h->nr,0.5,bu,h->nr,vr,1,1.0,fr,1);
		// per twice unit kinetic energy
		formattedvector(h->nr,fr,"fr");
		fprintf(stderr,"\n");
	}

	free(vr);
	free(vq);
	free(fr);
	free(bu);
	free(bv);
	return;
}
