#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "atomic.h"
#include "constantunits.h"
#include "intrinsic.h"
#include "programming.h"

#include "errors.h"
#include "outputs.h"
#include "interface.h"
#include "coordinates.h"
#include "calculate.h"
#include "cartesian.h"
#include "internal.h"
#include "vibrational.h"
#include "modewalk.h"

int moderun(int detail,THERMAL *t,INTERNAL *h)
{
	int i,j,k,m,v,nrot=3,nx=6,nm=h->na*(h->na-1)/2;
	double *bt=NULL;
	double *rd=NULL,*xp=NULL,*xn=NULL,*xs=NULL;
	double *dr=NULL,*dvr=NULL,*la=NULL,*me=NULL,*m0=NULL,*m1=NULL,*m2=NULL,*m3=NULL,*m4=NULL;
	double a,Ti,ccj,l1,l2,frequency,wavenumber;
	double dq,r,p,q,hv,kT,Tq=0,Eh=0,Ez=0,*dTdq=NULL,*gqr=NULL,*cr=NULL,*cv=NULL,*rs=NULL,*fs=NULL;
	int nz=h->nr-h->nv,nvib=WALK_FINER/(1+sqrt(h->na));	// fineness of mode walk
	double vib=1.0/nvib,l2sum;
	if(nz<0) nz=0;

	if(NULL==(dvr=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(dr=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(la=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(me=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(m0=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(m1=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(m2=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(m3=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(m4=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(bt=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(xs=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(xp=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(xn=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(rd=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(rs=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(fs=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(cr=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(cv=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(gqr=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(dTdq=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");

	for(k=1;k<h->na;k++)
		for(j=0;j<k;j++)
			me[mindex(j,k)]=distance(j,k,h->cc,bt);

	nrot=externalrotational(detail,h);
	rotationalcentrifugaldistortion(detail,h,nrot,t->Tr,cr);
	if(detail) fprintf(stderr,"\n");
	if(detail) formattedvector(h->nr,h->r,"r0");
	formattedvector(h->nr,cr,"cr");

	fDCOPY(h->nr,h->r,1,rs,1);
	fDAXPY(h->nr,1.0,cr,1,rs,1);
	fDCOPY(h->nc,h->cc,1,xs,1);
	ccj=excitemolecule(detail,h->nr,h->nc,h->z,0.0,h->vr+rindex(0,nz,h->nr,h->nr),rs,h->m,xs);
//	displacemolecule(detail,h->nr,h->nc,h->z,rs,h->m,xs);
	fprintf(stderr,"\n");

	for(k=1;k<h->na;k++)
		for(j=0;j<k;j++)
			dr[mindex(j,k)]=distance(j,k,xs,bt)-me[mindex(j,k)];

	nx=normalmode(detail,h);
	a=thermalzeropointfraction(t->Tv,h);
//	fprintf(stderr," zero point fraction = %lf\n",a);
	vibrationalcentrifugal(detail,h);	// fvt = force per unit kinetic energy
	vibrationalcentrifugalforce(detail,h,t->Tv,fs);	// to print out dv
	fDCLEAR(h->nr,fs,1);
	for(i=nz;i<h->nr;i++)
	{
		Ti=t->Tv[i-nz];
		frequency=sqrt(fabs(h->ev[i]))*(0.5*M_1_PI);
		wavenumber=frequency/(SpeedLight);
		hv=PlanckConstant*frequency;
		kT=BoltzmannConstant*Ti;
		Tq=0.5*(thermalvibrationalenergy(hv,kT)+a*0.5*hv);
		fDAXPY(h->nr,Tq,h->fvt+rindex(0,i,h->nr,h->nr),1,fs,1);
	}
	fDGEMV('N',h->nr,h->nr,1.0,h->gs,h->nr,fs,1,0.0,dTdq,1);
	formattedvector(h->nr,fs,"fs");
	formattedvector(h->nr,dTdq,"dTdq");

	fDCOPY(h->nr,h->r,1,rs,1);
	fDAXPY(h->nr,1.0,cr,1,rs,1);
	fDCOPY(h->nc,h->cc,1,xs,1);
	ccj=excitemolecule(detail,h->nr,h->nc,h->z,0.0,h->vr+rindex(0,nz,h->nr,h->nr),rs,h->m,xs);
//	displacemolecule(detail,h->nr,h->nc,h->z,rs,h->m,xs);
	if(detail) fprintf(stderr,"\n");
	if(detail) fprintf(stderr,"distorted geometry\n");
	if(detail) formattedvector(h->nc,h->cc,"x0");
	if(detail) formattedvector(h->nc,xs,"xs");
	fprintf(stderr,"\n");

	for(k=1;k<h->na;k++)
		for(j=0;j<k;j++)
			m0[mindex(j,k)]=distance(j,k,xs,bt);	// m0=me+dr+dvs

	fDCLEAR(nm,m0,1);
	fprintf(stderr," normal mode run \n");
	for(i=nz;i<h->nr;i++)
	{
		Ti=t->Tv[i-nz];
		frequency=sqrt(fabs(h->ev[i]))*(0.5*M_1_PI);
		wavenumber=frequency/(SpeedLight);
		hv=PlanckConstant*frequency;
		kT=BoltzmannConstant*Ti;
		l2=h_8pi2c/(wavenumber*tanh(hc_2k*wavenumber/Ti));
		l1=sqrt(l2);

		fprintf(stderr," evaluating amplitude and shrinkage by running\n");
		fprintf(stderr,"%5d: %12.4lf /cm l1= %lf sqrt(amu) A l2= %lf amu A^2 at %.0lf K\n"
			,i+1,wavenumber/PerCentimeter,l1/SqrtAMUAngstrom,l2/AMUAngstrom2,Ti/Kelvin);
		Eh=thermalvibrationalenergy(hv,kT);	// thermal energy
		Ez=0.5*hv;
		Tq=a*Ez+Eh-0;
		p=0.5*M_2_SQRTPI*M_SQRT2/l1;
		q=p;
		if(detail) fprintf(stderr," # zero point energy=  %le eu and kT= %le eu = %lf /cm\n",0.5*hv,kT,kT*EU_CM);
		if(detail) fprintf(stderr," # classical turning point= %le sqrt(amu)A at %le eu\n",sqrt(Eh/h->ev[i])/SqrtAMUAngstrom,(kT+0.5*hv));
		fprintf(stderr," # quantum turning point=   %le sqrt(amu)A at %le eu\n",l1/SqrtAMUAngstrom,Eh);

		if(REMOVE_SELF_FORCE)	// 1 removes self acceleration
			fDGEMV('N',h->nr,h->nr,-Tq,h->gs,h->nr,h->fvt+rindex(0,i,h->nr,h->nr),1,0.0,fs,1);
		else
			fDCLEAR(h->nr,fs,1);	// include self force
		fDAXPY(h->nr,1.0,dTdq,1,fs,1);
		dq=fDDOT(h->nr,fs,1,h->vq+rindex(0,i,h->nr,h->nr),1)/h->ev[i];
		fprintf(stderr," dq= %+lf l1= %lf f= %+le e= %+le\n",dq,l1,dq*h->ev[i],h->ev[i]);
		formattedvector(h->nr,h->vq+rindex(0,i,h->nr,h->nr),"vq");

		if(detail>1)
		{
			int j;
			for(j=nz;j<h->nr;j++)
			{
				double Tj=t->Tv[j-nz],fj=sqrt(fabs(h->ev[j]))*(0.5*M_1_PI);
				double hvj=PlanckConstant*fj,kTj=BoltzmannConstant*Tj;
				double Kj=0.5*(thermalvibrationalenergy(hvj,kTj)+0.5*hvj);
				fDCLEAR(h->nr,fs,1);
				fDGEMV('N',h->nr,h->nr,Kj,h->gs,h->nr,h->fvt+rindex(0,j,h->nr,h->nr),1,0.0,fs,1);
				double dqj=fDDOT(h->nr,fs,1,h->vq+rindex(0,i,h->nr,h->nr),1)/h->ev[i];
				fprintf(stderr," j= %d dqij= %+lf fij= %+le\n",j+1,dqj,dqj*h->ev[i]);
			}
		}
		fDCOPY(h->nc,xs,1,xp,1);
		fDCOPY(h->nr,rs,1,rd,1);
//WORSE		fDGEMV('N',h->nr,h->nr,1.0,h->gq,h->nr,rd,1,0.0,gqr,1);
//WORSE		fDAXPY(h->nr,dq,h->vq+rindex(0,i,h->nr,h->nr),1,gqr,1);
//WORSE		refinemolecule(detail,h->nr,h->nc,h->z,gqr,h->m,xp);
		ccj=excitemolecule(detail,h->nr,h->nc,h->z,dq,h->vr+rindex(0,i,h->nr,h->nr),rd,h->m,xp);

		fDCOPY(h->nc,xp,1,xn,1);

		if(detail>2)
			fprintf(stderr," # walking probability P= %le at q= %le sqrt(amu) A <T>= %lf \n"
				,p,0.,Tq/(PlanckConstant*SpeedLight)/PerCentimeter);
		for(k=1;k<h->na;k++)
			for(j=0;j<k;j++)
			{
				r=distance(j,k,xp,bt);
				m=mindex(j,k);
				m0[m]+=(r-me[m]);
				m1[m]=(r)*p;
				m2[m]=dsqr(r)*p;
				m3[m]=dsqr(r)*(r)*p;
				m4[m]=dsqr(dsqr(r))*p;
			}
		for(v=1;v<WALK_FARTHER*nvib;v++)
		{
			double lv = v*vib*l1;	// P = 1/(sqrt(2pi)l1) * exp(-0.5 x^2/l1^2)
			p = 0.5*M_2_SQRTPI*M_SQRT2/l1 * exp(-0.5*(v*vib)*(v*vib));
			q+=2*p;
			Tq=a*Ez+Eh-(0.5*h->ev[i]*lv*lv);
			if(Tq<0) Tq=0;
			if(detail>2)
				fprintf(stderr," # walking probability P= %le at q= %le sqrt(amu) A <T>= %le \n"
					,p,lv/SqrtAMUAngstrom,Tq/(PlanckConstant*SpeedLight)/PerCentimeter);

			fDCOPY(h->nr,rs,1,rd,1);
		//	fDAXPY(h->nr,(Tq-0.5*(Ez+Eh)),h->fvt+rindex(0,i,h->nr,h->nr),1,rd,1);
			ccj=excitemolecule(detail,h->nr,h->nc,h->z,dq+lv,h->vr+rindex(0,i,h->nr,h->nr),rd,h->m,xp);
			ccj=excitemolecule(detail,h->nr,h->nc,h->z,dq-lv,h->vr+rindex(0,i,h->nr,h->nr),rd,h->m,xn);
//WORSE			fDCOPY(h->nr,rs,1,rd,1);
//WORSE			fDGEMV('N',h->nr,h->nr,1.0,h->gq,h->nr,rd,1,0.0,gqr,1);
//WORSE			fDAXPY(h->nr,dq+lv,h->vq+rindex(0,i,h->nr,h->nr),1,gqr,1);
//WORSE			refinemolecule(detail,h->nr,h->nc,h->z,gqr,h->m,xp);
//WORSE			fDCOPY(h->nr,rs,1,rd,1);
//WORSE			fDGEMV('N',h->nr,h->nr,1.0,h->gq,h->nr,rd,1,0.0,gqr,1);
//WORSE			fDAXPY(h->nr,dq-lv,h->vq+rindex(0,i,h->nr,h->nr),1,gqr,1);
//WORSE			refinemolecule(detail,h->nr,h->nc,h->z,gqr,h->m,xn);
			for(k=1;k<h->na;k++)
				for(j=0;j<k;j++)
				{
					m=mindex(j,k);
					r=distance(j,k,xp,bt);
					m1[m]+=(r)*p;
					m2[m]+=dsqr(r)*p;
					m3[m]+=dsqr(r)*(r)*p;
					m4[m]+=dsqr(dsqr(r))*p;
					r=distance(j,k,xn,bt);
					m1[m]+=(r)*p;
					m2[m]+=dsqr(r)*p;
					m3[m]+=dsqr(r)*(r)*p;
					m4[m]+=dsqr(dsqr(r))*p;
				}
		}
		if(detail>2) fprintf(stderr," sum of probability = %le\n",q);
		for(j=0;j<nm;j++)
		{
			m1[j]/=q;
			m2[j]/=q;
			m3[j]/=q;
			m4[j]/=q;
			dvr[j]+=(m1[j]-me[j]);
			la[j]+=m2[j]-dsqr(m1[j]);
		}

		//formattedvector(nm,ma,"ra");
		//formattedvector(nm,mb,"la");
		fprintf(stderr,"%8s"," ra=");
		for(j=0;j<nm;j++)
			fprintf(stderr,"%8.4lf",(m1[j])/Angstrom);
		fprintf(stderr,"\n");
		fprintf(stderr,"%8s"," dv=");
		for(j=0;j<nm;j++)
			fprintf(stderr,"%8.4lf",(m1[j]-me[j])/Angstrom);
		fprintf(stderr,"\n");
		fprintf(stderr,"%8s"," la=");
		for(j=0;j<nm;j++)
			fprintf(stderr,"%8.4lf",sqrt(fabs(m2[j]-dsqr(m1[j])))/Angstrom);
		fprintf(stderr,"\n");
						
	}
	fprintf(stdout,"%8d",h->na*(h->na-1)/2);
	for(i=nz;i<h->nr;i++)
		fprintf(stdout,"  %.0lf",t->Tv[i-nz]/Kelvin);
	fprintf(stdout,"\n");
//	Ti=0; for(i=nz;i<h->nr;i++) Ti+=t->Tv[i-nz];
//	fprintf(stdout,"  %.0lf\n",Ti/(h->nr-nz));
	for(k=1;k<h->na;k++)
		for(j=0;j<k;j++)
		{

			l2sum=0;
			fDCLEAR(h->nc,bt,1);
			r=distance(j,k,xs,bt);
			for(i=nz;i<h->nr;i++)
			{
				Ti=t->Tv[i-nz];
				frequency=sqrt(fabs(h->ev[i]))*(0.5*M_1_PI);
				wavenumber=frequency/SpeedLight;
				l2=h_8pi2c/(wavenumber*tanh(hc_2k*wavenumber/Ti));
				l2sum+=l2*dsqr(fDDOT(h->nc,bt,1,h->vx+rindex(0,i,h->nc,h->nr),1));
			}

			m=mindex(j,k);
			fprintf(stdout,"  %-6s %-6s ",h->ai[j].name,h->ai[k].name);
			fprintf(stdout," re= %15.10lf",(me[m])/Angstrom);
			fprintf(stdout," dr= %15.10lf",(dr[m])/Angstrom);
			if(detail>1) fprintf(stdout," d0= %15.10lf",(m0[m])/Angstrom);
			if(detail>1) fprintf(stdout," du= %15.10lf",(dvr[m]-m0[m])/Angstrom);
			fprintf(stdout," dv= %15.10lf",(dvr[m])/Angstrom);
			fprintf(stdout," la= %15.10lf",sqrt(la[m])/Angstrom);
			fprintf(stdout," lh= %15.10lf",sqrt(l2sum)/Angstrom);
			fprintf(stdout," al= %15.10lf",sqrt((la[m]-l2sum)/(1.5*dsqr(l2sum)))*Angstrom);
			fprintf(stdout," ad= %15.10lf",estimatea(dvr[m]/Angstrom,sqrt(la[m])/Angstrom));
			fprintf(stdout,"\n");
		}

	free(dTdq);
	free(gqr);
	free(cr);
	free(cv);
	free(fs);
	free(rs);
	free(rd);
	free(xs);
	free(xp);
	free(xn);
	free(bt);
	free(m4);
	free(m3);
	free(m2);
	free(m1);
	free(m0);
	free(me);
	free(la);
	free(dvr);
	free(dr);
	return(0);
}

int moderun2(int detail,THERMAL *t,INTERNAL *h)
{
	int i,j,k,m,v,nrot=3,nx=6,nm=h->na*(h->na-1)/2;
	double *bt=NULL;
	double *rd=NULL,*xp=NULL,*xn=NULL,*xs=NULL;
	double *dr=NULL,*dvr=NULL,*la=NULL,*me=NULL,*m0=NULL,*m1=NULL,*m2=NULL,*m3=NULL,*m4=NULL;
	double Ti,ccj,l1,l2,frequency,wavenumber;
	double r,p,q,hv,kT,Tq=0,Eh=0,Ez=0,*dvp=NULL,*cr=NULL,*cv=NULL,*rs=NULL;
	int nz=h->nr-h->nv,nvib=WALK_FINER/(1+sqrt(h->na));	// fineness of mode walk
	double vib=1.0/nvib,l2sum;
	if(nz<0) nz=0;

	if(NULL==(dvr=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(dr=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(la=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(me=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(m0=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(m1=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(m2=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(m3=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(m4=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(bt=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(xs=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(xp=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(xn=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(rd=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(rs=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(cr=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(cv=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(dvp=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");

	for(k=1;k<h->na;k++)
		for(j=0;j<k;j++)
			me[mindex(j,k)]=distance(j,k,h->cc,bt);

	nrot=externalrotational(detail,h);
	rotationalcentrifugaldistortion(detail,h,nrot,t->Tr,cr);
	if(detail) fprintf(stderr,"\n");
	if(detail) formattedvector(h->nr,h->r,"r0");
	formattedvector(h->nr,cr,"cr");

	fDCOPY(h->nr,h->r,1,rs,1);
	fDAXPY(h->nr,1.0,cr,1,rs,1);
	fDCOPY(h->nc,h->cc,1,xs,1);
//	displacemolecule(detail,h->nr,h->nc,h->z,rs,h->m,xs);
	ccj=excitemolecule(detail,h->nr,h->nc,h->z,0.0,h->vr+rindex(0,nz,h->nr,h->nr),rs,h->m,xs);
	fprintf(stderr,"\n");

	for(k=1;k<h->na;k++)
		for(j=0;j<k;j++)
			dr[mindex(j,k)]=distance(j,k,xs,bt)-me[mindex(j,k)];

	nx=normalmode(detail,h);

	vibrationalcentrifugal(detail,h);
	vibrationalcentrifugaldistortion(detail,h,t->Tv,cv);
	if(detail) fprintf(stderr,"\n");
	if(detail) formattedvector(h->nr,h->r,"r0");
	formattedvector(h->nr,cv,"cv");

	fDCOPY(h->nr,h->r,1,rs,1);
	fDAXPY(h->nr,1.0,cr,1,rs,1);
	fDAXPY(h->nr,1.0,cv,1,rs,1);
	fDCOPY(h->nc,h->cc,1,xs,1);
//	displacemolecule(detail,h->nr,h->nc,h->z,rs,h->m,xs);
	ccj=excitemolecule(detail,h->nr,h->nc,h->z,0.0,h->vr+rindex(0,nz,h->nr,h->nr),rs,h->m,xs);
	if(detail) fprintf(stderr,"\n");
	if(detail) fprintf(stderr,"distorted geometry\n");
	if(detail) formattedvector(h->nc,h->cc,"x0");
	if(detail) formattedvector(h->nc,xs,"xs");
	fprintf(stderr,"\n");

	for(k=1;k<h->na;k++)
		for(j=0;j<k;j++)
			m0[mindex(j,k)]=distance(j,k,xs,bt);	// m0=me+dr+dvs

	fprintf(stderr," normal mode run \n");
	for(i=nz;i<h->nr;i++)
	{
		Ti=t->Tv[i-nz];
		frequency=sqrt(fabs(h->ev[i]))*(0.5*M_1_PI);
		wavenumber=frequency/(SpeedLight);
		hv=PlanckConstant*frequency;
		kT=BoltzmannConstant*Ti;
		l2=h_8pi2c/(wavenumber*tanh(hc_2k*wavenumber/Ti));
		l1=sqrt(l2);

	//	Tq=0;
	//	for(k=0;k<h->nc;k++) Tq+= h->m[k]*dsqr(vx[rindex(k,i,h->nc,h->nr)]);
	//	fprintf(stderr," kinetic energy of unit cartesian vector for %dth mode = %lf\n",i-nz+1,Tq);
		fprintf(stderr," evaluating amplitude and shrinkage by running\n");
		fprintf(stderr,"%5d: %12.4lf /cm l1= %lf sqrt(amu) A l2= %lf amu A^2 at %.0lf K\n"
			,i+1,wavenumber/PerCentimeter,l1/SqrtAMUAngstrom,l2/AMUAngstrom2,Ti/Kelvin);
		Eh=thermalvibrationalenergy(hv,kT);	// thermal energy
		Ez=0.5*hv;
		Tq=Ez+Eh-0;
		p=0.5*M_2_SQRTPI*M_SQRT2/l1;
		q=p;
		if(detail) fprintf(stderr," # zero point energy=  %le eu and kT= %le eu = %lf /cm\n",0.5*hv,kT,kT*EU_CM);
		if(detail) fprintf(stderr," # classical turning point= %le sqrt(amu)A at %le eu\n",sqrt(Eh/h->ev[i])/SqrtAMUAngstrom,(kT+0.5*hv));
		fprintf(stderr," # quantum turning point=   %le sqrt(amu)A at %le eu\n",l1/SqrtAMUAngstrom,Eh);

		localcentrifugal(detail,i,h,dvp);

		// scaled to instantaneous kinetic energy
		fDCLEAR(h->nr,rd,1);
		fDAXPY(h->nr,2*Tq,dvp,1,rd,1);	// borrowing rd for print
		formattedvector(h->nr,rd,"ddrc");

		fDCOPY(h->nc,xs,1,xp,1);
	//	for(k=0;k<h->nr;k++) rd[k]=rs[k]+(2*Tq)*dvp[k];
		fDCOPY(h->nr,rs,1,rd,1);
		fDAXPY(h->nr,(2*(Tq-0.5*(Ez+Eh))),dvp,1,rd,1);	// 0.5*Eh = average kinetic energy
		ccj=excitemolecule(detail,h->nr,h->nc,h->z,0.0,h->vr+rindex(0,i,h->nr,h->nr),rd,h->m,xp);

		fDCOPY(h->nc,xp,1,xn,1);

		if(detail>2)
			fprintf(stderr," # walking probability P= %le at q= %le sqrt(amu) A <T>= %lf \n"
				,p,0.,Tq/(PlanckConstant*SpeedLight)/PerCentimeter);
		for(k=1;k<h->na;k++)
			for(j=0;j<k;j++)
			{
				r=distance(j,k,xp,bt);
				m=mindex(j,k);
				m1[m]=(r-m0[m])*p;
				m2[m]=dsqr(r-m0[m])*p;
				m3[m]=dsqr(r-m0[m])*(r-m0[m])*p;
				m4[m]=dsqr(dsqr(r-m0[m]))*p;
			}
		for(v=1;v<WALK_FARTHER*nvib;v++)
		{
			double lv = v*vib*l1;	// P = 1/(sqrt(2pi)l1) * exp(-0.5 x^2/l1^2)
			p = 0.5*M_2_SQRTPI*M_SQRT2/l1 * exp(-0.5*(v*vib)*(v*vib));
			q+=2*p;
			Tq=Ez+Eh-(0.5*h->ev[i]*lv*lv);
			if(Tq<0) Tq=0;
			if(detail>2)
				fprintf(stderr," # walking probability P= %le at q= %le sqrt(amu) A <T>= %le \n"
					,p,lv/SqrtAMUAngstrom,Tq/(PlanckConstant*SpeedLight)/PerCentimeter);

		//	for(k=0;k<h->nr;k++) rd[k]=rs[k]+(2*(Tq-0.5*Eh))*dvp[k];
			fDCOPY(h->nr,rs,1,rd,1);
			fDAXPY(h->nr,(2*(Tq-0.5*(Ez+Eh))),dvp,1,rd,1);
			ccj=excitemolecule(detail,h->nr,h->nc,h->z, lv,h->vr+rindex(0,i,h->nr,h->nr),rd,h->m,xp);
			ccj=excitemolecule(detail,h->nr,h->nc,h->z,-lv,h->vr+rindex(0,i,h->nr,h->nr),rd,h->m,xn);
			for(k=1;k<h->na;k++)
				for(j=0;j<k;j++)
				{
					m=mindex(j,k);
					r=distance(j,k,xp,bt);
					m1[m]+=(r-m0[m])*p;
					m2[m]+=dsqr(r-m0[m])*p;
					m3[m]+=dsqr(r-m0[m])*(r-m0[m])*p;
					m4[m]+=dsqr(dsqr(r-m0[m]))*p;
					r=distance(j,k,xn,bt);
					m1[m]+=(r-m0[m])*p;
					m2[m]+=dsqr(r-m0[m])*p;
					m3[m]+=dsqr(r-m0[m])*(r-m0[m])*p;
					m4[m]+=dsqr(dsqr(r-m0[m]))*p;
				}
		}
		if(detail) fprintf(stderr," sum of probability = %le\n",q);
		for(j=0;j<nm;j++)
		{
			m1[j]/=q;
			m2[j]/=q;
			m3[j]/=q;
			m4[j]/=q;
			dvr[j]+=m1[j];
			la[j]+=m2[j]-dsqr(m1[j]);
		}

		//formattedvector(nm,ma,"ra");
		//formattedvector(nm,mb,"la");
		fprintf(stderr," ra=");
		for(j=0;j<nm;j++)
			fprintf(stderr,"%8.4lf",(m0[j]+m1[j])/Angstrom);
		fprintf(stderr,"\n");
		fprintf(stderr," dv=");
		for(j=0;j<nm;j++)
			fprintf(stderr,"%8.4lf",(m1[j])/Angstrom);
		fprintf(stderr,"\n");
		fprintf(stderr," la=");
		for(j=0;j<nm;j++)
			fprintf(stderr,"%8.4lf",sqrt(fabs(m2[j]-dsqr(m1[j])))/Angstrom);
		fprintf(stderr,"\n");
						
	}
	fprintf(stdout,"%8d",h->na*(h->na-1)/2);
	for(i=nz;i<h->nr;i++)
		fprintf(stdout,"  %.0lf",t->Tv[i-nz]/Kelvin);
	fprintf(stdout,"\n");
//	Ti=0; for(i=nz;i<h->nr;i++) Ti+=t->Tv[i-nz];
//	fprintf(stdout,"  %.0lf\n",Ti/(h->nr-nz));
	for(k=1;k<h->na;k++)
		for(j=0;j<k;j++)
		{

			l2sum=0;
			fDCLEAR(h->nc,bt,1);
			r=distance(j,k,xs,bt);
			for(i=nz;i<h->nr;i++)
			{
				Ti=t->Tv[i-nz];
				frequency=sqrt(fabs(h->ev[i]))*(0.5*M_1_PI);
				wavenumber=frequency/SpeedLight;
				l2=h_8pi2c/(wavenumber*tanh(hc_2k*wavenumber/Ti));
				l2sum+=l2*dsqr(fDDOT(h->nc,bt,1,h->vx+rindex(0,i,h->nc,h->nr),1));
			}

			m=mindex(j,k);
			fprintf(stdout,"  %-6s %-6s ",h->ai[j].name,h->ai[k].name);
			fprintf(stdout," re= %15.10lf",(me[m])/Angstrom);
			fprintf(stdout," dr= %15.10lf",(dr[m])/Angstrom);
		//	if(detail>2) fprintf(stdout," ds= %15.10lf",(m0[m]-me[m])/Angstrom);// steady
			if(detail>1) fprintf(stdout," d0= %15.10lf",(m0[m]-me[m]-dr[m])/Angstrom);
			if(detail>1) fprintf(stdout," du= %15.10lf",(dvr[m])/Angstrom);	// dvr from m0
			fprintf(stdout," dv= %15.10lf",(dvr[m]+m0[m]-me[m]-dr[m])/Angstrom);
			fprintf(stdout," la= %15.10lf",sqrt(la[m])/Angstrom);
			fprintf(stdout," lh= %15.10lf",sqrt(l2sum)/Angstrom);
			fprintf(stdout," al= %15.10lf",sqrt((la[m]-l2sum)/(1.5*dsqr(l2sum)))*Angstrom);
			fprintf(stdout," ad= %15.10lf",estimatea(dvr[m]/Angstrom,sqrt(la[m])/Angstrom));
			fprintf(stdout,"\n");
		}

	free(dvp);
	free(cr);
	free(cv);
	free(rs);
	free(rd);
	free(xs);
	free(xp);
	free(xn);
	free(bt);
	free(m4);
	free(m3);
	free(m2);
	free(m1);
	free(m0);
	free(me);
	free(la);
	free(dvr);
	free(dr);
	return(0);
}

int moderun1(int detail,THERMAL *t,INTERNAL *h)
{
	int i,j,k,m,v,nx=6,nm=h->na*(h->na-1)/2;
	double *bt=NULL;
	double *rd=NULL,*xp=NULL,*xn=NULL,*xs=NULL;
	double *dr=NULL,*dvr=NULL,*la=NULL,*me=NULL,*m0=NULL,*m1=NULL,*m2=NULL,*m3=NULL,*m4=NULL;
	double Ti,ccj,l1,l2,frequency,wavenumber;
	double r,p,q,hv,kT,Tq=0,Eh=0,*dvp=NULL,*cr=NULL,*cv=NULL,*rs=NULL;
	int nz=h->nr-h->nv,nvib=WALK_FINER/(1+sqrt(h->na));	// fineness of mode walk
	double vib=1.0/nvib,l2sum;
	if(nz<0) nz=0;

	if(NULL==(dvr=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(dr=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(la=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(me=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(m0=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(m1=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(m2=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(m3=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(m4=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(bt=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(xs=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(xp=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(xn=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(rd=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(rs=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(cr=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(cv=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(dvp=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");

	for(k=1;k<h->na;k++)
		for(j=0;j<k;j++)
			me[mindex(j,k)]=distance(j,k,h->cc,bt);

	externalrotational(detail,h);
	fprintf(stderr," evaluating rotational centrifugal distortion\n");
	fDCLEAR(h->nr,cr,1);
	for(i=0;i<3;i++)
	{
		Ti=t->Tr[i];
		kT=BoltzmannConstant*Ti;
		Eh=0.5*kT;	// thermal mean classical rotational energy = 0.5 kT ?
		fDGEMV('N',h->nr,h->nr,Eh,h->fi,h->nr,h->frt+rindex(0,i,h->nr,3),1,1.0,cr,1);
	}
	if(detail) formattedvector(h->nr,h->r,"r0");
	formattedvector(h->nr,cr,"cr");
	fDCOPY(h->nr,h->r,1,rs,1);
	fDAXPY(h->nr,1.0,cr,1,rs,1);
	fDCOPY(h->nc,h->cc,1,xs,1);
//	displacemolecule(detail,h->nr,h->nc,h->z,rs,h->m,xs);
	ccj=excitemolecule(detail,h->nr,h->nc,h->z,0.0,h->vr+rindex(0,nz,h->nr,h->nr),rs,h->m,xs);
	fprintf(stderr,"\n");
	for(k=1;k<h->na;k++)
		for(j=0;j<k;j++)
			dr[mindex(j,k)]=distance(j,k,xs,bt)-me[mindex(j,k)];
	if(detail)
	{
		fprintf(stderr," dr   = ");
		for(k=0;k<nm;k++)
			fprintf(stderr,"%+8.4lf",dr[k]);
		fprintf(stderr,"\n");
	}

	nx=normalmode(detail,h);
	fprintf(stderr," evaluating centrifugal distortion at steady state\n");
	fDCOPY(h->nr,h->r,1,rs,1);
	for(i=nz;i<h->nr;i++)
	{
		Ti=t->Tv[i-nz];
		frequency=sqrt(fabs(h->ev[i]))*(0.5*M_1_PI);
		wavenumber=frequency/(SpeedLight);
		hv=PlanckConstant*frequency;
		kT=BoltzmannConstant*Ti;

		Eh=hv*exp(-hv/kT)/(1-exp(-hv/kT)) + 0.5*hv;	// thermal energy
		Tq=0.5*Eh;
		fprintf(stderr,"%5d: %12.4lf /cm <T> = 0.5 Eh = %le eu, Ez = %le eu\n",i+1,wavenumber/PerCentimeter,Tq,0.5*hv);

		localcentrifugal(0,i,h,dvp);

		// scaled to instantaneous kinetic energy
		fDAXPY(h->nr,2*Tq,dvp,1,cv,1);	// scaled to momentum square
	}
	fprintf(stderr,"\n");
	if(detail) formattedvector(h->nr,h->r,"r0");
	formattedvector(h->nr,cv,"cv");

	fDCOPY(h->nr,h->r,1,rs,1);
	fDAXPY(h->nr,1.0,cr,1,rs,1);
	fDAXPY(h->nr,1.0,cv,1,rs,1);
	fDCOPY(h->nc,h->cc,1,xs,1);
//	displacemolecule(detail,h->nr,h->nc,h->z,rs,h->m,xs);
	ccj=excitemolecule(detail,h->nr,h->nc,h->z,0.0,h->vr+rindex(0,nz,h->nr,h->nr),rs,h->m,xs);
	fprintf(stderr,"\n");
	if(detail) fprintf(stderr,"distorted geometry\n");
	if(detail) formattedvector(h->nc,h->cc,"x0");
	if(detail) formattedvector(h->nc,xs,"xs");
	fprintf(stderr,"\n");

	fprintf(stderr," normal mode run \n");
	for(k=1;k<h->na;k++)
		for(j=0;j<k;j++)
			m0[mindex(j,k)]=distance(j,k,xs,bt);

	for(i=nz;i<h->nr;i++)
	{
		Ti=t->Tv[i-nz];
		frequency=sqrt(fabs(h->ev[i]))*(0.5*M_1_PI);
		wavenumber=frequency/(SpeedLight);
		hv=PlanckConstant*frequency;
		kT=BoltzmannConstant*Ti;
		l2=h_8pi2c/(wavenumber*tanh(hc_2k*wavenumber/Ti));
		l1=sqrt(l2);

	//	Tq=0;
	//	for(k=0;k<h->nc;k++) Tq+= h->m[k]*dsqr(vx[rindex(k,i,h->nc,h->nr)]);
	//	fprintf(stderr," kinetic energy of unit cartesian vector for %dth mode = %lf\n",i-nz+1,Tq);
		fprintf(stderr," evaluating amplitude and shrinkage by running\n");
		fprintf(stderr,"%5d: %12.4lf /cm l1= %lf sqrt(amu) A l2= %lf amu A^2 at %.0lf K\n"
			,i+1,wavenumber/PerCentimeter,l1/SqrtAMUAngstrom,l2/AMUAngstrom2,Ti/Kelvin);
		Eh=hv*exp(-hv/kT)/(1-exp(-hv/kT)) + 0.5*hv;	// thermal energy
		Tq=Eh;
		p=0.5*M_2_SQRTPI*M_SQRT2/l1;
		q=p;
		if(detail) fprintf(stderr," # zero point energy=  %le eu and kT= %le eu = %lf /cm\n",0.5*hv,kT,kT*EU_CM);
		if(detail) fprintf(stderr," # classical turning point= %le sqrt(amu)A at %le eu\n",sqrt(Eh/h->ev[i])/SqrtAMUAngstrom,(kT+0.5*hv));
		fprintf(stderr," # quantum turning point=   %le sqrt(amu)A at %le eu\n",l1/SqrtAMUAngstrom,Eh);

		localcentrifugal(detail,i,h,dvp);

		// scaled to instantaneous kinetic energy
		fDCLEAR(h->nr,rd,1);
		fDAXPY(h->nr,2*Tq,dvp,1,rd,1);	// borrowing rd for print
		formattedvector(h->nr,rd,"ddrc");

		fDCOPY(h->nc,xs,1,xp,1);
	//	for(k=0;k<h->nr;k++) rd[k]=rs[k]+(2*Tq)*dvp[k];
		fDCOPY(h->nr,rs,1,rd,1);
		fDAXPY(h->nr,(2*(Tq-0.5*Eh)),dvp,1,rd,1);	// 0.5*Eh = average kinetic energy
		ccj=excitemolecule(detail,h->nr,h->nc,h->z,0.0,h->vr+rindex(0,i,h->nr,h->nr),rd,h->m,xp);

		fDCOPY(h->nc,xp,1,xn,1);

		if(detail>2)
			fprintf(stderr," # walking probability P= %le at q= %le sqrt(amu) A <T>= %lf \n"
				,p,0.,Tq/(PlanckConstant*SpeedLight)/PerCentimeter);
		for(k=1;k<h->na;k++)
			for(j=0;j<k;j++)
			{
				r=distance(j,k,xp,bt);
				m=mindex(j,k);
				m1[m]=(r-m0[m])*p;
				m2[m]=dsqr(r-m0[m])*p;
				m3[m]=dsqr(r-m0[m])*(r-m0[m])*p;
				m4[m]=dsqr(dsqr(r-m0[m]))*p;
			}
		for(v=1;v<WALK_FARTHER*nvib;v++)
		{
			double lv = v*vib*l1;	// P = 1/(sqrt(2pi)l1) * exp(-0.5 x^2/l1^2)
			p = 0.5*M_2_SQRTPI*M_SQRT2/l1 * exp(-0.5*(v*vib)*(v*vib));
			q+=2*p;
			Tq=Eh-0.5*h->ev[i]*lv*lv;
			if(Tq<0) Tq=0;
			if(detail>2)
				fprintf(stderr," # walking probability P= %le at q= %le sqrt(amu) A <T>= %le \n"
					,p,lv/SqrtAMUAngstrom,Tq/(PlanckConstant*SpeedLight)/PerCentimeter);

			for(k=0;k<h->nr;k++) rd[k]=rs[k]+(2*(Tq-0.5*Eh))*dvp[k];
			ccj=excitemolecule(detail,h->nr,h->nc,h->z, lv,h->vr+rindex(0,i,h->nr,h->nr),rd,h->m,xp);
			ccj=excitemolecule(detail,h->nr,h->nc,h->z,-lv,h->vr+rindex(0,i,h->nr,h->nr),rd,h->m,xn);
			for(k=1;k<h->na;k++)
				for(j=0;j<k;j++)
				{
					m=mindex(j,k);
					r=distance(j,k,xp,bt);
					m1[m]+=(r-m0[m])*p;
					m2[m]+=dsqr(r-m0[m])*p;
					m3[m]+=dsqr(r-m0[m])*(r-m0[m])*p;
					m4[m]+=dsqr(dsqr(r-m0[m]))*p;
					r=distance(j,k,xn,bt);
					m1[m]+=(r-m0[m])*p;
					m2[m]+=dsqr(r-m0[m])*p;
					m3[m]+=dsqr(r-m0[m])*(r-m0[m])*p;
					m4[m]+=dsqr(dsqr(r-m0[m]))*p;
				}
		}
		if(detail) fprintf(stderr," sum of probability = %le\n",q);
		for(j=0;j<nm;j++)
		{
			m1[j]/=q;
			m2[j]/=q;
			m3[j]/=q;
			m4[j]/=q;
			dvr[j]+=m1[j];
			la[j]+=m2[j]-dsqr(m1[j]);
		}

		//formattedvector(nm,ma,"ra");
		//formattedvector(nm,mb,"la");
		fprintf(stderr," ra=");
		for(j=0;j<nm;j++)
			fprintf(stderr,"%8.4lf",(m0[j]+m1[j])/Angstrom);
		fprintf(stderr,"\n");
		fprintf(stderr," dv=");
		for(j=0;j<nm;j++)
			fprintf(stderr,"%8.4lf",(m1[j])/Angstrom);
		fprintf(stderr,"\n");
		fprintf(stderr," la=");
		for(j=0;j<nm;j++)
			fprintf(stderr,"%8.4lf",sqrt(fabs(m2[j]-dsqr(m1[j])))/Angstrom);
		fprintf(stderr,"\n");
						
	}
	fprintf(stdout,"%8d",h->na*(h->na-1)/2);
	for(i=nz;i<h->nr;i++)
		fprintf(stdout,"  %.0lf",t->Tv[i-nz]/Kelvin);
	fprintf(stdout,"\n");
//	Ti=0; for(i=nz;i<h->nr;i++) Ti+=t->Tv[i-nz];
//	fprintf(stdout,"  %.0lf\n",Ti/(h->nr-nz));
	for(k=1;k<h->na;k++)
		for(j=0;j<k;j++)
		{

			l2sum=0;
			fDCLEAR(h->nc,bt,1);
			r=distance(j,k,xs,bt);
			for(i=nz;i<h->nr;i++)
			{
				Ti=t->Tv[i-nz];
				frequency=sqrt(fabs(h->ev[i]))*(0.5*M_1_PI);
				wavenumber=frequency/SpeedLight;
				l2=h_8pi2c/(wavenumber*tanh(hc_2k*wavenumber/Ti));
				l2sum+=l2*dsqr(fDDOT(h->nc,bt,1,h->vx+rindex(0,i,h->nc,h->nr),1));
			}

			m=mindex(j,k);
			fprintf(stdout,"  %-6s %-6s ",h->ai[j].name,h->ai[k].name);
			fprintf(stdout," re= %15.10lf",me[m]/Angstrom);
			fprintf(stdout," dr= %15.10lf",dr[m]/Angstrom);
			fprintf(stdout," dv= %15.10lf",(dvr[m]+m0[m]-me[m]-dr[m])/Angstrom);
			fprintf(stdout," la= %15.10lf",sqrt(la[m])/Angstrom);
			fprintf(stdout," lh= %15.10lf",sqrt(l2sum)/Angstrom);
			fprintf(stdout," al= %15.10lf",sqrt((la[m]-l2sum)/(1.5*dsqr(l2sum)))*Angstrom);
			fprintf(stdout," ad= %15.10lf",estimatea(dvr[m]/Angstrom,sqrt(la[m])/Angstrom));
			fprintf(stdout,"\n");
		}

	free(dvp);
	free(cr);
	free(cv);
	free(rs);
	free(rd);
	free(xs);
	free(xp);
	free(xn);
	free(bt);
	free(m4);
	free(m3);
	free(m2);
	free(m1);
	free(m0);
	free(me);
	free(la);
	free(dvr);
	free(dr);
	return(0);
}


// This function evaluates the vibrational amplitudes by assuming a gaussian probabiliy distribution along normal mode coordinates to 
// generate molecular structures in the structural ensemble.  The vibrational amplitudes are computed from the width of that probability 
// distribution in terms of interal coordinates
// This function uses the same convergence routine as the function "modejump()".

int modewalk(int detail,THERMAL *t,INTERNAL *h)
{
	int i,j,k,m,v,nx=6,nm=h->na*(h->na-1)/2;
	double *bt=NULL;
	double *rd=NULL,*xp=NULL,*xn=NULL;
	double *dr=NULL,*la=NULL,*m0=NULL,*ma=NULL,*mb=NULL;
	double Ti,ccj,l1,l2,frequency,wavenumber;
	double r,p,q;
	int nz=h->nr-h->nv,nvib=WALK_FINER/(1+sqrt(h->na));	// fineness of mode walk
	double vib=1.0/nvib;

	if(NULL==(dr=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(la=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(m0=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");		// cartesian distance matrix
	if(NULL==(ma=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(mb=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(bt=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");	// transpose of the B matrix
	if(NULL==(xp=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");	// positively displaced cartesian coordinates
	if(NULL==(xn=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");	// negatively displaced cartesian coordinates
	if(NULL==(rd=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");

	nx=normalmode(detail,h);				// returns the number of internal coordinates  
								// this function calculates the eigenvalues and 
								// the eigenvectors of the force constant matrix (Hessian) 
								// in mass-weighted internal coordinates, 
								// internal coordinates, and cartesian coordinates

	for(k=1;k<h->na;k++)					// fill the cartesian distance matrix
		for(j=0;j<k;j++)
			m0[mindex(j,k)]=distance(j,k,h->cc,bt);

	fprintf(stderr," normal mode walk \n");
	if(nz<0) nz=0;						// number of excess internal degrees of freedom
	for(i=nz;i<h->nr;i++)					
	{
		Ti=t->Tv[i-nz];					// intitially assigned temperature of vibrational mode
		frequency=sqrt(fabs(h->ev[i]))*(0.5*M_1_PI);	// harmonic vibrational frequency in hz
		wavenumber=frequency/(SpeedLight);		
		l2=h_8pi2c/(wavenumber*tanh(hc_2k*wavenumber/Ti));	// calculate the squared mean amplitude of vibration (harmonic approximation)
		//if(i-nz+1==h->freeze)
		if(i-nz<h->freeze)
		{
			l2=h_8pi2c/(wavenumber);
			fprintf(stderr,"freezing %dth mode\n",i-nz+1);
		}
		l1=sqrt(l2);

		fprintf(stderr,"%5d: %12.4lf /cm l1= %lf sqrt(amu) A l2= %lf amu A^2 at %lf K\n"
			,i+1,wavenumber/PerCentimeter,l1/SqrtAMUAngstrom,l2/AMUAngstrom2,Ti/Kelvin);	

		if(detail)					// not executed by default
		{
			int more = detail;
			fprintf(stderr," convergence test by jumping\n");
			formattedvector(h->nr,h->vq+i*h->nr,"vq");
			formattedvector(h->nr,h->vr+i*h->nr,"vr");
			formattedvector(h->nc,h->vx+i*h->nc,"vx");
			fprintf(stderr,"\n");
			for(v=-1;v<=1;v++)
			{
				if(more>1) formatteddisplacedvector(h->nc,v*l1,h->vx+i*h->nc,h->cc,"cc ");
				if(more>1) formatteddisplacedvector(h->nr,v*l1,h->vr+i*h->nr,h->r ,"rc ");
				fDCOPY(h->nc,h->cc,1,xp,1);
			//	for(j=0;j<h->nr;j++)
			//		rd[j]=h->r[j]+v*l1*vz[rindex(j,i,h->nr,h->nr)];
			//	//formattedvector(h->nr,rd,"rc ");
			//	ccj=displacemolecule(detail,h->nr,h->nc,h->z,rd,h->m,xp);
				ccj=excitemolecule(detail,h->nr,h->nc,h->z,v*l1,h->vr+rindex(0,i,h->nr,h->nr),h->r,h->m,xp);
				fprintf(stderr,"     l= %+lf sqrt(amu) A  convergence= %le\n"
					,v*l1/(SqrtAMUAngstrom),sqrt(ccj));
				formattedvector(h->nc,xp,"rx ");
				if(more>0)
				{
					fprintf(stderr," mls=");
					for(k=1;k<h->na;k++)
						for(j=0;j<k;j++)
						{
							r=distance(j,k,xp,bt);
							fprintf(stderr," %10.5lf",r/Angstrom);
						}
					fprintf(stderr,"\n");
				}
				fprintf(stderr,"\n");
			}
		}


		fprintf(stderr,"\n");
		fprintf(stderr," evaluating amplitude and shrinkage by walking\n");
//		q=0;
//		for(j=0;j<nm;j++) ma[j]=mb[j]=0;
//		fDCOPY(h->nc,h->cc,1,xd,1);
//		for(v=-4*nvib;v<=4*nvib;v++)
//		{
//			q+=(p=0.5*M_2_SQRTPI*M_SQRT2*exp(-0.5*(v*vib)*(v*vib))/l1*vib);
//			// if(1) fprintf(stderr," v= %d p= %lf\n",v,p);
//			for(j=0;j<h->nr;j++)
//				rd[j]=h->r[j]+v*vib*l1*vz[rindex(j,i,h->nr,h->nr)];
//			ccj=displacemolecule(detail,h->nr,h->nc,h->z,rd,h->m,xd);
//		//	for(j=0;j<h->nc;j++) x0[j]=xd[j];
//			for(k=1;k<h->na;k++)
//				for(j=0;j<k;j++)
//				{
//					r=distance(j,k,xd,bt);
//					m=mindex(j,k);
//					ma[m]+=r*p;
//					mb[m]+=r*r*p;
//				}
//		}
		fDCOPY(h->nc,h->cc,1,xp,1);			// copy the cartesian coordinates into the xp array
		fDCOPY(h->nc,h->cc,1,xn,1);			// copy the cartesian coordinates into the xn array
		q=(p=0.5*M_2_SQRTPI*M_SQRT2/l1);		// p is the walking probability ?
		for(k=1;k<h->na;k++)
			for(j=0;j<k;j++)
			{
				r=distance(j,k,xp,bt);		// evaluate all unique cartesian distances at the equilibrium configuration
				m=mindex(j,k);
				ma[m]=r*p;			// ?
				mb[m]=r*r*p;			// ?
			}
		for(v=1;v<WALK_FARTHER*nvib;v++)
		{						// P = 1/(sqrt(2pi)l1) * exp(-0.5 x^2/l1^2)
			double lv = v*vib*l1;			// stepwise increased vibrational deviation, lv goes up to maximally 4*l1
			q+=2*(p = 0.5*M_2_SQRTPI*M_SQRT2/l1 * exp(-0.5*(v*vib)*(v*vib)) );
			if(detail) fprintf(stderr," # walking probability p= %le\n",p);
			ccj=excitemolecule(detail,h->nr,h->nc,h->z, lv,h->vr+rindex(0,i,h->nr,h->nr),h->r,h->m,xp); // displace molecule incrementally (+)
			ccj=excitemolecule(detail,h->nr,h->nc,h->z,-lv,h->vr+rindex(0,i,h->nr,h->nr),h->r,h->m,xn); // displace molecule incrementally (-)
		//	for(j=0;j<h->nr;j++)
		//		rd[j]=h->r[j] + v*vib*l1*vz[rindex(j,i,h->nr,h->nr)];
		//	ccj=displacemolecule(detail,h->nr,h->nc,h->z,rd,h->m,xp);
		//	for(j=0;j<h->nr;j++)
		//		rd[j]=h->r[j] - v*vib*l1*vz[rindex(j,i,h->nr,h->nr)];
		//	ccj=displacemolecule(detail,h->nr,h->nc,h->z,rd,h->m,xn);
			for(k=1;k<h->na;k++)			// this loop calculates the first and second moments of the 
				for(j=0;j<k;j++)		// (Gaussian) probability distribution function.
				{
					m=mindex(j,k);
					r=distance(j,k,xp,bt);
					ma[m]+=r*p;		// weigh the positive distance by the probability
					mb[m]+=r*r*p;		// weigh the positive squared distance by the probability
					r=distance(j,k,xn,bt);
					ma[m]+=r*p;		// weigh the negative distance by the probability
					mb[m]+=r*r*p;		// weigh the negative squared distance by the probability
				}
		}
		for(j=0;j<nm;j++)				// for all unique interatomic distances
		{
			ma[j]/=q;				// normalize by the total probability
			mb[j]/=q;		
			dr[j]+=ma[j]-m0[j];			// distortion of interatomic distance j
			la[j]+=mb[j]-ma[j]*ma[j];		// calculate mean amplitude of vibration
		}						// see Chuck's Thesis Eqn 2.63

		//formattedvector(nm,ma,"ra");
		//formattedvector(nm,mb,"la");
		if(detail)
		{
			fprintf(stderr," ra=");
			for(j=0;j<nm;j++)
				fprintf(stderr,"%8.4lf",ma[j]/Angstrom);
			fprintf(stderr,"\n");
		}
		fprintf(stderr," dr=");
		for(j=0;j<nm;j++)
			fprintf(stderr,"%8.4lf",(ma[j]-m0[j])/Angstrom);
		fprintf(stderr,"\n");
		fprintf(stderr," la=");
		for(j=0;j<nm;j++)
			fprintf(stderr,"%8.4lf",sqrt(fabs(mb[j]-ma[j]*ma[j]))/Angstrom);  
		fprintf(stderr,"\n");
						
	}
	fprintf(stdout,"%8d",h->na*(h->na-1)/2);
	for(i=nz;i<h->nr;i++)
		fprintf(stdout,"  %.0lf",t->Tv[i-nz]);
	fprintf(stdout,"\n");
	for(k=1;k<h->na;k++)
		for(j=0;j<k;j++)
		{
			m=mindex(j,k);
			la[m]=sqrt(fabs(la[m]));		// // see Chuck's Thesis Eqn 2.63
			fprintf(stdout,"  %-6s %-6s ",h->ai[j].name,h->ai[k].name);
			fprintf(stdout," re= %15.10lf",m0[m]/Angstrom);
			fprintf(stdout," dv= %15.10lf",dr[m]/Angstrom);
			fprintf(stdout," la= %15.10lf",la[m]/Angstrom);
			fprintf(stdout," ad= %15.10lf",estimatea(dr[m]/Angstrom,la[m]/Angstrom));
			fprintf(stdout,"\n");
		}
	free(rd);
	free(xp);
	free(xn);
	free(bt);
	free(ma);
	free(mb);
	free(m0);
	free(la);
	free(dr);
	return(0);
}


// This function calculates the mean squared amplitude of vibration and then displaces the molecular structural coordinates
// (internal and cartesian) by that amount.  Once the directions are identified, the molecule is further displaced 
// in the same cartesian direction until a convergence criterion is satisfied.
// From the extent of the displaced molecular structure, which is a second order approximation, extraplolated from the Hessian data,
// the mean amplitude of vibration is calculated.
// NEED REFERENCE DOCUMENTATION FOR THIS FUNCTION

int modejump(int detail,THERMAL *t,INTERNAL *h)
{
	int i,j,k,m,nx=6,nm=h->na*(h->na-1)/2;
	double *bt=NULL;
	double *rd=NULL,*xp=NULL,*xn=NULL;
	double *dr=NULL,*la=NULL,*m0=NULL;
	double Ti,ccj,l1,l2,frequency,wavenumber;
	int nz=h->nr-h->nv;					// number of excess internal coordinates

	if(NULL==(dr=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");		// displacement from equilibrium position due to vibrations
	if(NULL==(la=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");		// mean vibrational amplitudes
	if(NULL==(m0=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");		// cartesian distance matrix at equilibrium
	if(NULL==(bt=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");	// transpose of the B matrix
	if(NULL==(xp=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");	// positively displaced cartesian coordinates
	if(NULL==(xn=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");	// negatively displaced cartesian coordinates
	if(NULL==(rd=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");	// array not used

	nx=normalmode(detail,h);				// returns the number of internal coordinates

	for(k=1;k<h->na;k++)					// fill the cartesian distance matrix
		for(j=0;j<k;j++)
			m0[mindex(j,k)]=distance(j,k,h->cc,bt);

	fprintf(stderr," normal mode jump \n");
	if(nz<0) nz=0;
	if(detail) formattedvector(h->nc,h->cc,"cc ");
	for(i=nz;i<h->nr;i++)
	{
		Ti=t->Tv[i-nz];					// intitially assigned temperature of vibrational mode
		frequency=sqrt(fabs(h->ev[i]))*(0.5*M_1_PI);	// harmonic vibrational frequency in hz
		wavenumber=frequency/(SpeedLight);
		l2=h_8pi2c/(wavenumber*tanh(hc_2k*wavenumber/Ti));	// calculate the squared mean amplitude of vibration (harmonic approximation)
		l1=sqrt(l2);

		fprintf(stderr,"%5d: %12.4lf /cm l1= %lf sqrt(amu) A l2= %lf amu A^2 at %lf K\n"
			,i+1,wavenumber/PerCentimeter,l1/SqrtAMUAngstrom,l2/AMUAngstrom2,Ti/Kelvin);

		fDCOPY(h->nc,h->cc,1,xp,1);			// copy the cartesian coordinates into the xp array
	//	for(j=0;j<h->nr;j++)
	//		rd[j]=h->r[j]+l1*h->vr[rindex(j,i,h->nr,h->nr)];
	//	ccj=displacemolecule(detail,h->nr,h->nc,h->z,rd,h->m,xp);
		ccj=excitemolecule(detail,h->nr,h->nc,h->z, l1,h->vr+rindex(0,i,h->nr,h->nr),h->r,h->m,xp);
		fprintf(stderr,"     l= %+lf sqrt(amu) A  convergence= %le\n",l1/SqrtAMUAngstrom,sqrt(ccj));
		fDCOPY(h->nc,h->cc,1,xn,1);			// copy the cartesian coordinates into the xn array
	//	for(j=0;j<h->nr;j++)
	//		rd[j]=h->r[j]-l1*h->vr[rindex(j,i,h->nr,h->nr)];
	//	ccj=displacemolecule(detail,h->nr,h->nc,h->z,rd,h->m,xn);
		ccj=excitemolecule(detail,h->nr,h->nc,h->z,-l1,h->vr+rindex(0,i,h->nr,h->nr),h->r,h->m,xn);
		fprintf(stderr,"     l= %+lf sqrt(amu) A  convergence= %le\n",-l1/SqrtAMUAngstrom,sqrt(ccj));
		if(detail) formattedvector(h->nc,xp,"xp");	// print out the positively displaced cartesian coordinates
		if(detail) formattedvector(h->nc,xn,"xn");	// print out the negatively displaced cartesian coordinates

		fprintf(stderr," drijk= ");
		for(k=1;k<h->na;k++)
			for(j=0;j<k;j++)
			{
				m=mindex(j,k);
				double rp=distance(j,k,xp,bt);
				double rn=distance(j,k,xn,bt);
				double re=m0[m];		// cartesian equilibrium distance	
				double dm=0.5*(rp+rn)-re;	// midpoint displacement due to vibration between atom j and atom k
				dr[m]+=dm;			// sum up all midpoint displacements due to all normal modes
				la[m]+=0.5*((rp-re)*(rp-re)+(rn-re)*(rn-re));  // sum up "average variance" due to all normal modes, need reference
				fprintf(stderr,"%8.4lf",dm/Angstrom);
			}
		fprintf(stderr,"\n");
		fprintf(stderr,"\n");
						
	}
	fprintf(stdout,"%8d",h->na*(h->na-1)/2);		// print out the number of unique distances
	for(i=nz;i<h->nr;i++)
		fprintf(stdout,"  %.0lf",t->Tv[i-nz]/Kelvin);	// temperature of each vibrational mode / coordinate
	fprintf(stdout,"\n");
	for(k=1;k<h->na;k++)
		for(j=0;j<k;j++)
		{
			m=mindex(j,k);
			la[m]=sqrt(fabs((la[m]+dr[m]*dr[m])));			// need reference
			fprintf(stdout,"  %-6s %-6s ",h->ai[j].name,h->ai[k].name); 	// atom pair
			fprintf(stdout," re= %15.10lf",m0[m]/Angstrom);		// cartesian equilibrium distance
			fprintf(stdout," dv= %15.10lf",dr[m]/Angstrom);		// displacement of equilibrium distance due to vibrations
			fprintf(stdout," la= %15.10lf",la[m]/Angstrom);		// mean amplitude of vibration
			fprintf(stdout," ad= %15.10lf",estimatea(dr[m]/Angstrom,la[m]/Angstrom));	
			fprintf(stdout,"\n");
		}
	free(rd);
	free(xn);
	free(xp);
	free(bt);
	free(m0);
	free(la);
	free(dr);
	return(0);
}

