#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>

#include "programming.h"
#include "constantunits.h"

#include "errors.h"
#include "initialize.h"
#include "intrinsic.h"
#include "readinternal.h"


// The following function finds the specified string in a list of strings and returns the line number of the match.

int findzreference(char *reference, NUCLEAR *list,int n)
{
	int i;
	for(i=0;i<n;i++)
		if(!strcasecmp(reference,list[i].name)) 
			return(i);
//	ued3error(reference);
	return(-1);
}


// The following function reads the *.zmt file containing the redundant internal coordinates specified by the user and saves
// the values in the structure INTERNAL specified in programming.h
// The format for this *.zmt file is given in sample.zmt

int readredundant(FILE *fp,INTERNAL *h)
{
	int i,j,nr;
	char buffer[STRINGBUFFERSIZE];
	char t[6][STRINGBUFFERSIZE];

	if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) ued3error("reading internal coordinates");
	sscanf(buffer,"%s",h->molecule);
	if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) ued3error("reading internal coordinates");
	sscanf(buffer,"%d",&(h->na));
	for(i=0;i<MAXIMUM_NUMBER_ATOMS;i++)
	{
		if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) ued3error("reading references");
		if(1>sscanf(buffer,"%s",t[0])) break;
	} h->na=i; // blank line was read in loop
	for(i=0;i<MAXIMUM_NUMBER_INTERNAL_COORDINATE;i++)
	{
		if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) break;
		if(3>(nr=sscanf(buffer,"%s%s%s%s%s",t[0],t[1],t[2],t[3],t[4]))) break;
	} h->nr=i;
	rewind(fp);
	
	memoryinternalallocate(h);			// memory allocation for the structure INTERNAL

	if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) ued3error("reading internal coordinates");
	if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) ued3error("reading internal coordinates");
	for(i=0;i<h->na;i++)
	{
		if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) ued3error("reading references");
		sscanf(buffer,"%s",h->ai[i].name);
	}
	if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) ued3error("reading internal coordinates");
	
	for(i=0;i<h->nr;i++)				// read each line and assign it a redunant coordinate type
	{
		for(j=0;j<6;j++) sprintf(t[j],"");
		if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) ued3error("reading references");
		nr=sscanf(buffer,"%s%s%s%s%s%s",t[0],t[1],t[2],t[3],t[4],t[5]);
		if(isdigit(t[0][0])||'-'==t[0][0]) h->z[i].type = atoi(t[0]);
		else if(!strcasecmp(t[0],"r")) h->z[i].type = BOND_STRETCHING ;
		else if(!strcasecmp(t[0],"z")) h->z[i].type = BOND_RECIPROCAL;
		else if(!strcasecmp(t[0],"z2")) h->z[i].type = BOND_RECIPROCAL2;
		else if(!strcasecmp(t[0],"z6")) h->z[i].type = BOND_RECIPROCAL6;
		else if(!strcasecmp(t[0],"a")) h->z[i].type = BENDING_ANGLE;
		else if(!strcasecmp(t[0],"b")) h->z[i].type = BENDING_ANGLE;
		else if(!strcasecmp(t[0],"a2")) h->z[i].type = BENDING2_ANGLE;
		else if(!strcasecmp(t[0],"b2")) h->z[i].type = BENDING2_ANGLE;
		else if(!strcasecmp(t[0],"t")) h->z[i].type = DIHEDRAL_ANGLE;
		else if(!strcasecmp(t[0],"d")) h->z[i].type = DIHEDRAL_ANGLE;
		else if(!strcasecmp(t[0],"t2")) h->z[i].type = DIHEDRAL_360;
		else if(!strcasecmp(t[0],"d2")) h->z[i].type = DIHEDRAL_360;
		else h->z[i].type=atoi(t[0]);
		h->z[i].indices=nr-1;
		switch(h->z[i].type)
		{
			case BOND_STRETCHING: 			// the types are defined in intrinsic.h
				h->z[i].indices=2;
				h->z[i].f=1/Angstrom;
				break;
			case BOND_RECIPROCAL: 
				h->z[i].indices=2;
				h->z[i].f=1/(1/Angstrom);
				break;
			case BOND_RECIPROCAL2:
				h->z[i].indices=2;
				h->z[i].f=1/(1/Angstrom*1/Angstrom);
				break;
			case BOND_RECIPROCAL6:
				h->z[i].indices=2;
				h->z[i].f=pow(1/Angstrom,-6.0);
				break;
			case BENDING_ANGLE:
				h->z[i].indices=3;
				h->z[i].f=1/(M_PI/180);
				break;
			case BENDING2_ANGLE:
				h->z[i].indices=4;
				h->z[i].f=1/(M_PI/180);
				break;
			case DIHEDRAL_ANGLE:
			case DIHEDRAL_360:
			case TANGENT_DIHEDRAL2:
			case TANGENT_PI_DIHEDRAL2:
			case PLANAR_BENDING:
			case LINEAR_BENDING:
				h->z[i].indices=4;
				h->z[i].f=1/(M_PI/180);
				break;
			default:
				h->z[i].indices=0;
				h->z[i].f=1;
		}
	//	fprintf(stderr," type= %d indices= %d\n",h->z[i].type,h->z[i].indices);
		for(j=0;j<h->z[i].indices;j++)						// link each atom name in the redundant coordinates to
		{									// the line of a matching atom name in the first entry of the zmatrix
			if(isalpha(t[j+1][0]))
				h->z[i].id[j] = findzreference(t[j+1],h->ai,h->na);	
			else
				h->z[i].id[j] = atoi(t[j+1])-1;				 
			if(h->z[i].id[j]<0 || h->z[i].id[j]>=h->na)
			{
				fprintf(stderr,"& internal coordinate = %d\n",i+1);
				fprintf(stderr,"& atomic index = %s at %d\n",t[j+1],j+1);
				ued3error("wrong atomic index");
			}
		}
		h->z[i].a = atof(t[1+h->z[i].indices])/Angstrom;			// assign anharmonicity constant using the input value; fix it for angles
	}
	if(DETAIL)
		fprintf(stderr," read %d coordinates for %d atoms of %s\n"
			,h->nr,h->na,h->molecule);
	return(h->nr);
}


// The following function prints out the redundant internal coordinates saved in the structure INTERNAL

int writeredundant(FILE *fp,INTERNAL *h)
{
	int i,j;
	fprintf(fp,"molecule= %s\n",h->molecule);
	fprintf(fp,"number_of_atoms= %d\n",h->na);
	for(i=0;i<h->na;i++)
		fprintf(fp," %4d   %s\n",i+1,h->ai[i].name);
	for(i=0;i<h->nr;i++)
	{
		fprintf(fp," %4d %4d    ",i+1,h->z[i].type);
		for(j=0;j<h->z[i].indices;j++)
			fprintf(fp," %-4s",h->ai[h->z[i].id[j]].name);
		for(j=h->z[i].indices;j<5;j++)
			fprintf(fp," %4s","");
		fprintf(fp," a= %+lf",h->z[i].a);
		for(j=0;j<h->z[i].indices;j++)
			fprintf(fp," %4d",h->z[i].id[j]+1);			
		fprintf(fp,"\n");
	}
	fprintf(stderr,"\n");
	return(h->nr);
}

