#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "atomic.h"
#include "constantunits.h"
#include "intrinsic.h"
#include "programming.h"

#include "errors.h"
#include "outputs.h"
#include "interface.h"
#include "coordinates.h"
#include "calculate.h"
#include "cartesian.h"
#include "internal.h"
#include "vibrational.h"
#include "shrinkage.h"

int shrink(int detail,THERMAL *t,INTERNAL *h)
{
	int i,j,k,nx=0,nz=0,nm=(h->na)*(h->na-1)/2;
	double *bt=NULL,*bi=NULL;
	double *dr=NULL,*dc=NULL,*dm=NULL,*d2=NULL;
	double *drk=NULL,*dxk=NULL,*xk0=NULL,*xkp=NULL,*xkn=NULL,*khip=NULL,*khin=NULL,*khi0=NULL;
	double *rs=NULL,*xs=NULL,*dTdq=NULL;
	double a,Ti,dq,d1,l1,l2,frequency,wavenumber,hv,kT,Tq,Eh,Ez;

	if(NULL==(dr=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(dc=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(dm=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(d2=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(xk0=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(xkp=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(xkn=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(dxk=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(khi0=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(khip=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(khin=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(drk=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(bt=(double *)calloc(h->nc*h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(bi=(double *)calloc(h->nc*h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(rs=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(xs=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(dTdq=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");

	externalrotational(detail,h);
	nz = h->nr-h->nv;
	if(nz<0) nz=0;
	fprintf(stderr," evaluating rotational centrifugal distortion\n");
	fDCLEAR(h->nr,drk,1);
	rotationalcentrifugalforce(detail,h,3,t->Tr,dTdq);	// dTdx
	fDGEMV('N',h->nr,h->nr,1.0,h->fi,h->nr,dTdq,1,0.0,drk,1);
	fDGEMV('N',h->nc,h->nr,1.0,h->bi,h->nc,drk,1,0.0,dxk,1);
	if(detail) formattedvector(h->nc,drk,"cr ");
	for(k=1;k<h->na;k++)
		for(j=0;j<k;j++)
		{
			fDCLEAR(h->nc,bt,1);
			double re=distance(j,k,h->cc,bt);
			dr[mindex(j,k)]=fDDOT(h->nc,dxk,1,bt,1);
		}
	

	nx = normalmode(detail,h);
	a=thermalzeropointfraction(t->Tv,h);
	fprintf(stderr," second order shrinkage effect with local centrifugal\n");

	fDCLEAR(h->nr,drk,1);
	vibrationalcentrifugal(detail,h);
	vibrationalcentrifugalforce(detail,h,t->Tv,dTdq);	// dTdx
	formattedvector(h->nr,dTdq,"dTdx");
	fDGEMV('N',h->nr,h->nr,1.0,h->gs,h->nr,dTdq,1,0.0,drk,1);	// dTdq
	fDCOPY(h->nr,drk,1,dTdq,1);
	formattedvector(h->nr,dTdq,"dTdq");
	
	fDCOPY(h->nc,h->cc,1,xs,1);
	fDCOPY(h->nr,h->r,1,rs,1);
	instantaneous(h->nc,h->nr,h->z,xs,h->m,rs,bi);
	for(i=0;i<nz;i++)
	{
		frequency=sqrt(fabs(h->ev[i]))*(0.5*M_1_PI);
		wavenumber=frequency/SpeedLight;		// hv = hc/l
		fprintf(stderr,"%5d: %12.4lf /cm \n"
			,i+1,wavenumber/PerCentimeter);
	}
	for(i=nz;i<h->nr;i++)
	{
		Ti=t->Tv[i-nz];
		// mass-weighted hessian = omega^2 = (2 pi nu)^2 = 4 pi^2 c^2 nubar^2
		// h nu = hc/l = hc nubar
		frequency=sqrt(fabs(h->ev[i]))*(0.5*M_1_PI);
		wavenumber=frequency/SpeedLight;
		hv=PlanckConstant*frequency;
		kT=BoltzmannConstant*Ti;
		l2=h_8pi2c/(wavenumber*tanh(hc_2k*wavenumber/Ti));
		l1=sqrt(l2);
		dq=fDDOT(h->nr,dTdq,1,h->vq+rindex(0,i,h->nr,h->nr),1)/h->ev[i];
		if(REMOVE_SELF_FORCE)	// 1 remove self acceleration
		{
			Eh=thermalvibrationalenergy(hv,kT);
			Ez=0.5*hv;
			Tq=a*Ez+Eh;
			fDGEMV('N',h->nr,h->nr,Tq,h->gs,h->nr,h->fvt+rindex(0,i,h->nr,h->nr),1,0.0,drk,1);
			dq-=fDDOT(h->nr,drk,1,h->vq+rindex(0,i,h->nr,h->nr),1)/h->ev[i];
		}
		d1=sqrt(2*l2+dq*dq);	// with energy gain
		d1=M_SQRT2*l1;		// with fast canonical

		fprintf(stderr,"%5d: %12.4lf /cm l1= %lf sqrt(amu) A l2= %lf amu A^2 at %.0lf K\n"
			,i+1,wavenumber/PerCentimeter,l1/SqrtAMUAngstrom,l2/AMUAngstrom2,Ti/Kelvin);
		Eh=thermalvibrationalenergy(PlanckConstant*frequency,BoltzmannConstant*Ti);
		Ez=0.5*PlanckConstant*frequency;
		fprintf(stderr," dq= %+lf l1= %lf ev= %le\n"
			,dq/SqrtAMUAngstrom,d1/M_SQRT2/SqrtAMUAngstrom,h->ev[i]);

		//formattedvector(h->nr,h->vq+i*h->nr,"vq");
		formattedvector(h->nr,h->vr+i*h->nr,"vr ");
		//formattedvector(h->nc,h->vx+i*h->nc,"vx ");

		for(j=0;j<h->nc;j++) xkp[j]=xs[j]+(dq+d1)*h->vx[rindex(j,i,h->nc,h->nr)];
		calculatebtmatrix(h->nr,h->nc,h->z,khip,xkp,bt);
		formattedvector(h->nr,khip,"khi+");
		for(j=0;j<h->nc;j++) xkn[j]=xs[j]+(dq-d1)*h->vx[rindex(j,i,h->nc,h->nr)];
		calculatebtmatrix(h->nr,h->nc,h->z,khin,xkn,bt);
		formattedvector(h->nr,khin,"khi-");
#if(defined(SHRINK_PERTURB_XK0))
		for(j=0;j<h->nc;j++) xk0[j]=xs[j]+(dq)*h->vx[rindex(j,i,h->nc,h->nr)];
		calculatebtmatrix(h->nr,h->nc,h->z,khi0,xk0,bt);
		formattedvector(h->nr,khi0,"khi0");
#else
		fDCOPY(h->nc,xs,1,xk0,1);
		fDCOPY(h->nr,rs,1,khi0,1);
#endif
		for(j=0;j<h->nr;j++)
		{
			switch(h->z[j].type)
			{
			default:
				drk[j]=-0.25*(khip[j]+khin[j]-2*khi0[j]); 
				break;
			case DIHEDRAL_ANGLE:
			case DIHEDRAL_360:
				drk[j]=-0.25*(dihedraldifference(khip[j]-khi0[j])+dihedraldifference(khin[j]-khi0[j])); 
				break;
			}
//			fprintf(stderr,"#drk[%d]= %+lf %+lf\n",j,-0.25*(khip[j]+khin[j]-2*h->r[j]),drk[j]);
		}
		fDGEMV('N',h->nc,h->nr,1.0,bi,h->nc,drk,1,0.0,dxk,1);	
		if(detail)
		{
			formattedvector(h->nr,drk,"drk");
			formattedvector(h->nc,dxk,"dxk");
			fprintf(stderr," DRK^2= %le\n",fDDOT(h->nr,drk,1,drk,1));
			fprintf(stderr," DXK^2= %le\n",fDDOT(h->nc,dxk,1,dxk,1));
		}

		for(j=0;j<h->nc;j++)
			xkp[j]=xs[j]+(dq+M_SQRT1_2*d1)*h->vx[rindex(j,i,h->nc,h->nr)]+dxk[j];
		for(j=0;j<h->nc;j++)
			xkn[j]=xs[j]+(dq-M_SQRT1_2*d1)*h->vx[rindex(j,i,h->nc,h->nr)]+dxk[j];
		formattedvector(h->nc,xkp,"xkp");
		formattedvector(h->nc,xkn,"xkn");

		fprintf(stderr," drijk= ");
		for(k=1;k<h->na;k++)
			for(j=0;j<k;j++)
			{
				double dp=distance(j,k,xkp,bt);
				double dn=distance(j,k,xkn,bt);
				double d0=distance(j,k,xk0,bt);
				double de=distance(j,k,xs,bt);
				double dr=0.5*(dp+dn)-d0;
				dc[mindex(j,k)]+=d0-de;
				dm[mindex(j,k)]+=dr;
				d2[mindex(j,k)]+=0.5*((dp-d0)*(dp-d0)+(dn-d0)*(dn-d0));
				//d2[mindex(j,k)]+=(dp-de)*(dp-de)+(dn-de)*(dn-de);
				fprintf(stderr,"%8.4lf",dr/Angstrom);
			}
		fprintf(stderr,"\n");
		fprintf(stderr,"\n");
	}
//	fprintf(stderr," drij = ");
//	for(k=1;k<h->na;k++)
//		for(j=0;j<k;j++)
//			fprintf(stderr,"%8.4lf",dm[mindex(j,k)]/Angstrom);
//	fprintf(stderr,"\n");

//	fprintf(stderr," drij2= ");
//	for(k=1;k<h->na;k++)
//		for(j=0;j<k;j++)
//			fprintf(stderr,"%8.4lf",sqrt(d2[mindex(j,k)])/Angstrom);
//	fprintf(stderr,"\n");

	fprintf(stdout,"%8d",h->na*(h->na-1)/2);
	for(i=nz;i<h->nr;i++)
		fprintf(stdout,"  %.0lf",t->Tv[i-nz]/Kelvin);
	fprintf(stdout,"\n");
	for(k=1;k<h->na;k++)
		for(j=0;j<k;j++)
		{
			double re=distance(j,k,h->cc,bt);
			double r0=distance(j,k,xs,bt);
			int m=mindex(j,k);
			fprintf(stdout,"  %-6s %-6s ",h->ai[j].name,h->ai[k].name);
			fprintf(stdout," re= %15.10lf",re/Angstrom);
			fprintf(stdout," dr= %15.10lf",dr[mindex(j,k)]/Angstrom);
		//	if(detail>1) fprintf(stdout," dc= %15.10lf",(r0-re)/Angstrom);
		//	fprintf(stdout," dv= %15.10lf",(dm[m]+r0-re)/Angstrom);
			if(detail>1) fprintf(stdout," dc= %15.10lf",(dc[m])/Angstrom);
			if(detail>1) fprintf(stdout," du= %15.10lf",(dm[m])/Angstrom);
			fprintf(stdout," dv= %15.10lf",(dm[m]+dc[m])/Angstrom);
			fprintf(stdout," la= %15.10lf",sqrt(d2[m]+dm[m]*dm[m])/Angstrom);
		//	fprintf(stdout," la= %15.10lf",sqrt(d2[m]));
			fprintf(stdout,"\n");
		}

	free(dTdq);
	free(xs);
	free(rs);
	free(bi);
	free(bt);
	free(drk);
	free(khin);
	free(khip);
	free(khi0);
	free(dxk);
	free(xkn);
	free(xkp);
	free(xk0);
	free(d2);
	free(dm);
	free(dc);
	free(dr);
	return(0);
}

int shrinked(int detail,THERMAL *t,INTERNAL *h)
{
	int i,j,k,nx=0,nz=0,nm=(h->na)*(h->na-1)/2;
	double *bt=NULL,*bi=NULL;
	double *dr=NULL,*dc=NULL,*dm=NULL,*d2=NULL;
	double *drk=NULL,*dxk=NULL,*xkp=NULL,*xkn=NULL,*khip=NULL,*khin=NULL;
	double *rs=NULL,*xs=NULL,*dvt=NULL;
	double Ti,d1,l1,l2,frequency,wavenumber,kT,Eh;

	if(NULL==(dr=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(dc=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(dm=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(d2=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(xkp=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(xkn=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(dxk=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(khip=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(khin=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(drk=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(bt=(double *)calloc(h->nc*h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(bi=(double *)calloc(h->nc*h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(rs=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(xs=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(dvt=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");

	externalrotational(detail,h);
	nz = h->nr-h->nv;
	if(nz<0) nz=0;
	fprintf(stderr," evaluating rotational centrifugal distortion\n");
	fDCLEAR(h->nr,drk,1);
	for(i=0;i<3;i++)
	{
		Ti=t->Tr[i];
		kT=BoltzmannConstant*Ti;
		Eh=0.5*kT;
		fDGEMV('N',h->nr,h->nr,Eh,h->fi,h->nr,h->frt+rindex(0,i,h->nr,3),1,1.0,drk,1);
	}
	fDGEMV('N',h->nc,h->nr,1.0,h->bi,h->nc,drk,1,0.0,dxk,1);
	if(detail) formattedvector(h->nc,drk,"cr ");
	for(k=1;k<h->na;k++)
		for(j=0;j<k;j++)
		{
			fDCLEAR(h->nc,bt,1);
			double re=distance(j,k,h->cc,bt);
			dr[mindex(j,k)]=fDDOT(h->nc,dxk,1,bt,1);
		}
	

	nx = normalmode(detail,h);
	fprintf(stderr," second order shrinkage effect with local centrifugal\n");

	
	fprintf(stderr," evaluating centrifugal distortion at steady state\n");
	fDCLEAR(h->nr,drk,1);
	vibrationalcentrifugal(detail,h);
	vibrationalcentrifugaldistortion(detail,h,t->Tv,drk);
	fDGEMV('N',h->nc,h->nr,1.0,h->bi,h->nc,drk,1,0.0,dxk,1);
	if(detail) formattedvector(h->nc,drk,"cv ");
	for(k=1;k<h->na;k++)
		for(j=0;j<k;j++)
		{
			fDCLEAR(h->nc,bt,1);
			double re=distance(j,k,h->cc,bt);
			dc[mindex(j,k)]=fDDOT(h->nc,dxk,1,bt,1);
		}
	
//P	fDCOPY(h->nc,h->cc,1,xs,1);
//P	fDCOPY(h->nr,h->r,1,rs,1);
//P	fDAXPY(h->nr,1.0,drk,1,rs,1);
//P	excitemolecule(detail,h->nr,h->nc,h->z,0,h->vr+rindex(0,i,h->nr,h->nr),rs,h->m,xs);
//P	if(detail) formattedvector(h->nc,h->cc,"x0 ");
//P	if(detail) formattedvector(h->nc,xs,"xs");
//P	if(detail) formattedvector(h->nr,h->r,"r0 ");
//P	if(detail) formattedvector(h->nr,rs,"rs");
	fprintf(stderr," second order shrinkage effect\n");
	fDCOPY(h->nc,h->cc,1,xs,1);	// TEST to make du = dv shrunk
	fDCOPY(h->nr,h->r,1,rs,1);
	instantaneous(h->nc,h->nr,h->z,xs,h->m,rs,bi);
	for(i=0;i<nz;i++)
	{
		frequency=sqrt(fabs(h->ev[i]))*(0.5*M_1_PI);
		wavenumber=frequency/SpeedLight;		// hv = hc/l
		fprintf(stderr,"%5d: %12.4lf /cm \n"
			,i+1,wavenumber/PerCentimeter);
	}
	for(i=nz;i<h->nr;i++)
	{
		Ti=t->Tv[i-nz];
		// mass-weighted hessian = omega^2 = (2 pi nu)^2 = 4 pi^2 c^2 nubar^2
		// h nu = hc/l = hc nubar
		frequency=sqrt(fabs(h->ev[i]))*(0.5*M_1_PI);
		wavenumber=frequency/SpeedLight;
		l2=h_8pi2c/(wavenumber*tanh(hc_2k*wavenumber/Ti));
		l1=sqrt(l2);
		d1=M_SQRT2*l1; // factor of sqrt(2) seems to work but why ???

		fprintf(stderr,"%5d: %12.4lf /cm l1= %lf sqrt(amu) A l2= %lf amu A^2 at %lf K\n"
			,i+1,wavenumber/PerCentimeter,l1/SqrtAMUAngstrom,l2/AMUAngstrom2,Ti/Kelvin);

		//formattedvector(h->nr,h->vq+i*h->nr,"vq");
		formattedvector(h->nr,h->vr+i*h->nr,"vr ");
		//formattedvector(h->nc,h->vx+i*h->nc,"vx ");

		for(j=0;j<h->nc;j++) xkp[j]=xs[j]+d1*h->vx[rindex(j,i,h->nc,h->nr)];
		calculatebtmatrix(h->nr,h->nc,h->z,khip,xkp,bt);
		formattedvector(h->nr,khip,"khi+");
		for(j=0;j<h->nc;j++) xkn[j]=xs[j]-d1*h->vx[rindex(j,i,h->nc,h->nr)];
		calculatebtmatrix(h->nr,h->nc,h->z,khin,xkn,bt);
		formattedvector(h->nr,khin,"khi-");
		for(j=0;j<h->nr;j++)
		{
			switch(h->z[j].type)
			{
			default:
				drk[j]=-0.25*(khip[j]+khin[j]-2*rs[j]); 
				break;
			case DIHEDRAL_ANGLE:
			case DIHEDRAL_360:
				drk[j]=-0.25*(dihedraldifference(khip[j]-rs[j])+dihedraldifference(khin[j]-rs[j])); 
				break;
			}
//			fprintf(stderr,"#drk[%d]= %+lf %+lf\n",j,-0.25*(khip[j]+khin[j]-2*h->r[j]),drk[j]);
		}
		if(0)
		{
			for(j=0;j<h->nr;j++) fprintf(stderr,"%8.4lf",d1*h->vr[rindex(j,i,h->nr,h->nr)]);
			fprintf(stderr,"\n");
			for(j=0;j<h->nr;j++) fprintf(stderr,"%8.4lf",khip[j]-rs[j]);
			fprintf(stderr,"\n");
			for(j=0;j<h->nr;j++) fprintf(stderr,"%8.4lf",khin[j]-rs[j]);
			fprintf(stderr,"\n");
			for(j=0;j<h->nr;j++) fprintf(stderr,"%8.4lf",drk[j]);
			fprintf(stderr,"\n");
		}
		fDGEMV('N',h->nc,h->nr,1.0,bi,h->nc,drk,1,0.0,dxk,1);	
		if(detail)
		{
			formattedvector(h->nr,drk,"drk");
			formattedvector(h->nc,dxk,"dxk");
			fprintf(stderr," DRK^2= %le\n",fDDOT(h->nr,drk,1,drk,1));
			fprintf(stderr," DXK^2= %le\n",fDDOT(h->nc,dxk,1,dxk,1));
		}

		for(j=0;j<h->nc;j++)
			xkp[j]=xs[j]+M_SQRT1_2*d1*h->vx[rindex(j,i,h->nc,h->nr)]+dxk[j];
		for(j=0;j<h->nc;j++)
			xkn[j]=xs[j]-M_SQRT1_2*d1*h->vx[rindex(j,i,h->nc,h->nr)]+dxk[j];
		formattedvector(h->nc,xkp,"xkp");
		formattedvector(h->nc,xkn,"xkn");

		fprintf(stderr," drijk= ");
		for(k=1;k<h->na;k++)
			for(j=0;j<k;j++)
			{
				double dp=distance(j,k,xkp,bt);
				double dn=distance(j,k,xkn,bt);
				double d0=distance(j,k,xs,bt);
				double dr=0.5*(dp+dn)-d0;
			//	double de=distance(j,k,h->cc,bt);
			//	double dr=0.5*(dp+dn)-de;
				dm[mindex(j,k)]+=dr;
				d2[mindex(j,k)]+=0.5*((dp-d0)*(dp-d0)+(dn-d0)*(dn-d0));
				//d2[mindex(j,k)]+=(dp-de)*(dp-de)+(dn-de)*(dn-de);
				fprintf(stderr,"%8.4lf",dr/Angstrom);
			}
		fprintf(stderr,"\n");
		fprintf(stderr,"\n");
	}
//	fprintf(stderr," drij = ");
//	for(k=1;k<h->na;k++)
//		for(j=0;j<k;j++)
//			fprintf(stderr,"%8.4lf",dm[mindex(j,k)]/Angstrom);
//	fprintf(stderr,"\n");

//	fprintf(stderr," drij2= ");
//	for(k=1;k<h->na;k++)
//		for(j=0;j<k;j++)
//			fprintf(stderr,"%8.4lf",sqrt(d2[mindex(j,k)])/Angstrom);
//	fprintf(stderr,"\n");

	fprintf(stdout,"%8d",h->na*(h->na-1)/2);
	for(i=nz;i<h->nr;i++)
		fprintf(stdout,"  %.0lf",t->Tv[i-nz]/Kelvin);
	fprintf(stdout,"\n");
	for(k=1;k<h->na;k++)
		for(j=0;j<k;j++)
		{
			double re=distance(j,k,h->cc,bt);
			double r0=distance(j,k,xs,bt);
			int m=mindex(j,k);
			fprintf(stdout,"  %-6s %-6s ",h->ai[j].name,h->ai[k].name);
			fprintf(stdout," re= %15.10lf",re/Angstrom);
			fprintf(stdout," dr= %15.10lf",dr[mindex(j,k)]/Angstrom);
		//	if(detail>1) fprintf(stdout," dc= %15.10lf",(r0-re)/Angstrom);
		//	fprintf(stdout," dv= %15.10lf",(dm[m]+r0-re)/Angstrom);
			if(detail>1) fprintf(stdout," dc= %15.10lf",(dc[m])/Angstrom);
			if(detail>1) fprintf(stdout," du= %15.10lf",(dm[m])/Angstrom);
			fprintf(stdout," dv= %15.10lf",(dm[m]+dc[m])/Angstrom);
			fprintf(stdout," la= %15.10lf",sqrt(d2[m]+dm[m]*dm[m])/Angstrom);
		//	fprintf(stdout," la= %15.10lf",sqrt(d2[m]));
			fprintf(stdout,"\n");
		}

	free(dvt);
	free(xs);
	free(rs);
	free(bi);
	free(bt);
	free(drk);
	free(khip);
	free(khin);
	free(dxk);
	free(xkn);
	free(xkp);
	free(d2);
	free(dm);
	free(dc);
	free(dr);
	return(0);
}


// This function calculates the vibrational amplitudes and shrinkage corrected bond distances from the  
// quantum chemical equilibrium geometry and vibrational frequencies.
// The computational theory is outlined in Sipachev, Journal of Molecular Structure, 1985.
// NEED reference for definition of la

int shrunk(int detail,THERMAL *t,INTERNAL *h)
{
	int i,j,k,nx=0,nz=0,nm=(h->na)*(h->na-1)/2;
	double *bt=NULL;
	double *dm=NULL,*d2=NULL;
	double *drk=NULL,*dxk=NULL,*xkp=NULL,*xkn=NULL,*khip=NULL,*khin=NULL;
	double Ti,d1,l1,l2,frequency,wavenumber;

	fprintf(stderr," second order shrinkage effect\n");

	if(NULL==(dm=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(d2=(double *)calloc(nm,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(xkp=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");	//positively displaced cartesian coordinates
	if(NULL==(xkn=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory"); 	//negatively displaced cartesian coordinates
	if(NULL==(dxk=(double *)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(khip=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");	//positively displaced internal coordinates
	if(NULL==(khin=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory"); 	//negatively displaced internal coordinates
	if(NULL==(drk=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	if(NULL==(bt=(double *)calloc(h->nc*h->nr,sizeof(double)))) ued3error("lack of memory");
	
	nx = normalmode(detail,h);		// returns the number of internal coordinates
	nz = h->nr-h->nv;			// number of excess internal coordinates
	if(nz<0) nz=0;
	//fprintf(stderr,"h->nr= %d h->nv= %d h->nz= %d nz= %d\n",h->nr,h->nv,h->nz,nz);
	formattedvector(h->nr,h->r,"r0 ");
	for(i=0;i<nz;i++)
	{
		frequency=sqrt(fabs(h->ev[i]))*(0.5*M_1_PI);
		wavenumber=frequency/SpeedLight;		// hv = hc/l
		fprintf(stderr,"%5d: %12.4lf /cm \n"
			,i+1,wavenumber/PerCentimeter);
	}
	formattedvector(h->nc,h->cc,"cc ");
	for(i=nz;i<h->nr;i++)
	{
		Ti=t->Tv[i-nz];						// intitially assigned temperature of vibrational mode
		// mass-weighted hessian = omega^2 = (2 pi nu)^2 = 4 pi^2 c^2 nubar^2
		// h nu = hc/l = hc nubar
		frequency=sqrt(fabs(h->ev[i]))*(0.5*M_1_PI);		// harmonic vibrational frequency in hz
		wavenumber=frequency/SpeedLight;
		l2=h_8pi2c/(wavenumber*tanh(hc_2k*wavenumber/Ti));	// calculate the squared mean amplitude of vibration (harmonic approximation)
		l1=sqrt(l2);
		d1=M_SQRT2*l1; // arbitrary factor of sqrt(2) seems to work but why ??? (STP)

		fprintf(stderr,"%5d: %12.4lf /cm l1= %lf sqrt(amu) A l2= %lf amu A^2 at %lf K\n"
			,i+1,wavenumber/PerCentimeter,l1/SqrtAMUAngstrom,l2/AMUAngstrom2,Ti/Kelvin);

	//	formattedvector(h->nr,h->vq+i*h->nr,"vur");
		formattedvector(h->nr,h->vr+i*h->nr,"vr ");
	//	formattedvector(h->nc,h->vx+i*h->nc,"vx ");

		for(j=0;j<h->nc;j++) xkp[j]=h->cc[j]+d1*h->vx[rindex(j,i,h->nc,h->nr)];	// displace the cartesian coordinates along the direction of 
											// normal mode i
											// see Sipachev, JMS, 1985, pp. 149, first paragraph
		calculatebtmatrix(h->nr,h->nc,h->z,khip,xkp,bt);			// calculate the B Matrix Transpose at the updated geometry
		formattedvector(h->nr,khip,"khi+");					// updated redundant internal coordinates at new geometry
											// see Sipachev, JMS, 1985, pp. 149, first paragraph
		for(j=0;j<h->nc;j++) xkn[j]=h->cc[j]-d1*h->vx[rindex(j,i,h->nc,h->nr)];	// do the same in the negative direction of mode i
		calculatebtmatrix(h->nr,h->nc,h->z,khin,xkn,bt);
		formattedvector(h->nr,khin,"khi-");
		for(j=0;j<h->nr;j++)							// for all internal coordinates
		{
			switch(h->z[j].type)
			{
			default:
				drk[j]=-0.25*(khip[j]+khin[j]-2*h->r[j]); 		// see Sipachev, JMS, 1985, Eqn 21
				break;
			case DIHEDRAL_ANGLE:
			case DIHEDRAL_360:
				drk[j]=-0.25*(dihedraldifference(khip[j]-h->r[j])+dihedraldifference(khin[j]-h->r[j])); 
				break;
			}
//			fprintf(stderr,"#drk[%d]= %+lf %+lf\n",j,-0.25*(khip[j]+khin[j]-2*h->r[j]),drk[j]);
		}
		if(0)
		{
			for(j=0;j<h->nr;j++) fprintf(stderr,"%8.4lf",d1*h->vr[rindex(j,i,h->nr,h->nr)]);
			fprintf(stderr,"\n");
			for(j=0;j<h->nr;j++) fprintf(stderr,"%8.4lf",khip[j]-h->r[j]);
			fprintf(stderr,"\n");
			for(j=0;j<h->nr;j++) fprintf(stderr,"%8.4lf",khin[j]-h->r[j]);
			fprintf(stderr,"\n");
			for(j=0;j<h->nr;j++) fprintf(stderr,"%8.4lf",drk[j]);
			fprintf(stderr,"\n");
		}
		fDGEMV('N',h->nc,h->nr,1.0,h->bi,h->nc,drk,1,0.0,dxk,1);	// dxk := B^(-1) * drk, convert to cartesian coordinates 
										// see Sipachev, JMS, 1985, Eqn 21
		if(detail)
		{
			formattedvector(h->nr,drk,"drk");
			formattedvector(h->nc,dxk,"dxk");
			fprintf(stderr," DRK^2= %le\n",fDDOT(h->nr,drk,1,drk,1));
			fprintf(stderr," DXK^2= %le\n",fDDOT(h->nc,dxk,1,dxk,1));
		}

		for(j=0;j<h->nc;j++)
			xkp[j]=h->cc[j]+M_SQRT1_2*d1*h->vx[rindex(j,i,h->nc,h->nr)]+dxk[j]; 	// displace the cartesian coordinates again
												// from the updated midpoint 
												// along the direction of normal mode i
												// see Sipachev, JMS, 1985, pp. 149, middle paragraph
		for(j=0;j<h->nc;j++)
			xkn[j]=h->cc[j]-M_SQRT1_2*d1*h->vx[rindex(j,i,h->nc,h->nr)]+dxk[j];
		formattedvector(h->nc,xkp,"xkp");
		formattedvector(h->nc,xkn,"xkn");

		fprintf(stderr," drijk= ");
		for(k=1;k<h->na;k++)
			for(j=0;j<k;j++)
			{
				double dp=distance(j,k,xkp,bt);		// distance at the positive distortion
				double dn=distance(j,k,xkn,bt);		// distance at the negative distortion
				double de=distance(j,k,h->cc,bt);	// original equilibrium distance
				double dr=0.5*(dp+dn)-de;		// distortion from equilibrium distance
									// see Sipachev, JMS, 1985, Eqn 22
				dm[mindex(j,k)]+=dr;			// sum up all midpoint displacements due to all normal modes
									// see Sipachev, JMS, 1985, Eqn 19
									// this is the shrinkage correction
				d2[mindex(j,k)]+=0.5*((dp-de)*(dp-de)+(dn-de)*(dn-de)); 	// sum up "average variance" due to all normal modes
												//need reference
				//d2[mindex(j,k)]+=(dp-de)*(dp-de)+(dn-de)*(dn-de);
				fprintf(stderr,"%8.4lf",dr/Angstrom);
			}
		fprintf(stderr,"\n");
		fprintf(stderr,"\n");
	}
	fprintf(stderr," drij = ");		// overall distortion from equilibrium value
	for(k=1;k<h->na;k++)
		for(j=0;j<k;j++)
			fprintf(stderr,"%8.4lf",dm[mindex(j,k)]/Angstrom);
	fprintf(stderr,"\n");

	fprintf(stderr," drij2= ");
	for(k=1;k<h->na;k++)
		for(j=0;j<k;j++)
			fprintf(stderr,"%8.4lf",sqrt(d2[mindex(j,k)])/Angstrom);
	fprintf(stderr,"\n");

	fprintf(stdout,"%8d",h->na*(h->na-1)/2);
	for(i=nz;i<h->nr;i++)
		fprintf(stdout,"  %.0lf",t->Tv[i-nz]/Kelvin);
	fprintf(stdout,"\n");
	for(k=1;k<h->na;k++)
		for(j=0;j<k;j++)
		{
			double re=distance(j,k,h->cc,bt);
			fprintf(stdout,"  %-6s %-6s ",h->ai[j].name,h->ai[k].name);		// atom pair
			fprintf(stdout," re= %15.10lf",re/Angstrom);				// cartesian equilibrium distance
			fprintf(stdout," dv= %15.10lf",dm[mindex(j,k)]/Angstrom);		// overall distortion from equilibrium value
			//fprintf(stdout,"  la= %15.10lf",sqrt(d2[mindex(j,k)]));
			fprintf(stdout," la= %15.10lf",sqrt(d2[mindex(j,k)]+dm[mindex(j,k)]*dm[mindex(j,k)])/Angstrom);
			fprintf(stdout,"\n");
		}

	free(bt);
	free(drk);
	free(khip);
	free(khin);
	free(dxk);
	free(xkn);
	free(xkp);
	free(d2);
	free(dm);
	return(0);
}

