#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "programming.h"

#include "errors.h"
#include "outputs.h"
#include "interface.h"
#include "coordinates.h"
#include "cartesian.h"
#include "calculate.h"
#include "transformation.h"

int evaluatemmatrix(int detail,INTERNAL *h)
{
	calculatemmatrix(h->nc,h->m,h->mm,h->mi);
	return(h->nc);
}

#include "constantunits.h"
#include "intrinsic.h"


// This function evaluates the transpose of the B-matrix

int evaluatebtmatrix(int detail,INTERNAL *h)
{
	int i;
	if(detail>3)			// if more detail is requested
	{
		int j;
		double *x,*rp,*rn,*bt;
		if(NULL==(x=(double*)calloc(h->nc,sizeof(double)))) ued3error("lack of memory");
		if(NULL==(rp=(double*)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
		if(NULL==(rn=(double*)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
		if(NULL==(bt=(double*)calloc(h->nr*h->nc,sizeof(double)))) ued3error("lack of memory");
		for(j=0;j<h->nc;j++)
		{
			fDCOPY(h->nc,h->cc,1,x,1);			//copy the cartesian coordinates into the *x array
			x[j]+=DIFFSTEP*Angstrom;			//shift the current cartesian coordinate
			calculatebtmatrix(h->nr,h->nc,h->z,rp,x,h->bt);
			fDCOPY(h->nc,h->cc,1,x,1);			//return to the original coordinates
			x[j]-=DIFFSTEP*Angstrom;			//shift the same coordinate in the opposite direction
			calculatebtmatrix(h->nr,h->nc,h->z,rn,x,h->bt);
			for(i=0;i<h->nr;i++)
			{
				if((h->z[i].type==DIHEDRAL_ANGLE)||(h->z[i].type==DIHEDRAL_360))
					bt[rindex(j,i,h->nc,h->nr)]=dihedraldifference(rp[i]-rn[i])/(2*DIFFSTEP*Angstrom);
				else
					bt[rindex(j,i,h->nc,h->nr)]=(rp[i]-rn[i])/(2*DIFFSTEP*Angstrom);
			}
		}
		fprintf(stderr,"bt numerical =\n");
		printrectangularmatrix(h->nc,h->nr,bt);  		// nc rows, nr columns
		free(bt);
		free(rn);
		free(rp);
		free(x);
	}
	calculatebtmatrix(h->nr,h->nc,h->z,h->r,h->cc,h->bt);
	if(detail)
	{
		for(i=0;i<h->nr;i++)
			fprintf(stderr," %3d: type=%4d r=%10.5lf\n",i+1,h->z[i].type,h->z[i].r);
		fprintf(stderr,"\n");
	}
	return(h->nr);
}


// evaluate the B-Matrix from its transpose and save it in the strcture INTERNAL

int evaluatebmatrix(int detail,INTERNAL *h)
{
	evaluatebtmatrix(detail,h);
	calculatetranspose(h->nc,h->nr,h->bt,h->b);
	return(h->nr);
}


// calculate the G-Matrix and save it in the strcture INTERNAL

int evaluategmatrix(int detail,INTERNAL *h)
{
	if(detail) fprintf(stderr,"B=\n");	// B(nr*nc)
	if(detail) printrectangularmatrix(h->nr,h->nc,h->b);
	if(detail>1) fprintf(stderr,"B^T=\n");	// B^T(nc*nr)
	if(detail>1) printrectangularmatrix(h->nc,h->nr,h->bt);

	calculatemibtmatrix(h->nc,h->nr,h->bt,h->m,h->mibt); // M^-1 B^T 
	if(detail>2) fprintf(stderr,"M^-1 B^T=\n");
	if(detail>2) printrectangularmatrix(h->nc,h->nr,h->mibt);

	calculategtmatrix(h->nc,h->nr,h->bt,h->mibt,h->g); // G^T = G
	//calculategmatrix(h->nc,h->nr,h->b,h->mibt,h->g);
	//fDGEMM('N','N',h->nr,h->nr,h->nc,1.0,h->b,h->nr,h->mibt,h->nc,0.0,h->g,h->nr); // G= B (M^-1 B^T)
	if(detail) fprintf(stderr,"G=\n");
	if(detail) printrectangularmatrix(h->nr,h->nr,h->g);

	//calculatetriangular(h->nr,h->g,h->gt); // this will be done in calculategsmatrix
	//if(detail) fprintf(stderr,"tG=\n");
	//if(detail) printtriangularmatrix(h->nr,h->nr,h->gt);
	return(0);
}


// evaluate G- and B-Matrices

int evaluategsmatrix(int detail,INTERNAL *h)
{
	int INFO=0;
	calculategeigenvalue(h->nr,h->g,h->gi);				// to check the rank of G matrix, this statment seems obsolete
	h->nz=calculategsmatrix(h->nr,h->g,h->gi,h->gs,h->gq);		// calculate the number of eigenvlaues of G, which are below the tolerance threshold
	if(detail) fprintf(stderr," INFO= %d\n",INFO);
	if(detail) fprintf(stderr," G matrix is diagonalized\n");
	if(detail) fprintf(stderr," G^-1 = U L^-1 U^-1 =\n");
	if(detail) printrectangularmatrix(h->nr,h->nr,h->gi);

	if(detail>2) fprintf(stderr," G^1/2 = U L^1/2 U^-1 =\n");
	if(detail>2) printrectangularmatrix(h->nr,h->nr,h->gs);
	if(detail>2) fprintf(stderr," G^-1/2 = U L^-1/2 U^-1 =\n");
	if(detail>2) printrectangularmatrix(h->nr,h->nr,h->gq);

	calculatebimatrix(h->nc,h->nr,h->mibt,h->gi,h->bi); 		// B^-1 = (M^-1 B^T) G^-1
	// fDGEMM('N','N',h->nc,n,n,1.0,h->mibt,h->nc,st,n,0.0,h->bi,h->nc); // B^-1 = (M^-1 B^T) G^-1
	if(detail) fprintf(stderr,"B^-1 =\n");
	if(detail) printrectangularmatrix(h->nc,h->nr,h->bi);
	if(detail) checkvalidinverse(h->nc,h->nr,h->b,h->bi,"B"); 	// B B^-1 =? I

	if(detail>3)							// display the M B^-1 matrix product			
	{
		double *ms=NULL,*msbi=NULL;
		if(NULL==(ms=(double *)calloc(h->nc*h->nc,sizeof(double)))) ued3error("memory");
		if(NULL==(msbi=(double *)calloc(h->nc*h->nr,sizeof(double)))) ued3error("memory");
		int i,j;
		for(i=0;i<h->nc;i++)
			ms[rindex(i,i,h->nc,h->nc)]=(h->m[i]);		// ms = mass matrix (nc X nc)
		fDGEMM('N','N',h->nc,h->nr,h->nc,1.0,ms,h->nc,h->bi,h->nc,0.0,msbi,h->nc);
		fprintf(stderr,"M^1/2 B^-1 =\n");
		printrectangularmatrix(h->nc,h->nr,msbi);
		free(msbi);
		free(ms);
	}

	return(h->nz);
}

int evaluatebimatrix(int detail,INTERNAL *h)
{
	int i,j,info=0;
	fDCOPY(h->nr*h->nr,h->g,1,h->gi,1);
	fDPOTRF('U',h->nr,h->gi,h->nr,&info);	// cholesky factorization
	if(detail) fprintf(stderr,"info= %d\n",info);
	if(detail) fprintf(stderr,"U=\n");
	if(detail) printrectangularmatrix(h->nr,h->nr,h->gi);
	fDPOTRI('U',h->nr,h->gi,h->nr,&info); 	// inverse G
	if(detail) fprintf(stderr,"info= %d\n",info);
	if(detail) fprintf(stderr,"G^-1=\n");
	if(detail) printrectangularmatrix(h->nr,h->nr,h->gi);
	for(j=0;j<h->nr;j++)	// retrieve symmetry
		for(i=0;i<j;i++)
			h->gi[rindex(j,i,h->nr,h->nr)]=h->gi[rindex(i,j,h->nr,h->nr)];
	if(detail) fprintf(stderr,"G^-1=\n");
	if(detail) printrectangularmatrix(h->nr,h->nr,h->gi);
	
	//matrixmultiply(h->nc,h->nr,h->nr,h->mibt,h->nc,h->gi,h->nr,h->bi,h->nc); // B^-1 = (M^-1 B^T) G^-1
	fDGEMM('N','N',h->nc,h->nr,h->nr,1.0,h->mibt,h->nc,h->gi,h->nr,0.0,h->bi,h->nc); // B^-1 = (M^-1 B^T) G^-1
	if(detail) fprintf(stderr,"B^-1 =\n");
	if(detail) printrectangularmatrix(h->nc,h->nr,h->bi);
	if(detail) checkvalidinverse(h->nc,h->nr,h->b,h->bi,"B");
	if(0) checkvalidinverse(h->nr,h->nc,h->bi,h->b,"BI");
	return(0);
}


// evaluate the C^T tensor and the linear correction term

int evaluatectensor(int detail,INTERNAL *h)
{
	calculateb2ttensor(h->nc,h->nr,h->z,h->cc,h->m,h->ct);
	if(detail) fprintf(stderr,"evaluated curvature\n");
	if(detail>4) printttensor(h->nc,h->nc,h->nr,h->ct);
	calculatelinearcorrection(h->nc,h->nr,h->cc,h->z,h->lc);
	if(detail) fprintf(stderr,"evaluated linear correction\n");
	return(0);
}


// evaluate the dB^-1/dr tensor and the linear correction term

int evaluatebi2tensor(int detail,INTERNAL *h)
{
	calculatebi2tensor(h->nc,h->nr,h->z,h->cc,h->m,h->bi2);
	if(detail) fprintf(stderr,"evaluated curvature\n");
	if(detail>3) printttensor(h->nr,h->nr,h->nc,h->bi2);
	calculatelinearcorrection(h->nc,h->nr,h->cc,h->z,h->lc);
	if(detail) fprintf(stderr,"evaluated linear correction\n");
	return(0);
}


// evaluate the dG^-1/dr tensor in different ways.  The default is DGIDR_EXACT.

int evaluategi2tensor(int detail,INTERNAL *h)
{
	switch(h->dginverse)
	{
	default:
	case DGIDR_LINEAR:							// by subtraction of linear correction term
		calculatedgidrlineartensor(h->nc,h->nr,h->z,h->cc,h->m,h->gi2);	
		if(detail) fprintf(stderr,"evaluated curvature\n");
		if(detail>2) printrtensor(h->nr,h->nr,h->nr,h->gi2);
		fDCLEAR(h->nr,h->lc,1);
		if(detail) fprintf(stderr,"linear correction is implemented in GI2\n");
		break;
	case DGIDR_EXACT:							// by exact matrix multiplication (default)
		calculatedgidrtensor(h->nc,h->nr,h->z,h->cc,h->m,h->gi2);
		if(detail) fprintf(stderr,"evaluated curvature\n");
		if(detail>2) printrtensor(h->nr,h->nr,h->nr,h->gi2);
		calculatelinearcorrection(h->nc,h->nr,h->cc,h->z,h->lc);
		if(detail) fprintf(stderr,"evaluated linear correction\n");
		break;
	case DGIDR_SINGLE:
	case DGIDR_DOUBLE:
		calculatedgibitensor(h->nc,h->nr,h->z,h->cc,h->m,h->gi2);
		if(detail) fprintf(stderr,"evaluated curvature\n");
		if(detail>2) printrtensor(h->nr,h->nr,h->nr,h->gi2);
		calculatelinearcorrection(h->nc,h->nr,h->cc,h->z,h->lc);
		if(detail) fprintf(stderr,"evaluated linear correction\n");
		break;
	}
	return(0);
}


// this function assign the elements of the inverted hessian matrix (in internal coordinates) to the structure INTERNAL

int evaluatehimatrix(int detail,INTERNAL *h)
{
	calculatehimatrix(h->nr,h->fr,h->fi);
	if(detail) fprintf(stderr,"inversed hessian matrix\n");
	if(detail) fprintf(stderr,"Hr^-1=\n");
	if(detail) printrectangularmatrix(h->nr,h->nr,h->fi);
	return(0);
}


// convert cartesian gradient to gradient in internal coordinates, Gr = (B^-1)^T Gx
// convert cartesian Hessian to Hessian in internal coordinates (to first order only!), Hr = (B^-1)^T Hx B^-1 - (B^-1)^T C Gr B^-1

int convertgradient(int detail,INTERNAL *h,CARTESIAN *f)

 	
//       NR*nc NC*nc NC*nr
{
	double *u=NULL;
	fDGEMV('T',h->nc,h->nr,1.0,h->bi,h->nc,f->cg,1,0.0,h->gr,1);	  		//  Gr = (B^-1)^T Gx
	
	if(NULL==(u=(double*)calloc(h->nr*h->nc,sizeof(double))))
                ued3error("lack of memory");
	fDGEMM('T','N',h->nr,h->nc,h->nc,1.0,h->bi,h->nc,f->ff,h->nc,0.0,u,h->nr);	//  u = (B^-1)^T Hx 	
	if(detail) fprintf(stderr," u= \n");
	if(detail) printrectangularmatrix(h->nr,h->nc,u);
	fDGEMM('N','N',h->nr,h->nr,h->nc,1.0,u,h->nr,h->bi,h->nc,0.0,h->fr,h->nr);	//  Hr = (B^-1)^T Hx B^-1

	if(detail) fprintf(stderr," Gr= \n");
	if(detail) printrectangularmatrix(h->nr,1,h->gr);
	if(detail) fprintf(stderr," Hr= \n");
	if(detail) printrectangularmatrix(h->nr,h->nr,h->fr);
	free(u);
	return(0);
}


// This function evaluates the various transformation matrices and even higher order correction terms (tensors)

int transform(int detail,INTERNAL *h,CARTESIAN *f)
{
	int i;
	massweighting(f);
	
// copy data

	h->nv=f->nv;
	fDCOPY(h->nc,f->cc,1,h->cc,1);
	fDCOPY(h->nc,f->m ,1,h->m ,1);
	formattedvector(h->nc,h->m,"m");
	for(i=0;i<h->na;i++) h->an[i]=f->an[i];
	
// evaluate the various transfromation matrices
	
	//evaluatemmatrix(detail,h);
	//if(detail) fprintf(stderr,"evaluated M matrix\n");
	evaluatebmatrix(detail,h);				
	if(detail) fprintf(stderr,"evaluated B matrix\n");
	evaluategmatrix(detail,h);				 
	if(detail) fprintf(stderr,"evaluated G matrix\n");
	evaluategsmatrix(detail,h);
	if(detail) fprintf(stderr,"evaluated G^-1, G^-1/2, G^1/2, and B^-1 matrices\n");
	//evaluatebimatrix(detail,h);
	//if(detail) fprintf(stderr,"evaluated G^-1 and B^-1 matrix\n");

	convertgradient(detail,h,f);			// convert gradient and hessian to redundant internal coordinates
	if(detail) fprintf(stderr,"converted gradient\n");

// evaluate higher order correction terms (tensors)	
	
	evaluatectensor(detail,h);
	if(detail) fprintf(stderr,"evaluated C^T tensor\n");
	evaluatebi2tensor(detail,h);		// needs commenting			
	if(detail) fprintf(stderr,"evaluated BI2 tensor\n");
	evaluategi2tensor(detail,h);		// needs commenting
	if(detail) fprintf(stderr,"evaluated GI2 tensor\n");
	evaluatehimatrix(detail,h);		// calculate the inverted Hessian (to first order only!)
	if(detail) fprintf(stderr,"inverted H matrix\n");

	return(h->nr);
}

int evaluategmatrix_direct(int detail,INTERNAL *h)
{
	if(detail) fprintf(stderr,"B=\n");
	if(detail) printrectangularmatrix(h->nr,h->nc,h->b);
	calculategmatrix_direct(h->nc,h->nr,h->m,h->b,h->g);
	if(detail) fprintf(stderr,"G=\n");
	if(detail) printrectangularmatrix(h->nr,h->nr,h->g);
	return(0);
}

