#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "atomic.h"
#include "constantunits.h"
#include "intrinsic.h"
#include "programming.h"

#include "errors.h"
#include "outputs.h"
#include "interface.h"
#include "coordinates.h"
#include "calculate.h"
#include "cartesian.h"
#include "internal.h"

#include "vibrational.h"

void rotationalcentrifugalforce(int detail,INTERNAL *h,int n,double *T,double *fr)
{
	int i;
	double kT,Eh;
	fprintf(stderr," evaluating rotational centrifugal force\n");
	fDCLEAR(h->nr,fr,1);
	for(i=3-n;i<3;i++)
	{
		kT=BoltzmannConstant*T[i];
		Eh=0.5*kT;	// thermal mean classical rotational energy = 0.5 kT ?
		fDAXPY(h->nr,Eh,h->frt+rindex(0,i,h->nr,3),1,fr,1);	// scaled to energy
	}
	return;
}

void rotationalcentrifugaldistortion(int detail,INTERNAL *h,int n,double *T,double *cr)
{
	double *fr;
	if(NULL==(fr=(double*)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	fprintf(stderr," evaluating rotational centrifugal distortion\n");
	fDCLEAR(h->nr,cr,1);
	rotationalcentrifugalforce(detail,h,n,T,fr);
	fDGEMV('N',h->nr,h->nr,1.0,h->fi,h->nr,fr,1,0.0,cr,1);
	free(fr);
	return;
}

double thermalvibrationalenergy(double hv,double kT)
{
#if(defined(CLASSICAL_TESTING))
	return (kT-0.5*hv);
#else
	double Eh=0;
	if(kT>0) Eh=hv*exp(-hv/kT)/(1-exp(-hv/kT));
	return Eh;
#endif
}

double wWRS(double Ep)
{
#if(defined(CLASSICAL_TESTING))
	return (0.0);
#else
	return (exp(-Ep-0.5*M_PI));
#endif
}

double a_wWRS(double E,INTERNAL *h)
{
	int i,nz=h->nr-h->nv;
	double v1=0,v2=0,f;
	for(i=nz;i<h->nr;i++)
	{
		f=sqrt(fabs(h->ev[i]))*(0.5*M_1_PI);
		v1+=f;
		v2+=dsqr(f);
	}
	double beta = (h->nv-1)*v2/dsqr(v1);
	double Ep=E/(0.5*PlanckConstant*v1);
	double a=1-beta*wWRS(Ep);
	fprintf(stderr," WRS: beta= %le, Ep= %le, w= %le a= %lf\n",beta,Ep,wWRS(Ep),a);
	return a;
}

double thermalzeropointfraction(double *T,INTERNAL *h)
{
	int i,nz=h->nr-h->nv;
	double a=1,Ti,hv,kT,Eh=0,Ez=0,frequency;
	for(i=nz;i<h->nr;i++)
	{
		Ti=T[i-nz];
		frequency=sqrt(fabs(h->ev[i]))*(0.5*M_1_PI);
		hv=PlanckConstant*frequency;
		kT=BoltzmannConstant*(Ti);
		Eh+=thermalvibrationalenergy(hv,kT);	// thermal energy
		Ez+=0.5*hv;
	}
	a=a_wWRS(Eh,h);
	fprintf(stderr," WRS: a= %lf at E= %le Ez= %le\n",a,Eh,Ez);
	return (a);
}

double vibrationalcentrifugalforce(int detail,INTERNAL *h,double *T,double *fv)
{
	int i,nz=h->nr-h->nv;
	double a=1,Ti,hv,kT,Eh,Ez,Tq,wavenumber,frequency;
	a=thermalzeropointfraction(T,h);
	fDCLEAR(h->nr,fv,1);
	for(i=nz;i<h->nr;i++)
	{
		Ti=T[i-nz];
		frequency=sqrt(fabs(h->ev[i]))*(0.5*M_1_PI);
		wavenumber=frequency/(SpeedLight);
		hv=PlanckConstant*frequency;
		kT=BoltzmannConstant*(Ti);
		Eh=thermalvibrationalenergy(hv,kT);	// thermal energy
		Ez=0.5*hv;
		Tq=0.5*(Eh+a*Ez);	// mean kinetic energy of thermal energy + zero-point
		fprintf(stderr,"%5d: %12.4lf /cm T= %.0lf K <T> = 0.5 ( Eh = %le + Ez = %le ) eu\n"
			,i+1,wavenumber/PerCentimeter,Ti/Kelvin,Eh,Ez);
		// scaled to instantaneous kinetic energy
		fDAXPY(h->nr,Tq,h->fvt+rindex(0,i,h->nr,h->nr),1,fv,1);
		if(detail>1)
		{
			double *dr=NULL;
			if(NULL==(dr=(double*)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
			fDGEMV('N',h->nr,h->nr,Tq,h->fi,h->nr,h->fvt+rindex(0,i,h->nr,h->nr),1,0.0,dr,1);
			if(detail>2) formattedvector(h->nr,h->vq+rindex(0,i,h->nr,h->nr),"vq");
			formattedvector(h->nr,dr,"dr");
			free(dr);
		}
	}
	fprintf(stderr,"\n");
	return(a);
}

double vibrationalcentrifugaldistortion(int detail,INTERNAL *h,double *T,double *cv)
{
	double a=1,*fv=NULL;
	if(NULL==(fv=(double *)calloc(h->nr,sizeof(double)))) ued3error("lack of memory");
	fDCLEAR(h->nr,cv,1);
	a=vibrationalcentrifugalforce(detail,h,T,fv);
	fDGEMV('N',h->nr,h->nr,1.0,h->fi,h->nr,fv,1,0.0,cv,1);
	free(fv);
	fprintf(stderr,"\n");
	return(a);
}

