#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "instrumental.h"
#include "programming.h"
#include "status.h"

#include "readdata.h"
#include "mathematical.h"
#include "calculate.h"

// UED3 uses ra (without dr) and lh (not lm)

int makediffractionlength(int detail,MOLECULAR *molecule)
{
	MLS *mls=&(molecule->mls[molecule->i]);
	int i;
	double lh2;
	if(UED3)
	{
		for(i=0;i<mls->nl;i++)
		{
			lh2=mls->lh[i]*mls->lh[i];
			mls->lm[i]=mls->lh[i];
			mls->dr[i]=0;
			mls->rg[i]=mls->re[i]+1.5*mls->a[i]*lh2;
			mls->ra[i]=mls->rg[i]-lh2/mls->re[i];
			mls->k[i]=0;
		}
	}
	else
	{
		for(i=0;i<mls->nl;i++)
		{
			lh2=mls->lh[i]*mls->lh[i];
			mls->lm[i]=sqrt(lh2+1.5*mls->a[i]*mls->a[i]*lh2*lh2);
			mls->dr[i]=1.5*mls->a[i]*mls->lm[i]*mls->lm[i];
			mls->rg[i]=mls->re[i]+1.5*mls->a[i]*lh2+mls->dr[i];
			mls->ra[i]=mls->rg[i]-lh2/mls->re[i];
			mls->k[i]=lh2*lh2/6;
		}
	}

	if(detail>DETAIL_DIFFRACTION_LENGTH) 
	{
		for(i=0;i<mls->nl;i++)
			fprintf(stderr,"%5d %s %s %d %lf %lf %lf\n",i
				,mls->a1[i],mls->a2[i],mls->d[i],mls->ra[i],mls->lh[i],mls->k[i]);
	}

	return(mls->nl);
}

double waveconstant(double KE) // 1021.9979, 12.3984186
{
	double reciprocal_lambda=sqrt((1022.000+KE)*KE)/12.398521;
	return(4*PI*reciprocal_lambda);
//	double tmp=1022.000+KE;
//	tmp=sqrt(tmp*KE)/12.398521;
//	return(4*PI*tmp);
}

// Calculate the values of s corresponding to individual pixels of the
// raw data.  First find the angle using tan(theta) = (position on detector)
// divided by (camera_distance), and then calculate s from s = 2ksin(theta/2). 

int calculatespix(DIFFRACTION *data,INSTRUMENTAL *instrument)
{
	int is;
	// int j;
	double wc=waveconstant(instrument->KineticEnergy);
	double rc=1.0E-4*instrument->PixelSize/instrument->CameraLength;
	double cd2=instrument->CameraLengthVariance;
	double s,ds;
	data->ds[0]=2*wc*(sin(0.5*atan(rc*0.5)));	// ds[0] will be used to normalize
	for(is=0;is<MAX_NUM_DATA;is++)
	//for(is=0;is<data->se;is++) 
	{
		data->theta[is]=atan(rc*is);
		s=wc*sin(0.5*data->theta[is]);
		// s=wc*sin(0.5*atan(rc*is));
		ds=wc*(sin(0.5*atan(rc*(is+0.5)))-sin(0.5*atan(rc*(is-0.5))));
		data->s[is]=s;
		data->s2[is]=s*s;
		data->s4[is]=data->s2[is]*data->s2[is];
		data->ds[is]=ds;
		data->damping[is]=exp(-0.5*data->s2[is]*cd2);
		// data->ps[0][is]=1;	// need a leveling information, will be done in makesdata
		// for(j=1;j<MAX_NUM_POLY;j++)
		// 	data->ps[j][is]=data->ps[j-1][is]*s;
//		if(DEBUG)fprintf(stderr,"pixel= %5d s= %10.5lf ds= %le\n",is,s,ds);
	}
	return(is);
}

int makesdata(DIFFRACTION *data,INSTRUMENTAL *instrument)
{
	int is,j,nb=0;
	double sf=0,r=0,rsh=1/data->s[data->se/2];
	double d,w,isigma=1;
	//if(data->chisqr>0) isigma=sqrt((data->ce-data->cs)/data->chisqr);
	//for(is=1;is<data->se-1;is++)
	//{
	//	//d=0.5*(data->imx[is-1]+data->imx[is+1])-data->imx[is];
	//	if(data->snx[is]>0)
	//	{
	//		//w=10*d/data->snx[is];
	//		//data->wg[is]=exp(-instrument->noise*w*w*w*w);
	//		////data->wg[is]=1/(1+instrument->noise*w*w*w*w);
	//		data->wg[is]=erfc(fabs(instrument->noise*data->chi[is]*isigma));
	//	}
	//	else
	//		data->wg[is]=0;
	//}
	for(is=0;is<data->se;is++)
	{
		data->wg[is]=0;
		switch(instrument->sMs)
		{
			default:
			case INDEX_LEVELING_FIFJ: sf=data->s[is]/data->fifj[is]; break;
			case INDEX_LEVELING_IA  : sf=data->s[is]/data->irt[is]; break;
		}
		data->sf[is]=sf;
		if(instrument->Reference) 
			sf*=data->irt[is];
		else 
			sf*=data->ds[0]/data->ds[is];
		data->smx[is]=data->imx[is]*sf;
		data->sdx[is]=data->snx[is]*sf;
		data->sme[is]=-data->smx[is];
		if(data->sdx[is]>0) 
			data->w[is]=1/(data->sdx[is]*data->sdx[is]);
			// data->w[is]=1/(data->sdx[is]*data->sdx[is]); // this goes along with error bar in NRiC
			// data->w[is]=1/(data->sdx[is]*data->sdx[is])*(data->ds[is]); // this doesn't go along with error bar in NRiC
			// data->w[is]=1/(data->sdx[is]*data->sdx[is])*(data->ds[is]/SBINNING);
			// data->w[is]=1/(data->s[is])/(data->sdx[is]*data->sdx[is])*data->ds[is]/SBINNING;
		else 
			data->w[is]=0;

		switch(instrument->polynomials)
		{
			case POLYNOMIAL : nb=0;
				data->ps[0][is]=data->sf[is];
				for(j=1;j<MAX_NUM_POLY;j++)
					data->ps[j][is]=data->ps[j-1][is]*(rsh*data->s[is]);
				break;
			default:
			case POLYNOMIALXE : nb=0;
				data->ps[0][is]=data->sf[is]*data->irt[is];
				for(j=1;j<MAX_NUM_POLY;j++)
					data->ps[j][is]=data->ps[j-1][is]*(rsh*data->s[is]);
				break;
			case RECIPROCAL : nb=0;
				data->ps[0][is]=data->sf[is];
				for(j=1;j<MAX_NUM_POLY;j++)
					data->ps[j][is]=data->ps[j-1][is]/(rsh*data->s[is]);
				break;
			case RECIPROCALXE : nb=0;
				data->ps[0][is]=data->sf[is]*data->irt[is];
				for(j=1;j<MAX_NUM_POLY;j++)
					data->ps[j][is]=data->ps[j-1][is]/(rsh*data->s[is]);
				break;
			case RECIPROCAL2 : nb=0;
				data->ps[0][is]=data->sf[is];
				for(j=1;j<MAX_NUM_POLY;j++)
					data->ps[j][is]=data->ps[j-1][is]/sqrt(rsh*data->s[is]);
				break;
		}
	}
	data->sf[0]=0;
//	fprintf(stderr," background polynomial is from %dth order in intensity or %dth in sM(s)\n",nb,nb+1);
	return(is);
}

int makefifj(DIFFRACTION *data,int ne,ELEMENTAL *ele,INSTRUMENTAL *instrument)
{
	int is;
	int i=findnucleartype(instrument->fi,ne,ele);
	int j=findnucleartype(instrument->fj,ne,ele);
	for(is=data->ss;is<data->se;is++)
		data->fifj[is]=ele[i].fi[is]*ele[j].fi[is];
	return(1);	// leveling (1/fifj) doesn't affect X2 or fitting. it only affects frd curve. skip fifj for fitting
}

int makereference(DIFFRACTION *data,int ne,ELEMENTAL *ele,INSTRUMENTAL *instrument)
{
	int is;
	double irs;
	int ir=findnucleartype(instrument->ReferenceAtom,ne,ele);
	for(is=0;is<data->se;is++)
		data->irt[is]=0;
	for(is=1;is<data->se;is++)
	{
		irs = ele[ir].fi[is]*ele[ir].fi[is]
			+4* ele[ir].si[is]/(BOHR2*data->s4[is]);
		//// if(UED3)
		////	data->irt[is]+=irs;
		//// else
		//boxcaradd(data->irt,is,irs); // better
		profileadd(data->irt,is,irs,instrument->e);
	}
	return(ir);
}


int makesatmolecule(DIFFRACTION *data,MOLECULAR *molecule,int ne,ELEMENTAL *ele,INSTRUMENTAL *instrument)
{
	MLS *mls=&(molecule->mls[molecule->i]);
	int i,is,e1;
	double s,s4,ias;

	for(is=0;is<data->se;is++)
		data->iat[molecule->i][is]=0;	// initialize first for smoothing
	for(is=1;is<data->se;is++)
	{
		ias=0;
		s=data->s[is];
		s4=data->s4[is];
		for(i=0;i<mls->na;i++)
		{
			e1=mls->et0[i];
			ias+=mls->ne0[i]*(ele[e1].fi[is]*ele[e1].fi[is]+4*ele[e1].si[is]/(BOHR2*s4));
		}
		//boxcaradd(data->iat[molecule->i],is,ias);
		profileadd(data->iat[molecule->i],is,ias,instrument->e);
	}
	for(is=0;is<data->se;is++)
		data->sat[molecule->i][is]=data->sf[is]*data->iat[molecule->i][is];
	//fprintf(stderr,"na= %d\n",mls->na);
	return(mls->nl);
}

double sr_exps2l2_sinsr_ged(double s,double s2,int i,MLS *mls,double dsi)
{
	double sinsr = exp(-0.5*s2*mls->lm[i]*mls->lm[i])
			 * sin(s*(mls->ra[i]-mls->k[i]*s2))/(s*mls->re[i]);
	return(sinsr);
}

double sr_exps2l2_sinsr_ued3(double s,double s2,int i,MLS *mls,double dsi)
{ // lm=lh, re=ra, k=0 from ged for UED3 
	double sinsr = exp(-0.5*s2*mls->lh[i]*mls->lh[i])
			 * sin(s*mls->ra[i])/(s*mls->ra[i]);
	return(sinsr);
}

double sr_exps2l2_sinsr_ued4(double s,double s2,int i,MLS *mls,double dsi)
{
	double sinsr = exp(-0.5*s2*mls->lm[i]*mls->lm[i])
			 * sin(s*(mls->ra[i]-mls->k[i]*s2))/(s*mls->re[i])
			 * 2*sin(0.5*dsi*mls->ra[i])/(dsi*mls->ra[i]);
	return(sinsr);
}

double sr_exps2l2_sinsr_ued4l(double s,double s2,int i,MLS *mls,double dsi)
{
	long double s2l2=s2*mls->lm[i]*mls->lm[i];
	long double z2l2=mls->lm[i]*mls->lm[i]/(mls->re[i]*mls->re[i]);
	long double sinsrasre=sinl(s*(mls->rg[i]-mls->k[i]*s2))/(s*mls->re[i])*(1+(1-s2l2)*z2l2)
			-cosl(s*(mls->rg[i]-mls->k[i]*s2))*(z2l2+(3-s2l2)*z2l2*z2l2);
	long double sinsr = expl(-0.5*s2l2)
			 * sinsrasre
			 * 2*sin(0.5*dsi*mls->rg[i])/(dsi*mls->rg[i]);
	return(sinsr);
}

int makesmtmolecule(DIFFRACTION *data,MOLECULAR *molecule,int ne,ELEMENTAL *ele,INSTRUMENTAL *instrument)
{
	MLS *mls=&(molecule->mls[molecule->i]);
	int i,is,e1,e2;
	double s,s2;
	double (*sinsr)(double s,double s2,int i,MLS *mls,double dsi) = &sr_exps2l2_sinsr_ged ;

	if(UED3)
		sinsr = &sr_exps2l2_sinsr_ued3 ;
	else
	{
		if(UED4L) sinsr = &sr_exps2l2_sinsr_ued4l ;
		else if(UED4) sinsr = &sr_exps2l2_sinsr_ued4 ;
		else sinsr = &sr_exps2l2_sinsr_ged ;
	}

	for(is=0;is<data->se;is++)
		data->imt[molecule->i][is]=0;	// initialize first for smoothing

	for(is=1;is<data->se;is++)
	{
		long double ims=0;
		s=data->s[is];
		s2=data->s2[is];
		for(i=0;i<mls->nl;i++)
		{
			e1=mls->et1[i];
			e2=mls->et2[i];
			ims += 2 * mls->d[i] 
				 * ele[e1].fi[is]*ele[e2].fi[is]
				 * cos(ele[e1].ni[is]-ele[e2].ni[is])
				 * sinsr(s,s2,i,mls,data->ds[is]) ;
		}
		ims*=data->damping[is];
		//boxcaradd(data->imt[molecule->i],is,ims);
		profileadd(data->imt[molecule->i],is,ims,instrument->e);
	}

	for(is=0;is<data->se;is++)
		data->sam[molecule->i][is]=data->sat[molecule->i][is]+(data->smt[molecule->i][is]=data->sf[is]*data->imt[molecule->i][is]);
	return(mls->nl);
}

int makescaledata(DIFFRACTION *data,MOLECULAR *molecule)
{
	int i,is,n=molecule->n,j=molecule->j;
	double b,a;
	for(is=0;is<data->se;is++)
	{
		a=0;
		for(i=j;i<n;i++)
			a+=molecule->fraction[i]*data->smt[i][is];
		data->sMt[is]=a;

		a=0;
		for(i=0;i<n;i++)
			a+=molecule->fraction[i]*data->sat[i][is];
		data->sAt[is]=a;
		b=0;
		b+=data->atomicbackground*data->sat[0][is];
		for(i=0;i<data->np;i++)
			b+=data->pbc[i]*data->ps[i][is];
		data->bkg[is]=b;
		data->smc[is]=data->scale*data->smx[is]-a-b;
	}
	for(i=0;i<j;i++)
	{
		for(is=0;is<data->se;is++)
			data->smc[is]-=molecule->fraction[i]*data->smt[i][is];
	}
	for(is=0;is<data->se;is++)
		data->chi[is]=data->smc[is]-data->sMt[is];
		
	return(data->se);
}

int makefrd(DIFFRACTION *data,INSTRUMENTAL *instrument)
{
	int is,j;
	double rj,dssinsr,ds[MAX_NUM_DATA];
	double frx,frt;
	double sd=data->s[1]-data->s[0],fse=data->s[data->ce-1];
	int ns=(int)(fse/sd);
	int nr=ns*FRD_FFT_BIN;
	double f=FRDR(ns,sd,FRD_FFT_BIN);
	if(nr>MAX_NUM_FRD) nr=MAX_NUM_FRD;
	for(is=0;is<data->ss;is++)
		data->sMx[is]=0;
	for(is=data->ss;is<data->cs;is++)
		data->sMx[is]=data->sMt[is];
	for(is=data->cs;is<data->ce;is++)
		data->sMx[is]=data->smc[is];
	for(is=data->ce;is<data->se;is++)
		data->sMx[is]=data->sMt[is];
	for(is=0;is<data->se;is++)
		ds[is]=data->ds[is]*exp(-instrument->dampingk*data->s2[is]);
	fprintf(stderr," Fourier transformation form %d to %d out of %d\n",0,data->ce,data->se);
	for(j=0;j<nr;j++)
	{
		frx=0;
		frt=0;
		rj=f*j;
		for(is=1;is<data->ce;is++) // data->se;
		{
			dssinsr=ds[is]*sin(rj*data->s[is]);
			frt+=data->sMt[is]*dssinsr;
			frx+=data->sMx[is]*dssinsr;
		}
		data->frt[j]=frt;
		data->frx[j]=frx;
		// fprintf(stdout," %5d %10lf %10lf %10lf\n",j,rj,frt,frx);
	}
	return(nr);
}

double evaluatechisqr(int detail,DIFFRACTION *data,MOLECULAR *molecule)
{
	int is;
	double chi2=0;

	for(is=data->cs;is<data->ce;is++)
		chi2+=data->chi[is]*data->chi[is]*data->w[is];
	data->chisqr=chi2;
	if(detail>DETAIL_CHISQUARE) fprintf(stderr,"    Chi square: X2= %lf \n",chi2);
	return (chi2);
}

double evaluateR(int detail,DIFFRACTION *data,MOLECULAR *molecule)
{
	int is;
	double R,the2=0,chi2=0;

	for(is=data->cs;is<data->ce;is++)
	{
		chi2+=data->chi[is]*data->chi[is]*data->w[is];
		the2+=data->sMt[is]*data->sMt[is]*data->w[is];
	}
	R=sqrt(chi2/the2);
	data->R=R;
	data->chisqr=chi2;

	if(detail>DETAIL_CHISQUARE) fprintf(stderr,"    Chi square: X2= %lf R= %lf\n",chi2,R);
	return (R);
}

double evaluaterawchisqr(int detail,DIFFRACTION *data,MOLECULAR *molecule)
{
	int is,i;
	double chi,chi2=0;

	for(is=data->cs;is<data->ce;is++)
	{
		chi=(data->scale*data->smx[is] - data->bkg[is]);
		for(i=0;i<molecule->n;i++)
			chi-=molecule->fraction[i]*(data->sam[i][is]);
		chi2+=(chi*chi*data->w[is]);
	}
	if(detail) fprintf(stderr,"    Chi square: X2= %lf for 1/s= %lf\n",chi2/(data->scale*data->scale),1/data->scale);
	data->chisqr=chi2;
	return (chi2);
}

double evaluaterawR(int detail,DIFFRACTION *data,MOLECULAR *molecule)
{
	int is,i;
	double R,chi,t,chi2=0,the2=0;

	for(is=data->cs;is<data->ce;is++)
	{
		t=data->sMt[is];
		chi=(data->scale*data->smx[is] - data->bkg[is]);
		for(i=0;i<molecule->n;i++)
			chi-=molecule->fraction[i]*(data->sam[i][is]);
		the2+=t*t*data->w[is];
		chi2+=chi*chi*data->w[is];
	}
	R=sqrt(chi2/the2);
	data->chisqr=chi2;
	if(detail) fprintf(stderr,"    Chi square: X2= %le R= %lf for 1/s= %lf\n",chi2/(data->scale*data->scale),R,1/data->scale);
	return (R);
}

double CalculateError(int detail)
{
	return(-1.0);
}

