#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "errors.h"
#include "intrinsic.h"
#include "coordinates.h"

double evaluatebtcolumn(int type,int *ijkl,double *q,double *btc) // B^T in C storage
{
	double r=0;
	switch(type)
	{
		case BOND_STRETCHING:
			r=bt_distance(ijkl[0],ijkl[1],q,btc);
			break;
		case BOND_RECIPROCAL:
			r=bt_reciprocal(ijkl[0],ijkl[1],q,btc);
			break;
		case BOND_RECIPROCAL2:
			r=bt_reciprocal2(ijkl[0],ijkl[1],q,btc);
			break;
		case BOND_RECIPROCAL6:
			r=bt_reciprocal6(ijkl[0],ijkl[1],q,btc);
			break;
		case BENDING_ANGLE:
			r=bt_bending(ijkl[0],ijkl[1],ijkl[2],q,btc);
			break;
		case DIHEDRAL_ANGLE:
			r=bt_dihedral(ijkl[0],ijkl[1],ijkl[2],ijkl[3],q,btc);
			break;
		case BENDING2_ANGLE:
			r=bt_bending2(ijkl[0],ijkl[1],ijkl[2],ijkl[3],q,btc);
			break;
		case TANGENT_DIHEDRAL2:
			r=bt_tangent2dihedral(ijkl[0],ijkl[1],ijkl[2],ijkl[3],q,btc);
			break;
		case TANGENT_PI_DIHEDRAL2:
			r=bt_tangentpi2dihedral(ijkl[0],ijkl[1],ijkl[2],ijkl[3],q,btc);
			break;
		default:
			uerror("coordinates.c","unknown coordinate");
			break;
	}
	return(r);
}

double dihedraldifference(double d)
{
	int n=(int)floor(d/(2*M_PI)+0.5);
	double dd=d-n*2*M_PI;
	return(dd);
}

double ddot3(double *qi,double *qj)
{
	return(qi[0]*qj[0]+qi[1]*qj[1]+qi[2]*qj[2]);
}

void ftimes3(double f,double *qi,double *qj,double *q)
{
	q[0]=f*(qi[1]*qj[2]-qj[1]*qi[2]);
	q[1]=f*(qi[2]*qj[0]-qj[2]*qi[0]);
	q[2]=f*(qi[0]*qj[1]-qj[0]*qi[1]);
}

double displacement3(int i,int j,double *q,double *qij)
{
	int i3=i*3, j3=j*3, n;
	for(n=0;n<3;n++)
		qij[n]=q[i3+n]-q[j3+n];
	return(sqrt(ddot3(qij,qij)));
}

double bt_distance(int i,int j,double *q,double *bt)
{
	int i3=i*3, j3=j*3, n;
	double qij[3]={0};
	double rij=displacement3(i,j,q,qij);
	double sij=1/rij;
	for(n=0;n<3;n++)
		bt[j3+n]=-(bt[i3+n]=sij*qij[n]);
	return(rij);
}

double bt_reciprocal(int i,int j,double *q,double *bt)
{
	int i3=i*3, j3=j*3, n;
	double qij[3]={0};
	double zij=1/displacement3(i,j,q,qij);
	double zij3=zij*zij*zij;
	for(n=0;n<3;n++)
		bt[j3+n]=-(bt[i3+n]=-zij3*qij[n]);
	return(zij);
}

double bt_reciprocal2(int i,int j,double *q,double *bt)
{
	int i3=i*3, j3=j*3, n;
	double qij[3]={0};
	double zij=1/displacement3(i,j,q,qij);
	double zij2=zij*zij;
	double zij4=zij2*zij2;
	for(n=0;n<3;n++)
		bt[j3+n]=-(bt[i3+n]=-2*zij4*qij[n]);
	return(zij2);
}

double bt_reciprocal6(int i,int j,double *q,double *bt)
{
	int i3=i*3, j3=j*3, n;
	double qij[3]={0};
	double zij=1/displacement3(i,j,q,qij);
	double zij2=zij*zij;
	double zij4=zij2*zij2;
	double zij6=zij2*zij4;
	double zij8=zij4*zij4;
	for(n=0;n<3;n++)
		bt[j3+n]=-(bt[i3+n]=-6*zij8*qij[n]);
	return(zij6);
}

double inrange(double x)
{
	if(x>1) x=1;
	if(x<-1) x=-1;
	return(x);
}
	
double bt_bending(int i,int j,int k,double *q,double *bt) // p140 in JMSp v160 pp117 : wrong
{
	int i3=i*3,j3=j*3,k3=k*3,n;
	double qij[3]={0},qjk[3]={0};
	double rij=displacement3(i,j,q,qij);
	double rjk=displacement3(j,k,q,qjk);
	double sij=1/rij, sjk=1/rjk;
	double cosbijk=inrange(-sij*sjk*ddot3(qij,qjk)); 
	double bijk=acos(cosbijk);
	// ea = ejk X eji / sin bijk = eij X ejk / sin = dij x djk / rij rjk sin
	double f=sij*sjk/sin(bijk);
	double ea[3]={0};
	ftimes3(-f,qij,qjk,ea);
	// bi = ea X eji / rij = eij X ea / rij
	ftimes3(sij*sij,qij,ea,bt+i3);
	// bk = ejk X ea / rjk
	ftimes3(sjk*sjk,qjk,ea,bt+k3);
	for(n=0;n<3;n++)
		bt[j3+n]=-(bt[i3+n]+bt[k3+n]);
	return(bijk);
}

double bt_dihedral(int i,int j,int k,int l,double *q,double *bt)
{
	int i3=i*3,j3=j*3,k3=k*3,l3=l*3,n;
	double qij[3]={0},qjk[3]={0},qkl[3]={0};
	double rij=displacement3(i,j,q,qij);
	double rjk=displacement3(j,k,q,qjk);
	double rkl=displacement3(k,l,q,qkl);
	double sij=1/rij, sjk=1/rjk, skl=1/rkl;
	double cosbijk=inrange(-sij*sjk*ddot3(qij,qjk));
	double cosbjkl=inrange(-sjk*skl*ddot3(qjk,qkl));
	double ej[3]={0},ek[3]={0},ejk[3]={0};
	// ftimes3(-sij*sij*sjk/(1-cosbijk*cosbijk),qij,qjk,b+i3);
	// ftimes3(-skl*skl*sjk/(1-cosbjkl*cosbjkl),qkl,qjk,b+l3);
	ftimes3(sij*sjk,qij,qjk,ej);	// |ej| = rij rjk sin(ijk) sij sjk = sin(ijk)
	ftimes3(sjk*skl,qjk,qkl,ek);	// |ek| = rjk rkl sin(jkl) skl sjk = sin(jkl)
	double rj=sqrt(ddot3(ej,ej));
	double rk=sqrt(ddot3(ek,ek));
	for(n=0;n<3;n++)
	{
		bt[i3+n]=-sij/(1-cosbijk*cosbijk)*ej[n];
		bt[l3+n]=+skl/(1-cosbjkl*cosbjkl)*ek[n];
		bt[j3+n]=-sjk*((rjk-rij*cosbijk)*bt[i3+n]+rkl*cosbjkl*bt[l3+n]);
		bt[k3+n]=-sjk*((rjk-rkl*cosbjkl)*bt[l3+n]+rij*cosbijk*bt[i3+n]);
	}
	double cosdijkl=inrange(ddot3(ej,ek)/(rj*rk));
	double dijkl=acos(cosdijkl);
	ftimes3(1.,ej,ek,ejk);
	if(ddot3(ejk,qjk)>0) dijkl*=-1;
	return(dijkl);
}

double bt_bending2(int i,int j,int k,int l,double *q,double *bt) // p140 in JMSp v160 pp117 : wrong
{
	int i3=i*3,j3=j*3,k3=k*3,l3=l*3,n;
	double qij[3]={0},qjk[3]={0},qjl[3]={0};
	double rij=displacement3(i,j,q,qij);
	double rjk=displacement3(j,k,q,qjk);
	double rjl=displacement3(j,l,q,qjl);
	double sij=1/rij, sjk=1/rjk,sjl=1/rjl;
	double cosbijk=inrange(-sij*sjk*ddot3(qij,qjk)); 
	double cosbijl=inrange(-sij*sjl*ddot3(qij,qjl)); 
	double bijk=acos(cosbijk);
	double bijl=acos(cosbijl);
	// ea = ejk X eji / sin bijk = eij X ejk / sin = dij x djk / rij rjk sin
	double f=sij*sjk/sin(bijk);
	double g=sij*sjl/sin(bijl);
	double ea[3]={0};
	double eb[3]={0};
	ftimes3(-f,qij,qjk,ea);
	ftimes3(-g,qij,qjl,eb);
	ftimes3(sij*sij,qij,ea,bt+i3); // bi = ea X eji / rij = eij X ea / rij
	ftimes3(sjk*sjk,qjk,ea,bt+k3); // bk = ejk X ea / rjk
	ftimes3(sij*sij,qij,eb,ea);
	for(n=0;n<3;n++)
		bt[i3+n]-=ea[n];
	ftimes3(-sjl*sjl,qjl,eb,bt+l3); 
	for(n=0;n<3;n++)
		bt[j3+n]=-(bt[i3+n]+bt[k3+n]+bt[l3+n]);

	return(bijk-bijl);
}

double bt_tangent2dihedral(int i,int j,int k,int l,double *q,double *bt)
{
	int i3=i*3,j3=j*3,k3=k*3,l3=l*3,n;
	double r=0.5*bt_dihedral(i,j,k,l,q,bt); // dr/dx = dr/dy dy/dx 
	double z=tan(r); // dr/dy = s c^-1 ' = 1 + s^2 c^-2 = 1 + cot^2 = c^-2
	double cosr=cos(r);
	double f=0.5/(cosr*cosr);
	for(n=0;n<3;n++)
	{
		bt[i3+n]*=f;
		bt[l3+n]*=f;
		bt[j3+n]*=f;
		bt[k3+n]*=f;
	}
	return(z);
}

double bt_tangentpi2dihedral(int i,int j,int k,int l,double *q,double *bt)
{
	int i3=i*3,j3=j*3,k3=k*3,l3=l*3,n;
	double r=0.5*bt_dihedral(i,j,k,l,q,bt)-M_PI_2; // dr/dx = dr/dy dy/dx 
	double z=tan(r); // dr/dy = s c^-1 ' = 1 + s^2 c^-2 = 1 + cot^2 = c^-2
	double cosr=cos(r);
	double f=0.5/(cosr*cosr);
	for(n=0;n<3;n++)
	{
		bt[i3+n]*=f;
		bt[l3+n]*=f;
		bt[j3+n]*=f;
		bt[k3+n]*=f;
	}
	return(z);
}

