#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "programming.h"
#include "instrumental.h"
#include "status.h"

#include "empirical.h"

double harmonicamplitude(double mu, double nu,double T)
{
	double e2x=exp(47.96*nu/T);			// e2x=exp(hv/2kT)^2, 47.96=2*23.98
	double lh2=0.5047/(mu*nu)*(e2x+1)/(e2x-1);
	return (sqrt(lh2));
}

double rl2H(double mu, double nu,double T)
{
	double e2x=exp(47.96*nu/T);			// e2x=exp(hv/2kT)^2, 47.96=2*23.98
	double rl2=(mu*nu)*(e2x-1)/(e2x+1)/0.5047;
	return (rl2);
}

double drl2dnu(double mu, double nu, double T)
{
	double e2x=exp(47.96*nu/T);		// 23.98 = 0.5*h/k in K and THz(ps)
	double tanhx=(e2x-1)/(e2x+1);
	double d=mu/0.5047*(tanhx + nu*(23.98/T)*(1-tanhx*tanhx));
	return(d);
}

double estimatefrequency(int detail,double mu, double le,double T)
{
	int i;
	double rle2=1/(le*le),rli2,dv;
	double nu=0.5047/mu*rle2;
	// if(detail>DETAIL_EMPIRICAL_FREQUENCY) 
	//	fprintf(stderr,"    estimating frequency %5d: nu= %lf for le= %lf\n",0,nu,le);
	for(i=0;i<8;i++)
	{
		rli2=rl2H(mu,nu,T);
		dv=(rle2-rli2)/drl2dnu(mu,nu,T);
		nu+=dv;
		// if(detail>DETAIL_EMPIRICAL_FREQUENCY) 
		//	fprintf(stderr,"    estimating frequency %5d: nu= %lf for l2= %lf dv=%lf\n",i,nu,1/(rli2),dv);
		if(fabs(dv)<TOLERANCE_FREQUENCY) break;
	}
	// if(detail>DETAIL_EMPIRICAL_FREQUENCY) 
	// {
	// 	fprintf(stderr,"    estimated  frequency %5d: nu= %lf THz (%lf/cm) for le= %lf A (%lf)\n"
	// 			,i,nu,nu/2.99792458E-2,le,1/sqrt(rli2));
	// }
	return(nu);
}

double estimatefrequency_slower(int detail,double mu, double le,double T)
{
	int i;
	double c=0.5047/(mu*le*le);
	double bn=c;
	double e2x=exp(47.96*bn/T);
	double nu=c*(e2x+1)/(e2x-1);
	//fprintf(stderr,"    estimating frequency %5d: nu= %lf for le= %lf\n",0,nu,le);
	for(i=0;i<64;i++)
	{
		e2x=exp(47.96*nu/T);
		nu=c*(e2x+1)/(e2x-1);
		//fprintf(stderr,"    estimating frequency %5d: nu= %lf\n",i,nu);
		if(fabs(nu-bn)<TOLERANCE_FREQUENCY) break; else bn=nu;
	}
	return(nu);
}

int estimateamplitude(int detail,MOLECULAR *molecule,int ne,ELEMENTAL *ele)
{
	MLS *mls=&(molecule->mls[molecule->i]);
	int i,j,k;	// int ij;
	double r,l,f;	// double ratio;
	double T=molecule->temperature[molecule->i];

	if(detail>DETAIL_EMPIRICAL_AMPLITUDE) fprintf(stderr,"    temperature for %dth molecule is %lf\n",1+molecule->i,T);
	for(k=0;k<mls->nl;k++)
	{
		i=mls->et1[k];
		j=mls->et2[k];
		r=mls->re[k];
		if(r>1.85*(1+0.084*sqrt(i*j/36))) 	// is 36 integer ? please check this
		{
			mls->a[k]=0.0;
			if((1==j)||(1==i)) 
				l=0.050134+0.02738 *r;
			else
				l=0.013837+0.023398*r;	// empirical formula
				// l=0.015693+0.028125*sqrt(r)-0.00014*r; // from hydrocarbons
		}
		else 
		{
			mls->a[k]=2.0;
			if((1==j)||(1==i)) 
				l=0.050134+0.02738 *r-0.001805*r*r;	// if hydrogen(0)	// change it to find out H
			else
				l=0.013837+0.023398*r-0.000147*r*r;
		}

		mls->nu[k]=estimatefrequency(detail,mls->mu[k],l,300.0);
	}
	mls->temperature=300;
	return(mls->nl);
}

int scaleamplitude(int detail,MOLECULAR *molecule,int ne,ELEMENTAL *ele)
{
	MLS *mls=&(molecule->mls[molecule->i]);
	int k;
	double T=molecule->temperature[molecule->i];
//	if(detail>DETAIL_EMPIRICAL_AMPLITUDE) fprintf(stderr,"    temperature for %dth molecule is %lf\n",1+molecule->i,T);
	for(k=0;k<mls->nl;k++)
	{
		if(INDIVIDUAL_TEMPERATURE) T=mls->T[k];
		mls->lh[k]=harmonicamplitude(mls->mu[k],mls->nu[k],T);
		// mls->l[k]=mls->lh[k];
	}
	return(mls->nl);
}

int printamplitude0(int detail,MOLECULAR *molecule,int ne,ELEMENTAL *ele)
{
	MLS *mls=&(molecule->mls[molecule->i]);
	int i,j,k;
	double r,f,l,l300;
	double T=molecule->temperature[molecule->i];
	for(k=0;k<mls->nl;k++)
	{
		i=mls->et1[k];
		j=mls->et2[k];
		r=mls->re[k];
		f=mls->nu[k];
		l=mls->lh[k];
		l300=(harmonicamplitude(mls->mu[k],f,300));
		if(INDIVIDUAL_TEMPERATURE) T=mls->T[k];
		fprintf(stderr,"%5d %3d %3d: re= %10.6lf A  lh= %10.6lf A [%10.6lf] f= %8.4lf THz (%8.2lf/cm) rg= %10.6lf ra= %10.6lf\n"
			,k,i,j,r,l,l300,f,f/2.99792458E-2,mls->rg[k],mls->ra[k]);
	}
	return(mls->nl);
}

int printamplitude(int detail,MOLECULAR *molecule,int ne,ELEMENTAL *ele)
{
	MLS *mls=&(molecule->mls[molecule->i]);
	int i,j,k;
	double r,f,l,l300;
	double T=molecule->temperature[molecule->i];
	for(k=0;k<mls->nl;k++)
	{
		i=mls->et1[k];
		j=mls->et2[k];
		r=mls->re[k];
		f=mls->nu[k];
		l=mls->lh[k];
		l300=(harmonicamplitude(mls->mu[k],f,300));
		if(INDIVIDUAL_TEMPERATURE) T=mls->T[k];
		fprintf(stderr,"%5d %3d %3d:",k,i,j);
		fprintf(stderr," re= %10.6lf",mls->re[k]);
		fprintf(stderr," lh= %10.6lf",mls->lh[k]);
		fprintf(stderr," lm= %10.6lf",mls->lm[k]);
		fprintf(stderr,"  a= %10.6lf",mls->a[k]);
		fprintf(stderr," rg= %10.6lf",mls->rg[k]);
		fprintf(stderr," ra= %10.6lf",mls->ra[k]);
		fprintf(stderr,"\n");
	}
	return(mls->nl);
}

