#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "programming.h"
#include "instrumental.h"

#include "calculate.h"
#include "evaluate.h"
#include "gaussjordan.h"
#include "jacobi.h"
#include "numericalfit.h"
#include "montecarlo.h"
#include "outputs.h"
#include "errors.h"
#include "status.h"
#include "tools.h"
#include "fitjacobieigen.h"

int FitJacobiEigen(int detail,int np,int mb,double *p,double **r,double *beta,double **alpha,double **b
		,double **dyda,double **d2yda2,double **covar,double *oneda,double *da
		,DIFFRACTION *data,MOLECULAR *molecule,int ne,ELEMENTAL *elemental,INSTRUMENTAL *instrument,FITTING *fit)
{
	double error,sigma,uncorrelator=0.01;
	int i,k,j=0,m=-1,ng=0,n=(int)(1+np*2+fit->iteration/(np*100));
	double delchi = 1.0,oldchi;
	double alambda = 1.0;
	double d_tol= 0.1; d_tol=1e-4;
	double *pc=NULL;

	if(n<0) n=1+np*4;
	differentiate2(np,STEPSCALE,r,dyda,d2yda2,beta,alpha,data,molecule,ne,elemental,instrument,fit);
	oldchi=data->chisqr;

	for(j=0;j<np;j++)
		p[j] = *(r[j]);
	
	fprintf(stderr,"  jacobi marquardt will try %d iterations where fit->iteration= %ld\n",n,fit->iteration);
	for(j=0;j<n;j++)
	{
		ng=jacobieigen(detail,np,mb,p,r,beta,alpha,b,dyda,d2yda2,covar,oneda,da,&alambda,&delchi
				,data,molecule,ne,elemental,instrument,fit);
		fprintf(stderr,"    marquardt: j= %d ng= %d lambda= %le\n",j,ng,alambda);
		if(alambda > 256 )
		{
			uedmessage("Lambda Overflow");
			m=0; break;
		}
		else if(alambda < 0)
		{
			uedmessage("Limited Jacobi converged");
			m=1; break;
		}
		if((delchi<0.0)&&(fabs(delchi)<d_tol))
		{
			if(j<5) { fprintf(stderr," tight tolerance\n");  d_tol *=1.0e-3; continue; }
			else fprintf(stderr," chi square is hardly reduced\n");
			ng=jacobieigen(detail,np,mb,p,r,beta,alpha,b,dyda,d2yda2,covar,oneda,da,&alambda,&delchi
				,data,molecule,ne,elemental,instrument,fit);
			fprintf(stderr,"    marquardt: j= %d ng= %d lambda= %le\n",j,ng,alambda);
			m=1; break;
		}
	}
	if(data->chisqr<oldchi) m=1;
	if(MAX_ITERATION<=j)
	{
		uedmessage("NMAX exceeded");
	}
	if(detail)
	{
		differentiate(np,STEPSCALE,r,dyda,beta,alpha,data,molecule,ne,elemental,instrument,fit);
		sigma=sqrt(data->chisqr/(data->ce-data->cs-np));
		for(j=0;j<np;j++)
		{
			b[0][j]=beta[j];
			covar[j][j]=alpha[j][j];
			for(k=0;k<j;k++)
				covar[k][j]=(covar[j][k]=alpha[j][k]/(1+uncorrelator));
		}
		ng=cjacobisolve(covar,np,b,mb,JACOBI_NONLINEAR_SMALL,JACOBI_NONLINEARS);

		for(j=0;j<np;j++)
			b[0][j]=beta[j];
		ng=cjacobisolve(alpha,np,b,mb,JACOBI_NONLINEAR_SMALL,JACOBI_NONLINEARS);
		fprintf(stderr,"# using uncorrelator= %lf\n",0.0);
		for(i=0;i<np;i++)
		{
			error=sqrt(fabs(alpha[i][i]));
			if(alpha[i][i]<0) error*=-1;
			fprintf(stderr,"#UED4# r[%d] = %lf +/- (%lf x scalefactor)\n",i,*(r[i]),error);
		}
		fprintf(stderr,"\n");

		fprintf(stderr,"# using uncorrelator= %lf\n",uncorrelator);
		for(i=0;i<np;i++)
		{
			error=sqrt(fabs(covar[i][i]));
			if(covar[i][i]<0) error*=-1;
			fprintf(stderr,"#UED4# r[%d] = %lf +/- (%lf x scalefactor)\n",i,*(r[i]),error);
		}
		if(UED3) 
		{
			fprintf(stderr,"\n");
			fprintf(stderr,"    X2= %lf x scalefactor^-2)\n",data->chisqr);
			for(i=0;i<np;i++)
			{
				error=sigma*sqrt(fabs(covar[i][i]));
				if(covar[i][i]<0) error*=-1;
				fprintf(stderr,"#UED3# r[%d] = %lf +/- (%lf x scalefactor x scalefactor^-1)\n",i,*(r[i]),error);
			}
			fprintf(stderr,"\n");
		}

		differentiate2(np,STEPSCALE,r,dyda,d2yda2,beta,alpha,data,molecule,ne,elemental,instrument,fit);
		for(j=0;j<np;j++)
		{
			b[0][j]=beta[j];
			covar[j][j]=alpha[j][j];
			for(k=0;k<j;k++)
				covar[k][j]=(covar[j][k]=alpha[j][k]/(1+uncorrelator));
		}
		ng=cjacobisolve(covar,np,b,mb,JACOBI_NONLINEAR_SMALL,JACOBI_NONLINEARS);

		for(j=0;j<np;j++)
			b[0][j]=beta[j];
		ng=cjacobisolve(alpha,np,b,mb,JACOBI_NONLINEAR_SMALL,JACOBI_NONLINEARS);
		fprintf(stderr,"# using uncorrelated hessian and uncorrelator= %lf\n",0.0);
		for(i=0;i<np;i++)
		{
			error=sqrt(fabs(alpha[i][i]));
			if(alpha[i][i]<0) error*=-1;
			fprintf(stderr,"#UED4# r[%d] = %lf +/- (%lf x scalefactor)\n",i,*(r[i]),error);
		}
		fprintf(stderr,"\n");

		fprintf(stderr,"# using uncorrelated hessian and uncorrelator= %lf\n",uncorrelator);
		for(i=0;i<np;i++)
		{
			error=sqrt(fabs(covar[i][i]));
			if(covar[i][i]<0) error*=-1;
			fprintf(stderr,"#UED4# r[%d] = %lf +/- (%lf x scalefactor)\n",i,*(r[i]),error);
		}

	}
	return(m);
}

int jacobieigen(int detail,int np,int mb,double *p,double **r,double *beta,double **alpha,double **b
		,double **dyda,double **d2yda2,double **covar,double *oneda,double *da,double *alambda,double *delchi
		,DIFFRACTION *data,MOLECULAR *molecule,int ne,ELEMENTAL *elemental,INSTRUMENTAL *instrument,FITTING *fit)
{
	int j,k,ng=0,method=JACOBI_METHOD;
	double ochisq=data->chisqr,*pc=NULL;
	// double step=STEPSCALE,beta0=beta[0];
	if(detail>DETAIL_SO_MUCH) diagnosis(np,alpha);
	// adjust alpha and beta matrices to calulate da 
	for(j=0;j<np;j++)
	{
		for(k=0;k<np;k++) 
			covar[j][k] = alpha[j][k];
	}
	for(j=0;j<np;j++)
		b[0][j] = beta[j];

	switch(method)
	{
		default:
		case 1: ng=cgaussjordan(covar,np,b,mb); break;
		case 2:
			if(NULL==(pc=(double *)calloc(np,sizeof(double)))) uerror("jacobieigen","lack of memory");
			ng=choleskydecompose(np,covar,pc);
			choleskysolve(np,covar,pc,b[0],b[0]);
			//choleskyinverselowertriangle(np,covar,pc);
			//choleskyinverse(np,covar,pc);
			free(pc);
			break;
		case 3: ng=cjacobisolve(covar,np,b,mb,JACOBI_NONLINEAR_SMALL,JACOBI_NONLINEARS);
			break;
	}
	if(ng)
	{
		for(j=0;j<np;j++)
			oneda[j]=b[0][j]/(1+*alambda);
	}
	else
		fprintf(stderr,"    gaussjordan failed...\n");

	for(j=0;j<np;j++) da[j] = oneda[j]; 		// transfer the calculated result 
	for(j=0;j<np;j++) *(r[j]) = p[j] + da[j]; 	// adjust atry value 

	// backup the present value of alpha and beta for the case of the failure of the trial value 
	for(j=0;j<np;j++)
	{
		da[j] = beta[j];
		for(k=0;k<np;k++) 
			covar[j][k] = alpha[j][k];
	}

	/* calculate the theoretical with the new trial model */
	singlepoint(0,0,data,molecule,ne,elemental,instrument,fit); // will be done in differentiate
	*delchi = data->chisqr - ochisq;
	if(detail)
	{
		fprintf(stderr,"    X2= %lf <-- %lf\n",data->chisqr,ochisq);
		for(j=0;j<np;j++)
			fprintf(stderr,"        r[%d]-> %+lf dr= %+lf beta= %+le alpha= %+le\n"
					,j,*(r[j]),oneda[j],da[j],covar[j][j]);
		//fprintf(stderr,"\n");
	}

	if(*delchi < 0.0) 	// if it succeded 
	{
		// step=STEPSCALE*sqrt(0.01+*alambda);	///////////////////////////////////////////// not confirmed.
		// *alambda *= 0.1;
		*alambda *= 0.5;
		for(j=0;j<np;j++) p[j] = *(r[j]);
		//differentiate(np,STEPSCALE,r,dyda,beta,alpha,data,molecule,ne,elemental,instrument,fit); // numerical recipe
		differentiate2(np,STEPSCALE,r,dyda,d2yda2,beta,alpha,data,molecule,ne,elemental,instrument,fit); // faster
		// ochisq = data->chisqr; // will be done in beginning
	}
	else 	// or if it failed 
	{
		// *alambda *= 10.0; 	// original value was -> *alambda *= 10.0;
		*alambda *= 16.0; 	// original value was -> *alambda *= 10.0;
		if(*alambda<1) *alambda=1;
		for(j=0;j<np;j++)       // restore to old value 
		{
			*(r[j])=p[j];
			beta[j] = da[j];
			for(k=0;k<np;k++) 
				alpha[j][k] = covar[j][k];
		}
		singlepoint(0,0,data,molecule,ne,elemental,instrument,fit);
		data->chisqr=ochisq;
		if(detail) uedmessage("**** wrong direction ****");
	}
	return(ng);
}

