#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "programming.h"
#include "instrumental.h"

#include "calculate.h"
#include "evaluate.h"
#include "gaussjordan.h"
#include "jacobi.h"
#include "numericalfit.h"
#include "montecarlo.h"
#include "outputs.h"
#include "errors.h"
#include "status.h"
#include "tools.h"
#include "fitmarquardt.h"

int FitMarquardt(int detail,int np,int mb,double *p,double **r,double *beta,double **alpha,double **b
		,double **dyda,double **d2yda2,double **covar,double *oneda,double *da
		,DIFFRACTION *data,MOLECULAR *molecule,int ne,ELEMENTAL *elemental,INSTRUMENTAL *instrument,FITTING *fit)
{
	double error,sigma,uncorrelator=0.01;
	int i,k,j=0,m=-1,ng=0,n=(int)(1+MAX_ITERATION*fit->iteration/(np*100));
	double delchi = 1.0,oldchi;
	double alambda = -0.1;
	double d_tol= 0.1; d_tol=1e-4;
	double *pc=NULL;
	if(n<0) n=1+4*np;

	// d_tol=0.1; 
	// n=(int)sqrt((double)(np));

	differentiate2(np,STEPSCALE,r,dyda,d2yda2,beta,alpha,data,molecule,ne,elemental,instrument,fit);
	delchi=(oldchi=data->chisqr);

	// n=MAX_ITERATION;
	d_tol=1.0e-4;
	// d_tol=1.0e-8;
	// alambda = 32.0;	// trial value
	// alambda = 16.0;	// trial value
	// alambda = 4.0;	// trial value
	// alambda = 16.0;	// trial value
	alambda = 1.0;	// trial value

	for(j=0;j<np;j++)
		p[j] = *(r[j]);
	
	fprintf(stderr,"  marquardt will try %d iterations where fit->iteration= %ld\n",n,fit->iteration);
	for(j=0;j<n;j++)
	{
		ng=marquardt(detail,np,mb,p,r,beta,alpha,b,dyda,d2yda2,covar,oneda,da,&alambda,&delchi
				,data,molecule,ne,elemental,instrument,fit);
		fprintf(stderr,"    marquardt: j= %d ng= %d lambda= %le\n",j,ng,alambda);
		// if(alambda > 65536 )	// 1E5
		if(alambda > 256 )
		{
			uedmessage("Lambda Overflow");
			// CalculateError(detail);
			m=0; break;
		}
		if((delchi<0.0)&&(fabs(delchi)<d_tol))
		{
			if(j<5) { fprintf(stderr," tight tolerance\n");  d_tol *=1.0e-3; continue; }
			else fprintf(stderr," chi square is hardly reduced\n");
			// CalculateError(detail);
			ng=marquardt(detail,np,mb,p,r,beta,alpha,b,dyda,d2yda2,covar,oneda,da,&alambda,&delchi
				,data,molecule,ne,elemental,instrument,fit);
			fprintf(stderr,"    marquardt: j= %d ng= %d lambda= %le\n",j,ng,alambda);
			m=1; break;
		}
	}
	if(data->chisqr<oldchi) m=1;
	if(MAX_ITERATION<=j)
	{
		uedmessage("NMAX exceeded");
		//return(m);
	}
	if(detail)
	{
		// differentiate2(np,STEPSCALE,r,dyda,d2yda2,beta,alpha,data,molecule,ne,elemental,instrument,fit); 
		// da = 0 at minimum no matter how you define alpha
		// however, alpha^-1 gives error bar 
		// when you define alpha = \sum_i 1/\sigma_i^2 \del^2f/\del a_j\del a_k
		// therefore, you need to use differentiate.
		differentiate(np,STEPSCALE,r,dyda,beta,alpha,data,molecule,ne,elemental,instrument,fit); // numerical recipe
		sigma=sqrt(data->chisqr/(data->ce-data->cs-np));
		// sigma=sqrt(data->chisqr/(data->s[data->ce]-data->s[data->cs])*data->ds[0]);
		for(j=0;j<np;j++)
		{
			covar[j][j]=alpha[j][j];
			for(k=0;k<j;k++)
				covar[k][j]=(covar[j][k]=alpha[j][k]/(1+uncorrelator));
			// b[0][j] = beta[j];
		}
		switch(JACOBI_METHOD)
		{
			default:
			case 1: ng=cgaussjordan(covar,np,b,mb); break;
			case 2: if(NULL==(pc=(double *)calloc(np,sizeof(double)))) uerror("FitMarquardt","lack of memory");
				ng=choleskydecompose(np,covar,pc);
				//choleskysolve(np,covar,pc,b[0],b[0]);
				choleskyinverselowertriangle(np,covar,pc);
				choleskyinverse(np,covar,pc);
				free(pc);
				break;
			case 3: ng=cjacobisolve(covar,np,b,mb,JACOBI_NONLINEAR_SMALL,JACOBI_NONLINEARS); break;
		}
		switch(JACOBI_METHOD)
		{
			default:
			case 1: ng=cgaussjordan(alpha,np,b,mb); break;
			case 2: if(NULL==(pc=(double *)calloc(np,sizeof(double)))) uerror("FitMarquardt","lack of memory");
				ng=choleskydecompose(np,alpha,pc);
				//choleskysolve(np,covar,pc,b[0],b[0]);
				choleskyinverselowertriangle(np,alpha,pc);
				choleskyinverse(np,alpha,pc);
				free(pc);
				break;
			case 3: ng=cjacobisolve(alpha,np,b,mb,JACOBI_NONLINEAR_SMALL,JACOBI_NONLINEARS); break;
		}
		//ng=cgaussjordan(alpha,np,b,mb);
		fprintf(stderr,"# without using uncorrelator\n");
		for(i=0;i<np;i++)
		{
			error=sqrt(fabs(alpha[i][i]));
			if(alpha[i][i]<0) error*=-1;
			fprintf(stderr,"#UED4# r[%d] = %lf +/- (%lf x scalefactor)\n",i,*(r[i]),error);
		}
		fprintf(stderr,"\n");
		fprintf(stderr,"# using uncorrelator= %lf\n",uncorrelator);
		for(i=0;i<np;i++)
		{
			error=sqrt(fabs(covar[i][i]));
			if(covar[i][i]<0) error*=-1;
			fprintf(stderr,"#UED4# r[%d] = %lf +/- (%lf x scalefactor)\n",i,*(r[i]),error);
		}
		if(UED3) 
		{
			fprintf(stderr,"\n");
			fprintf(stderr,"    X2= %lf x scalefactor^-2)\n",data->chisqr);
			for(i=0;i<np;i++)
			{
				error=sigma*sqrt(fabs(covar[i][i]));
				if(covar[i][i]<0) error*=-1;
				fprintf(stderr,"#UED3# r[%d] = %lf +/- (%lf x scalefactor x scalefactor^-1)\n",i,*(r[i]),error);
			}
			fprintf(stderr,"\n");
		}

	}
	return(m);
}

int marquardt(int detail,int np,int mb,double *p,double **r,double *beta,double **alpha,double **b
		,double **dyda,double **d2yda2,double **covar,double *oneda,double *da,double *alambda,double *delchi
		,DIFFRACTION *data,MOLECULAR *molecule,int ne,ELEMENTAL *elemental,INSTRUMENTAL *instrument,FITTING *fit)
{
	int j,k,ng=0,method=JACOBI_METHOD;
	double ochisq=data->chisqr,*pc=NULL;
	// double step=STEPSCALE,beta0=beta[0];
	if(detail>DETAIL_SO_MUCH) diagnosis(np,alpha);
	// adjust alpha and beta matrices to calulate da 
	for(j=0;j<np;j++)
	{
		for(k=0;k<np;k++) 
			covar[j][k] = alpha[j][k];
		covar[j][j] = alpha[j][j]*(1.0 + *alambda);
	}
	for(j=0;j<np;j++)
		b[0][j] = beta[j];

	switch(method)
	{
		default:
		case 1: ng=cgaussjordan(covar,np,b,mb); break;
		case 2:
			if(NULL==(pc=(double *)calloc(np,sizeof(double)))) uerror("marquardt","lack of memory");
			ng=choleskydecompose(np,covar,pc);
			choleskysolve(np,covar,pc,b[0],b[0]);
			//choleskyinverselowertriangle(np,covar,pc);
			//choleskyinverse(np,covar,pc);
			free(pc);
			break;
		case 3: ng=cjacobisolve(covar,np,b,mb,JACOBI_NONLINEAR_SMALL,JACOBI_NONLINEARS);
			break;
	}
	if(ng)
	{
		for(j=0;j<np;j++)
		{
		// 	oneda[j]=0;
		// 	for(k=0;k<np;k++)
		// 		oneda[j]+=covar[j][k]*beta[k];
			oneda[j]=b[0][j];
		}
	}
	else
		fprintf(stderr,"    gaussjordan failed...\n");

	for(j=0;j<np;j++) da[j] = oneda[j]; 		// transfer the calculated result 
	for(j=0;j<np;j++) *(r[j]) = p[j] + da[j]; 	// adjust atry value 

	// backup the present value of alpha and beta for the case of the failure of the trial value 
	for(j=0;j<np;j++)
	{
		da[j] = beta[j];
		for(k=0;k<np;k++) 
			covar[j][k] = alpha[j][k];
	}

	/* calculate the theoretical with the new trial model */
	singlepoint(0,0,data,molecule,ne,elemental,instrument,fit); // will be done in differentiate
	*delchi = data->chisqr - ochisq;
	if(detail)
	{
		fprintf(stderr,"    X2= %lf <-- %lf\n",data->chisqr,ochisq);
		for(j=0;j<np;j++)
			fprintf(stderr,"                                 r[%d]-> %+lf dr= %+lf beta= %+le alpha= %+le\n"
					,j,*(r[j]),oneda[j],da[j],covar[j][j]);
		//fprintf(stderr,"\n");
	}

	if(*delchi < 0.0) 	// if it succeded 
	{
		// step=STEPSCALE*sqrt(0.01+*alambda);	///////////////////////////////////////////// not confirmed.
		// *alambda *= 0.1;
		*alambda *= 0.5;
		for(j=0;j<np;j++) p[j] = *(r[j]);
		//differentiate(np,STEPSCALE,r,dyda,beta,alpha,data,molecule,ne,elemental,instrument,fit); // numerical recipe
		differentiate2(np,STEPSCALE,r,dyda,d2yda2,beta,alpha,data,molecule,ne,elemental,instrument,fit); // faster
		// ochisq = data->chisqr; // will be done in beginning
	}
	else 	// or if it failed 
	{
		// *alambda *= 10.0; 	// original value was -> *alambda *= 10.0;
		*alambda *= 4.0; 	// original value was -> *alambda *= 10.0;
		for(j=0;j<np;j++)       // restore to old value 
		{
			*(r[j])=p[j];
			beta[j] = da[j];
			for(k=0;k<np;k++) 
				alpha[j][k] = covar[j][k];
		}
		singlepoint(0,0,data,molecule,ne,elemental,instrument,fit); // will be done in differentiate
		data->chisqr=ochisq;
		if(detail) uedmessage("**** wrong direction ****");
	}
	return(ng);
}

