#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "gaussjordan.h"

#define SWAP(a,b) {double temp=(a);(a)=(b);(b)=temp;}

int gerror(char *s)
{
	fprintf(stderr,"%s\n",s);
	return(1);
}

int choleskydecompose(int n, double **a,double *p)
{
	int i,j,k;
	double sum;
	for(i=0;i<n;i++)
	{
		sum=a[i][i];
		for(k=0;k<i;k++)
			sum-=a[i][k]*a[i][k];
		if(sum>0)
			p[i]=sqrt(sum);
		else
		{
			fprintf(stderr,"i= %d sum= %lf\n",i,sum);
			gerror("cholesky decomposition fails");
		}
		for(j=i+1;j<n;j++)
		{
			sum=a[i][j];
			for(k=0;k<i;k++)
				sum-=a[i][k]*a[j][k];
			a[j][i]=sum/p[i];
		}
	}
	return(n);
}

int choleskysolve(int n, double **a,double *p,double *b,double *x)
{
	int i,k;
	double sum;
	for(i=0;i<n;i++)
	{
		sum=b[i];
		for(k=i-1;k>=0;k--) sum-=a[i][k]*x[k];
		x[i]=sum/p[i];
	}
	for(i=n-1;i>=0;i--)
	{
		sum=x[i];
		for(k=i+1;k<n;k++) sum-=a[k][i]*x[k];
		x[i]=sum/p[i];
	}
	return(n);
}

int choleskyinverselowertriangle(int n,double **a,double *p)
{
	int i,j,k;
	double sum;
	for(i=0;i<n;i++)
	{
		a[i][i]=1/p[i];
		for(j=i+1;j<n;j++)
		{
			sum=0;
			for(k=i;k<j;k++) sum-=a[j][k]*a[k][i];
			a[j][i]=sum/p[j];
		}
	}
	return(n);
}

int choleskyinversediagonals(int n,double **a)
{
	int i,k;
	return(n);
}

int choleskyinverse(int n,double **a,double *p)
{
	int i,j,k;
	double t;
	for(i=0;i<n;i++)
	{
		for(j=i+1;j<n;j++)
		{
			t=0;
			for(k=j;k<n;k++)
				t+=a[k][i]*a[k][j];
			a[i][j]=t;
		}
		a[i][i]*=a[i][i];
		for(j=i+1;j<n;j++)
		{
			a[i][i]+=a[j][i]*a[j][i];
			a[j][i]=a[i][j];
		}
	}
	return(n);
}

int gaussjordan(double **a,int n,double **b,int m)
{
	int *indxc, *indxr, *ipiv;
	int i, icol=0, irow=0, j, k, l, ll;
	double big, dum, pivinv;

	if(NULL==(indxc=(int *)calloc(n,sizeof(int)))) if(gerror("lack of memory")) return(0);
	if(NULL==(indxr=(int *)calloc(n,sizeof(int)))) if(gerror("lack of memory")) return(0);
	if(NULL==(ipiv =(int *)calloc(n,sizeof(int)))) if(gerror("lack of memory")) return(0);
	for(i=0;i<n;i++)
	{
		ipiv[i]=0;
		indxr[i]=-1;
		indxc[i]=-1;
	}
	for(i=0;i<n;i++)
	{
		big=0.0;
		for(j=0;j<n;j++)
		{
			if(ipiv[j]!=1)
			{
				for(k=0;k<n;k++)
				{
					if(ipiv[k]==0)
					{
						if(fabs(a[j][k])>=big)
						{
							big=fabs(a[j][k]);
							irow=j;
							icol=k;
						}
					}
					else if(ipiv[k]>1) 
					{
						if(gerror("GAUSSJ: Singular Matrix-1")) return(0);
						else exit(1);
					}
				}
			}
		}
		++(ipiv[icol]);
		if(irow != icol)
		{
			for(l=0;l<n;l++) SWAP(a[irow][l],a[icol][l]) ;
			for(l=0;l<m;l++) SWAP(b[irow][l],b[icol][l]) ;
		}
		indxr[i] = irow;
		indxc[i] = icol;
		if(a[icol][icol] == 0.0)
			gerror("GAUSSJ: Singular Matrix-2");
		pivinv = 1.0 / a[icol][icol];
		a[icol][icol] = 1.0;
		for(l=0;l<n;l++) a[icol][l] *= pivinv;
		for(l=0;l<m;l++) b[icol][l] *= pivinv;
		for(ll=0;ll<n;ll++)
		{
			if(ll!=icol)
			{
				dum = a[ll][icol];
				a[ll][icol] = 0.0;
				for(l=0;l<n;l++) a[ll][l] -= a[icol][l]*dum;
				for(l=0;l<m;l++) b[ll][l] -= b[icol][l]*dum;
			}
		}
	}
	for(l=n-1;l>=0;l--)
	{
		if(indxr[l] != indxc[l])
			for (k=0;k<n;k++) SWAP(a[k][indxr[l]],a[k][indxc[l]]);
	}
	free(ipiv);
	free(indxr);
	free(indxc);
	return(n);
}

// a[k][j] = a(j,k) to use fortran source ; a[n][n] b[m][n]
int cgaussjordan(double **a,int n,double **b,int m)
{
	int *indxc, *indxr, *ipiv;
	int i, icol=0, irow=0, j, k, l, ll;
	double big, dum, pivinv;

	if(NULL==(indxc=(int *)calloc(n,sizeof(int)))) if(gerror("lack of memory")) return(0);
	if(NULL==(indxr=(int *)calloc(n,sizeof(int)))) if(gerror("lack of memory")) return(0);
	if(NULL==(ipiv =(int *)calloc(n,sizeof(int)))) if(gerror("lack of memory")) return(0);
	for(i=0;i<n;i++)
		ipiv[i]=0;
	for(i=0;i<n;i++)
	{
		big=0.0;
		for(j=0;j<n;j++)
		{
			if(ipiv[j]!=1)
			{
				for(k=0;k<n;k++)
				{
					if(ipiv[k]==0)
					{
						if(fabs(a[k][j])>=big)
						{
							big=fabs(a[k][j]);
							irow=j;
							icol=k;
						}
					}
					else if(ipiv[k]>1) 
					{
						if(gerror("GAUSSJORDAN: Singular Matrix-1")) return(0);
						else exit(1);
					}
				}
			}
		}
		++(ipiv[icol]);
		if(irow != icol)
		{
			for(l=0;l<n;l++) SWAP(a[l][irow],a[l][icol]) ;
			for(l=0;l<m;l++) SWAP(b[l][irow],b[l][icol]) ;
		}
		indxr[i] = irow;
		indxc[i] = icol;
		if(a[icol][icol] == 0.0)
		{
			if(gerror("GAUSSJORDAN: Singular Matrix-2")) return(0); 
			else exit(1);
		}
		pivinv = 1.0 / a[icol][icol];
		a[icol][icol] = 1.0;
		for(l=0;l<n;l++) a[l][icol] *= pivinv;
		for(l=0;l<m;l++) b[l][icol] *= pivinv;
		for(ll=0;ll<n;ll++)
		{
			if(ll!=icol)
			{
				dum = a[icol][ll];
				a[icol][ll] = 0.0;
				for(l=0;l<n;l++) a[l][ll] -= a[l][icol]*dum;
				for(l=0;l<m;l++) b[l][ll] -= b[l][icol]*dum;
			}
		}
	}
	for(l=n-1;l>=0;l--)
	{
		if(indxr[l] != indxc[l])
			for (k=0;k<n;k++) SWAP(a[indxr[l]][k],a[indxc[l]][k]);
	}
	free(ipiv);
	free(indxr);
	free(indxc);
	return(n);
}

#undef SWAP
