#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "errors.h"
#include "jacobi.h"

#define ROTATE(a,i,j,k,l,s,tau) {double g=a[i][j]; double h=a[k][l]; a[i][j]=g-s*(h+g*tau); a[k][l]=h+s*(g-h*tau);}

int jacobidiagonalize(int n, double **a, double *d, double **v, int *nrot)
{
	int j,iq,ip,i;
	double tresh,theta,tau,t,sm,s,h,g,c,*b,*z;
	if(NULL==(b=(double *)calloc(n,sizeof(double)))) uerror("jacobidiagonalize","lack of memory");
	if(NULL==(z=(double *)calloc(n,sizeof(double)))) uerror("jacobidiagonalize","lack of memory");
	for(ip=0;ip<n;ip++)
	{
		for(iq=0;iq<n;iq++)
			v[ip][iq]=0;
		v[ip][ip]=1;
	}
	for(ip=0;ip<n;ip++)
	{
		b[ip]=d[ip]=a[ip][ip];
		z[ip]=0;
	}
	*nrot=0;
	for(i=0;i<50;i++)
	{
		sm=0;
		for(ip=0;ip<n;ip++)
		{
			for(iq=ip+1;iq<n;iq++)
				sm += fabs(a[ip][iq]);
		}
		// if(1) fprintf(stderr," jacobi iteration = %d sm= %le\n",i,sm);
		if(sm == 0.0)
		{
			free(b);
			free(z);
			return(i);
		}
		if(i<4) 
			tresh=0.2*sm/(n*n);
		else
			tresh=0;
		for(ip=0;ip<n;ip++)
		{
			for(iq=ip+1;iq<n;iq++)
			{
				g=100*fabs(a[ip][iq]);
				if(i>4 && (fabs(d[ip])+g)==fabs(d[ip]) && (fabs(d[iq])+g)==fabs(d[iq]))
					a[ip][iq]=0;
				else if(fabs(a[ip][iq])>tresh)
				{
					h=d[iq]-d[ip];
					if((fabs(h)+g)==fabs(h))
						t=a[ip][iq]/h;
					else
					{
						theta=0.5*h/(a[ip][iq]);
						t=1/(fabs(theta)+sqrt(1+theta*theta));
						if(theta<0) t=-t;
					}
					c=1/sqrt(1+t*t);
					s=t*c;
					tau=s/(1+c);
					h=t*a[ip][iq];
					z[ip]-=h;
					z[iq]+=h;
					d[ip]-=h;
					d[iq]+=h;
					a[ip][iq]=0;
					for(j=0;j<ip;j++) ROTATE(a,j,ip,j,iq,s,tau);
					for(j=ip+1;j<iq;j++) ROTATE(a,ip,j,j,iq,s,tau);
					for(j=iq+1;j<n;j++) ROTATE(a,ip,j,iq,j,s,tau);
					for(j=0;j<n;j++) ROTATE(v,j,ip,j,iq,s,tau);
					++(*nrot);
				}
			}
		}
		for(ip=0;ip<n;ip++)
		{
			b[ip]+=z[ip];
			d[ip]=b[ip];
			z[ip]=0;
		}
	}
	uerror("jacobidiagonalize","Too many iterations in jacobidiagonalize");
	return(i);
}

int jacobisort(int n,double *d, double **v)
{
	int i,j,k;
	double p;
	for(i=0;i<n-1;i++)
	{
		p=d[k=i];
		for(j=i+1;j<n;j++)
			if(d[j]<=p) p=d[k=j];
		if(k!=i)
		{
			d[k]=d[i];
			d[i]=p;
			for(j=0;j<n;j++)
			{
				p=v[j][i];
				v[j][i]=v[j][k];
				v[j][k]=p;
			}
		}
	}
	return(n);
}

double jacobiinverse(int n,double **a,double *e,double **v,double c,int l)
{
	int i,j,k;
	double d=1,u,t=log(fabs(e[n-1]))-c*M_LN10;
	for(i=0;i<n;i++)
	{
		d*=e[i];
		if(fabs(e[i])<1E-128) e[i]=0;
		else if((u=log(fabs(e[i])))<t)
		{
			if(JACOBI_DEBUG)
				fprintf(stderr,"     too small eigenvalue %le by order of %lf at %d of %d\n"
				,e[i],(t+c*M_LN10-u)*M_LOG10E,i+1,n);
			e[i]=0/e[i];
		}
		else e[i]=1/e[i];
	}
	if(l)
	{
		for(i=0;i<n-l;i++) e[i]=0;
		for(i=n-l;i<n;i++) fprintf(stderr," i= %d e= %le\n",i+1,e[i]);
	}
	for(i=0;i<n;i++)
	{
		for(j=0;j<n;j++)
		{
			t=0;
			for(k=0;k<n;k++)
				t+=v[i][k]*e[k]*v[j][k];
			a[i][j]=t; 
		}
	}
	return(d);
}

int jacobisolve(double **a,int n,double **b,int m,double c,int l)
{
	int i,j,k;
	double t,*e,**v;
	if(NULL==(e=(double *)calloc(n,sizeof(double)))) uerror("jacobisolve","e");
	if(NULL==(v=(double **)calloc(n,sizeof(double*)))) uerror("jacobisolve","v");
	for(i=0;i<n;i++)
		if(NULL==(v[i]=(double *)calloc(n,sizeof(double)))) uerror("jacobisolve","v");

	jacobidiagonalize(n,a,e,v,&k);
	jacobisort(n,e,v);
	if(l) for(i=0;i<n;i++)
	{
		t=0;
		for(j=0;j<n;j++)
			t+=b[j][0]*v[j][i];
		fprintf(stderr," i= %d e=%+le g=%+le\n",i+1,e[i],t);
	}
	jacobiinverse(n,a,e,v,c,l);

	for(k=0;k<m;k++)
	{
		for(i=0;i<n;i++)
			e[i]=b[i][k];
		for(i=0;i<n;i++)
		{
			t=0;
			for(j=0;j<n;j++)
				t+=a[i][j]*e[j];
			b[i][k]=t;
		}
	}

	for(i=0;i<n;i++) free(v[i]);
	free(v);
	free(e);
	return(n);
}

// in C, matrix[column][row]

int cjacobidiagonalize(int n, double **a, double *d, double **v, int *nrot)
{
	int j,iq,ip,i;
	double tresh,theta,tau,t,sm,s,h,g,c,*b,*z;
	if(NULL==(b=(double *)calloc(n,sizeof(double)))) uerror("cjacobidiagonalize","lack of memory");
	if(NULL==(z=(double *)calloc(n,sizeof(double)))) uerror("cjacobidiagonalize","lack of memory");
	for(ip=0;ip<n;ip++)
	{
		for(iq=0;iq<n;iq++)
			v[iq][ip]=0;
		v[ip][ip]=1;
	}
	for(ip=0;ip<n;ip++)
	{
		b[ip]=d[ip]=a[ip][ip];
		z[ip]=0;
	}
	*nrot=0;
	for(i=0;i<50;i++)
	{
		sm=0;
		for(ip=0;ip<n;ip++)
		{
			for(iq=ip+1;iq<n;iq++)
				sm += fabs(a[iq][ip]);
		}
		// if(1) fprintf(stderr," jacobi iteration = %d sm= %le\n",i,sm);
		if(sm == 0.0)
		{
			free(b);
			free(z);
			return(i);
		}
		if(i<4) 
			tresh=0.2*sm/(n*n);
		else
			tresh=0;
		for(ip=0;ip<n;ip++)
		{
			for(iq=ip+1;iq<n;iq++)
			{
				g=100*fabs(a[iq][ip]);
				if(i>4 && (fabs(d[ip])+g)==fabs(d[ip]) && (fabs(d[iq])+g)==fabs(d[iq]))
					a[iq][ip]=0;
				else if(fabs(a[iq][ip])>tresh)
				{
					h=d[iq]-d[ip];
					if((fabs(h)+g)==fabs(h))
						t=a[iq][ip]/h;
					else
					{
						theta=0.5*h/(a[iq][ip]);
						t=1/(fabs(theta)+sqrt(1+theta*theta));
						if(theta<0) t=-t;
					}
					c=1/sqrt(1+t*t);
					s=t*c;
					tau=s/(1+c);
					h=t*a[iq][ip];
					z[ip]-=h;
					z[iq]+=h;
					d[ip]-=h;
					d[iq]+=h;
					a[iq][ip]=0;
					for(j=0;j<ip;j++) ROTATE(a,ip,j,iq,j,s,tau);
					for(j=ip+1;j<iq;j++) ROTATE(a,j,ip,iq,j,s,tau);
					for(j=iq+1;j<n;j++) ROTATE(a,j,ip,j,iq,s,tau);
					for(j=0;j<n;j++) ROTATE(v,ip,j,iq,j,s,tau);
					++(*nrot);
				}
			}
		}
		for(ip=0;ip<n;ip++)
		{
			b[ip]+=z[ip];
			d[ip]=b[ip];
			z[ip]=0;
		}
	}
	uerror("cjacobidiagonalize","Too many iterations in jacobidiagonalize");
	return(i);
}

int cjacobisort(int n,double *d, double **v)
{
	int i,j,k;
	double p;
	for(i=0;i<n-1;i++)
	{
		p=d[k=i];
		for(j=i+1;j<n;j++)
			if(d[j]<=p) p=d[k=j];
		if(k!=i)
		{
			d[k]=d[i];
			d[i]=p;
			for(j=0;j<n;j++)
			{
				p=v[i][j];
				v[i][j]=v[k][j];
				v[k][j]=p;
			}
		}
	}
	return(n);
}

double cjacobiinverse(int n,double **a,double *e,double **v,double c,int l)
{
	int i,j,k;
	double d=1,u,t=log(fabs(e[n-1]))-c*M_LN10;
	for(i=0;i<n;i++)
	{
		d*=e[i];
		if(fabs(e[i])<1E-128) e[i]=0;
		else if((u=log(fabs(e[i])))<t)
		{
			if(JACOBI_DEBUG) 
				fprintf(stderr,"     too small eigenvalue %le by order of %lf at %d of %d\n"
				,e[i],(t+c*M_LN10-u)*M_LOG10E,i+1,n);
			e[i]=0/e[i];
		}
		else e[i]=1/e[i];
	}
	if(l)
	{
		for(i=0;i<n-l;i++) e[i]=0;
		for(i=n-l;i<n;i++) fprintf(stderr," i= %d e= %le\n",i+1,e[i]);
	}
	for(i=0;i<n;i++)
	{
		for(j=0;j<n;j++)
		{
			t=0;
			for(k=0;k<n;k++)
				t+=v[k][i]*e[k]*v[k][j];
			a[j][i]=t; 
		}
	}
	return(d);
}

int cjacobisolve(double **a,int n,double **b,int m,double c,int l)
{
	int i,j,k;
	double t,*e,**v;
	if(NULL==(e=(double *)calloc(n,sizeof(double)))) uerror("cjacobisolve","e");
	if(NULL==(v=(double **)calloc(n,sizeof(double*)))) uerror("cjacobisolve","v");
	for(i=0;i<n;i++)
		if(NULL==(v[i]=(double *)calloc(n,sizeof(double)))) uerror("cjacobisolve","v");

	cjacobidiagonalize(n,a,e,v,&k);
	cjacobisort(n,e,v);
	if(l) for(i=0;i<n;i++)
	{
		t=0;
		for(j=0;j<n;j++)
			t+=b[0][j]*v[i][j];
		fprintf(stderr," i= %d e=%+le g=%+le",i+1,e[i],t);
		for(j=0;j<n;j++)
			fprintf(stderr," %+6.3lf",v[i][j]);
		fprintf(stderr,"\n");
	}
	cjacobiinverse(n,a,e,v,c,l);

	for(k=0;k<m;k++)
	{
		for(i=0;i<n;i++)
			e[i]=b[k][i];
		for(i=0;i<n;i++)
		{
			t=0;
			for(j=0;j<n;j++)
				t+=a[i][j]*e[j];
			b[k][i]=t;
		}
	}

	for(i=0;i<n;i++) free(v[i]);
	free(v);
	free(e);
	return(n);
}
