#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "programming.h"
#include "instrumental.h"
#include "status.h"

#include "tools.h"
#include "calculate.h"
#include "gaussjordan.h"
#include "jacobi.h"
#include "errors.h"
#include "leastsquare.h"


int calculatelinears(int detail,DIFFRACTION *data,MOLECULAR *molecule,FITTING *fit)
{
	int i,j,ng=0;
	int is=data->cs,ie=data->ce;
	int range=ie-is;
	int np=0,nb=0,na=0,ns=0,nfs=0,nf=0,nc=0;
	int jf[MAX_MIXTURE]={0};
	int jc[MAX_MIXTURE]={0};
	double **x,*y,*l,*m,*p,*q,**r,fsum=molecule->fsum,rsigma=1;

	if(!fit->linears) return(0);
	if(fit->background) nb=data->np;
	if(fit->atomicbackground) na=1;
	if(fit->fractions) nfs=1;
	else if(fit->scalefactor&&fsum) ns=1;
	for(i=0;i<molecule->n;i++) 
		if(fit->fraction[i]) jf[nf++]=i;
		else jc[nc++]=i;
	if(nc&&fit->scalefactor) ns=1; // to fix ratios
	np=nb+na+ns+nf;
	if(np<1) return(0);
	//if(nfs) no l;

	if(NULL==(x=(double **)calloc(np,sizeof(double *)))) uerror("calculatelinears","lack of memory");
	if(NULL==(y=(double *)calloc(range,sizeof(double)))) uerror("calculatelinears","lack of memory");

	if(NULL==(l=(double *)calloc(np,sizeof(double)))) uerror("calculatelinears","lack of memory");
	if(NULL==(m=(double *)calloc(np,sizeof(double)))) uerror("calculatelinears","lack of memory");
	if(NULL==(p=(double *)calloc(np,sizeof(double)))) uerror("calculatelinears","lack of memory");
	if(NULL==(q=(double *)calloc(np,sizeof(double)))) uerror("calculatelinears","lack of memory");
	if(NULL==(r=(double **)calloc(np,sizeof(double*)))) uerror("calculatelinears","lack of memory");

	if(nb)
	{
		for(i=0;i<nb;i++)
		{
			r[i]=&(data->pbc[i]);
			x[i]=data->ps[i]+is;
		}
	}
	else
	{
		for(i=0;i<data->np;i++)
			for(j=is;j<ie;j++)
				y[j-is]-=data->pbc[i]*data->ps[i][j];
	}
	if(na)
	{
		r[nb]=&(data->atomicbackground);
		x[nb]=data->sat[0]+is;
	}
	else
	{
		for(j=is;j<ie;j++)
			y[j-is]-=data->atomicbackground*data->sat[0][j];
	}
	if(ns)
	{
		r[nb+na]=&(data->scale);
		x[nb+na]=data->sme+is;
	}
	else
	{
		for(j=is;j<ie;j++)
			y[j-is]-=data->scale*data->sme[j];
	}

	if(nf)
	{
		for(i=0;i<nf;i++)
		{
			if(fit->coupled[0]==jf[i]) m[i+nb+na+ns]= 1;
			if(fit->coupled[1]==jf[i]) m[i+nb+na+ns]=-1;
			l[i+nb+na+ns]=1;
			r[i+nb+na+ns]=&(molecule->fraction[jf[i]]);
			x[i+nb+na+ns]=data->sam[jf[i]]+is;
		}
	}

	if(nc)
	{
		for(i=0;i<nc;i++)
		{
			fsum-=molecule->fraction[jc[i]];
			for(j=is;j<ie;j++)
				y[j-is]-=molecule->fraction[jc[i]]
					*data->sam[jc[i]][j];
		}
	}
	if(nfs)
		for(i=0;i<nf;i++)
			l[i+nb+na+ns]=0;
	
	for(i=0;i<np;i++)
		q[i]=*(r[i]);

	ng=weightedlagrangian2cleastsquare(detail,np,range,p,data->w+is,y,x,fsum,l,m);
	if(*(data->noise)>0)
	{
		makescaledata(data,molecule);
		rsigma=sqrt((data->ce-data->cs)/data->chisqr);
		if(detail) fprintf(stderr,"leastsquare: chisqr= %lf rsigma= %lf\n",data->chisqr,rsigma);
		for(j=is;j<ie;j++)
			data->n[j]=data->w[j]*(data->wg[j]=erfc(fabs(*(data->noise)*data->chi[j]*rsigma)));
		ng=weightedlagrangian2cleastsquare(detail,np,range,p,data->n+is,y,x,fsum,l,m);
	}

	if(ng)
	{
		for(i=0;i<np;i++)
			*(r[i])=p[i];
		// if(detail)
		// {
		// 	fprintf(stderr,"    for the time being, i will automatically update\n");
		// 	for(i=0;i<np;i++)
		// 		fprintf(stderr,"    result for r[%d]= %lf\n",i,p[i]);
		// }
	}
	else
	{
		fprintf(stderr,"    make sure you not fit scale factor and fraction simultaneously\n");
		fprintf(stderr,"    try to reduce the order of polynomial in background\n");
		fprintf(stderr,"    restoring original values\n");
		for(i=0;i<np;i++)
			*(r[i])=q[i];
	}
//	makescaledata(data,molecule);	// this will be done in singlepoint
//	if(detail) temporaryfile(range,np,p,y,x);

	free(r);
	free(q);
	free(p);
	free(m);
	free(l);
	free(y);
	free(x);

	return(np);
}

double weightedlagrangian2cleastsquare(int detail,int np,int nx,double *p,double *w,double *y,double **x,double fsum,double *l,double *m)
{
	int method=JACOBI_METHOD;
	int i,j,k,ng=0,ns=3,check=0;
	double **a,**b,*pc,wyj;
	double c1=0,d1=0,e1=0,c2=0,d2=0,e2=0,lambda=0,lambda2=0,d=0;
	double chisqr,sigma,error;
	
	if(NULL==(a=(double **)calloc(np,sizeof(double *)))) uerror("weightedlagrangian2cleastsquare","lack of memory");
	for(i=0;i<np;i++)
		if(NULL==(a[i]=(double *)calloc(np,sizeof(double)))) uerror("weightedlagrangian2cleastsquare","lack of memory");
	if(NULL==(b=(double **)calloc(ns,sizeof(double *)))) uerror("weightedlagrangian2cleastsquare","lack of memory");
	for(i=0;i<ns;i++)
		if(NULL==(b[i]=(double *)calloc(np,sizeof(double)))) uerror("weightedlagrangian2cleastsquare","lack of memory");
	if(NULL==(pc=(double *)calloc(np,sizeof(double)))) uerror("weightedlagrangian2cleastsquare","lack of memory");

	for(i=0;i<np;i++)
	{
		for(j=0;j<=i;j++)
		{
			a[i][j]=0;
			for(k=0;k<nx;k++)
				a[i][j]+=w[k]*x[i][k]*x[j][k];
			a[j][i]=a[i][j];
		}
	}
	//printmatrix(np,a);
	for(i=0;i<np;i++)
	{
		b[2][i]=m[i];
		b[1][i]=l[i];
		b[0][i]=0;
		for(k=0;k<nx;k++)
			b[0][i]+=w[k]*y[k]*x[i][k];
	}
	if(check) {
		fprintf(stderr,"hessian\n");
		for(i=0;i<np;i++) {
			for(j=0;j<np;j++)
				fprintf(stderr,"%+10le ",a[i][j]);
			fprintf(stderr,"\n");
		}
	}
	if(detail>DETAIL_SO_MUCH) diagnosis(np,a);
	switch(method)
	{
		default:
		case 1: ng=cgaussjordan(a,np,b,ns);
			break;
		case 2: ng=choleskydecompose(np,a,pc);
			for(i=0;i<ns;i++)
				choleskysolve(np,a,pc,b[i],b[i]);
			// if(check) printmatrix(np,a);
			// if(check) check_cholesky();
			break;
		case 3: ng=cjacobisolve(a,np,b,ns,JACOBI_LINEAR_SMALL,0);
			break;
	}
	if(ng)
	{
		c1=0;
		d1=0;
		e1=0;
		c2=0;
		d2=0;
		e2=0;
		for(i=0;i<np;i++)
		{
			c1+=b[0][i]*l[i];
			d1+=b[1][i]*l[i];
			e1+=b[2][i]*l[i];
			c2+=b[0][i]*m[i];
			d2+=b[1][i]*m[i];
			e2+=b[2][i]*m[i];
		}
		d=d1*e2-d2*e1;
		if(fabs(d)>TOLERANCE_ZERO) // or 10^-JACOBI_LINEAR_SMALL
		{
			if(detail) fprintf(stderr,"applying lagrangians with determinant = %lf\n",d);
			lambda = ((fsum-c1)*e2+c2*e1)/d;
			lambda2=-((fsum-c1)*d2+c2*d1)/d;
		}
		else
		{
			if(detail) fprintf(stderr,"determinant = %le\n",d);
			if(d1) lambda = (fsum-c1)/d1;
			else   lambda = 0;
			if(e2) lambda2= -c2/e2;
			else   lambda2= 0;
		}
		for(i=0;i<np;i++)
			p[i]=b[0][i]+lambda*b[1][i]+lambda2*b[2][i];

		if(detail)
		{
			if(2==method) 
			{
				choleskyinverselowertriangle(np,a,pc);
				choleskyinverse(np,a,pc);
			}
			//printmatrix(np,a);
			for(i=0;i<np;i++)
			{
				error=sqrt(fabs(a[i][i]));
				if(a[i][i]<0) error*=-1;
				fprintf(stderr,"#UED4# result for r[%d]= %+lf +/- (%lf x scalefactor)\n",i,p[i],error);
			}
			if(UED3)
			{
				fprintf(stderr,"\n");
				chisqr=0;
				for(k=0;k<nx;k++)
				{
					wyj=y[k];
					for(i=0;i<np;i++)
						wyj-=p[i]*x[i][k];
					chisqr+=wyj*wyj*w[k];
				}
				sigma=sqrt(chisqr/(nx-np));
				fprintf(stderr,"    X2= %lf x scalefactor^-2)\n",chisqr);
				for(i=0;i<np;i++)
				{
					error=sigma*sqrt(fabs(a[i][i]));
					if(a[i][i]<0) error*=-1;
					fprintf(stderr,"#UED3# result for r[%d]= %+lf +/- (%lf x scalefactor x scalefactor^-1)\n",i,p[i],error);
				}
				fprintf(stderr,"\n");
			}
		}
	}

	free(pc);
	for(i=0;i<ns;i++)
		free(b[i]);
	free(b);
	for(i=0;i<np;i++)
		free(a[i]);
	free(a);
	return(ng);
}

void temporaryfile(int range,int np,double *p,double *y,double **x)
{
	int i,j;
	FILE *fp;
	if(NULL==(fp=fopen("_temporary.dat","wt"))) exit(1);
	else
	{
		fprintf(fp,"%20.10lf ",1.0);
		for(i=0;i<np;i++)
			fprintf(fp,"%20.10lf ",p[i]);
		fprintf(fp,"\n");
		for(j=0;j<range;j++)
		{
			fprintf(fp,"%20.10lf ",y[j]);
			for(i=0;i<np;i++)
				fprintf(fp,"%20.10lf ",x[i][j]);
			fprintf(fp,"\n");
		}
	}
	fclose(fp);
	return;
}

int printmatrix(int n,double **a)
{
	int i,j;
	fprintf(stderr,"\n");
	for(i=0;i<n;i++)
	{
		for(j=0;j<n;j++)
			fprintf(stderr,"%15.5le",a[i][j]);
		fprintf(stderr,"\n");
	}
	fprintf(stderr,"\n");
	return(n);
}


int calculatedifferencefractions(int detail,DIFFRACTION *data,MOLECULAR *molecule,FITTING *fit)
{
	int i,j,ng=0;
	int is=data->cs,ie=data->ce;
	int range=ie-is;
	int np=0,nb=0,na=0,ns=0,nf=0,nc=0,nfs=0;
	int jf[MAX_MIXTURE]={0};
	int jc[MAX_MIXTURE]={0};
	double **x,*y,*l,*m,*p,*q,**r,fsum=molecule->fsum,rsigma=1;

	if(!fit->linears) return(0);
	if(fit->background) nb=data->np;
	if(fit->atomicbackground) na=1;
	for(i=1;i<molecule->n;i++) 
		if(fit->fraction[i]) jf[nf++]=i;
		else jc[nc++]=i;
	np=nb+na+ns+nf;
	if(np<1) return(0);
	//if(nfs) no l;

	if(NULL==(x=(double **)calloc(np,sizeof(double *)))) uerror("calculatedifferencefractions","lack of memory");
	for(i=0;i<np;i++)
		if(NULL==(x[i]=(double *)calloc(range,sizeof(double)))) uerror("calculatedifferencefractions","lack of memory");
	if(NULL==(y=(double *)calloc(range,sizeof(double)))) uerror("calculatedifferencefractions","lack of memory");

	if(NULL==(l=(double *)calloc(np,sizeof(double)))) uerror("calculatedifferencefractions","lack of memory");
	if(NULL==(m=(double *)calloc(np,sizeof(double)))) uerror("calculatedifferencefractions","lack of memory");
	if(NULL==(p=(double *)calloc(np,sizeof(double)))) uerror("calculatedifferencefractions","lack of memory");
	if(NULL==(q=(double *)calloc(np,sizeof(double)))) uerror("calculatedifferencefractions","lack of memory");
	if(NULL==(r=(double **)calloc(np,sizeof(double*)))) uerror("calculatedifferencefractions","lack of memory");

	if(nb)
	{
		for(i=0;i<nb;i++)
		{
			r[i]=&(data->pbc[i]);
			for(j=is;j<ie;j++)
				x[i][j-is]=data->ps[i][j];
		}
	}
	else
	{
		for(i=0;i<data->np;i++)
			for(j=is;j<ie;j++)
				y[j-is]-=data->pbc[i]*data->ps[i][j];
	}
	if(na)
	{
		r[nb]=&(data->atomicbackground);
		for(j=is;j<ie;j++)
			x[nb][j-is]=data->sat[0][j];
	}
	else
	{
		for(j=is;j<ie;j++)
			y[j-is]-=data->atomicbackground*data->sat[0][j];
	}
	if(ns)
	{
		r[nb+na]=&(data->scale);
		for(j=is;j<ie;j++)
			x[nb+na][j-is]=data->sme[j];
	}
	else
	{
		for(j=is;j<ie;j++)
			y[j-is]-=data->scale*data->sme[j];
	}

	if(nf)
	{
		for(i=0;i<nf;i++)
		{
			if(fit->coupled[0]==jf[i]) m[i+nb+na+ns]= 1;
			if(fit->coupled[1]==jf[i]) m[i+nb+na+ns]=-1;
			//l[i+nb+na+ns]=1;
			r[i+nb+na+ns]=&(molecule->fraction[jf[i]]);
			for(j=is;j<ie;j++)
				x[i+nb+na+ns][j-is]=(data->sam[jf[i]][j]-data->sam[0][j]);
		}
	}

	if(nc)
	{
		for(i=0;i<nc;i++)
		{
			fsum-=molecule->fraction[jc[i]];
			for(j=is;j<ie;j++)
				y[j-is]-=molecule->fraction[jc[i]]
					*(data->sam[jc[i]][j]-data->sam[0][j]);
		}
	}
	if(nfs)
	{
		r[nb+na+ns+nf]=&(molecule->fsum);
		for(j=is;j<ie;j++)
			x[nb+na+ns+nf][j-is]=data->sam[0][j];
	}
	else
	{
		for(j=is;j<ie;j++)
			y[j-is]-=molecule->fsum*data->sam[0][j];
	}
	
	for(i=0;i<np;i++)
		q[i]=*(r[i]);

	ng=weightedlagrangian2cleastsquare(detail,np,range,p,data->w+is,y,x,fsum,l,m);
	if(*(data->noise)>0)
	{
		makescaledata(data,molecule);
		rsigma=sqrt((data->ce-data->cs)/data->chisqr);
		if(detail) fprintf(stderr,"leastsquare: chisqr= %lf rsigma= %lf\n",data->chisqr,rsigma);
		for(j=is;j<ie;j++)
			data->n[j]=data->w[j]*(data->wg[j]=erfc(fabs(*(data->noise)*data->chi[j]*rsigma)));
		ng=weightedlagrangian2cleastsquare(detail,np,range,p,data->n+is,y,x,fsum,l,m);
	}

	if(ng)
	{
		for(i=0;i<np;i++)
			*(r[i])=p[i];
		if(1)
		{
			molecule->fraction[0]=molecule->fsum;
			for(i=1;i<molecule->n;i++)
				molecule->fraction[0]-=molecule->fraction[i];
		}
		// if(detail)
		// {
		// 	fprintf(stderr,"    for the time being, i will automatically update\n");
		// 	for(i=0;i<np;i++)
		// 		fprintf(stderr,"    result for r[%d]= %lf\n",i,p[i]);
		// }
	}
	else
	{
		fprintf(stderr,"    make sure you not fit scale factor and fraction simultaneously\n");
		fprintf(stderr,"    try to reduce the order of polynomial in background\n");
		fprintf(stderr,"    restoring original values\n");
		for(i=0;i<np;i++)
			*(r[i])=q[i];
	}
//	makescaledata(data,molecule);	// this will be done in singlepoint
//	if(detail) temporaryfile(range,np,p,y,x);

	free(r);
	free(q);
	free(p);
	free(m);
	free(l);
	free(y);
	free(x);

	return(np);
}

