#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "programming.h"
#include "instrumental.h"
#include "scanmode.h"

#include "calculate.h"
#include "evaluate.h"
#include "gaussjordan.h"
#include "leastsquare.h"
#include "numericalfit.h"
#include "outputs.h"
#include "errors.h"

#include "montecarlo.h"

double randomnumber()
{
	return ((double)(random())/RAND_MAX);
}

//	sample some random guesses
//	sort them

long preparesearch(int detail,int np,FITTING *fit)
{
	int j;
	int mode=0;
	double bins=1,bin=0.0025; bin=0.005;

	for(j=0;j<np;j++)
	{
		switch(fit->nonlinear[j])
		{
			case INDEX_FIT_CAMERA : 
				fit->range[j]=0.1;
				fit->step[j]=0.001;	// too small than 0.01
				fit->bin[j]=bin;
				break;
			case INDEX_FIT_TEMPERATURE :
				fit->range[j]=200;
				fit->step[j]=1.0;	// originally 10.0
				fit->bin[j]=10.0;
				break;
			case INDEX_FIT_COORDINATE:
				fit->range[j]=SCANMODE_R[mode];
				fit->step[j]=0.01;
				fit->bin[j]=bin;
				break;
			case INDEX_FIT_COORDINATE_BONDLENGTH :
				fit->range[j]=SCANMODE_R[mode];
				fit->step[j]=0.01;
				fit->bin[j]=bin;
				break;
			case INDEX_FIT_COORDINATE_BENDINGANGLE :
				fit->range[j]=SCANMODE_A[mode];
				fit->step[j]=0.5;
				fit->bin[j]=bin*180/PI;
				break;
			case INDEX_FIT_COORDINATE_DIHEDRAL :
				fit->range[j]=SCANMODE_D[mode];
				fit->step[j]=2.0;
				fit->bin[j]=bin*180/PI;
				break;
			case INDEX_FIT_POTENTIAL_BONDLENGTH :
				fit->range[j]=SCANMODE_R[mode];
				fit->step[j]=1.0;
				fit->bin[j]=10.0;
				break;
			case INDEX_FIT_POTENTIAL_BENDINGANGLE :
				fit->range[j]=SCANMODE_A[mode];
				fit->step[j]=1.0;
				fit->bin[j]=10.0;
				break;
			case INDEX_FIT_POTENTIAL_DIHEDRAL :
				fit->range[j]=SCANMODE_D[mode];
				fit->step[j]=1.0;
				fit->bin[j]=10.0;
				break;
			default :
				fprintf(stderr,"  %d is unknown type in monte carlo ....\n"
					,fit->nonlinear[j]);
				fit->range[j]=1;
				fit->step[j]=0.01;
				fit->bin[j]=bin;
				break;
		}
		if(UED3)
			bins*=50;	// =(1.0/rad_conv) where rad_conv=0.02;
		else
			bins*=fit->range[j]/fit->bin[j];
		if(detail)
			fprintf(stderr,"    %5d: range= %lf bin= %lf step= %lf\n"
					,j,fit->range[j],fit->bin[j],fit->step[j]);
	}
	if(UED3)
		fit->iteration=(long)sqrt(bins);
	else
		fit->iteration=(long)sqrt(2*bins);
	if(fit->iteration<100L*np*np) 
		fit->iteration=100L*np*np;
	if(fit->iteration<0) 
		fit->iteration=100L*np*np;
	fprintf(stderr," the number of iteration is %ld from %d variables\n",fit->iteration,np);
	return(fit->iteration);
}

int SearchMonteCarlo(int detail,int np,int mb,double *p,double **r,double *beta,double **alpha,double **b
		,double **dyda,double **d2yda2,double **covar,double *oneda,double *da
		,DIFFRACTION *data,MOLECULAR *molecule,int ne,ELEMENTAL *elemental,INSTRUMENTAL *instrument,FITTING *fit)
{
	int i,j,k,n=0,h=MAX_HISTORY;
	double dc,R,ra0[MAX_MIXTURE][MAX_NUM_BOND],ra1[MAX_MIXTURE][MAX_NUM_BOND];
	int *g;
	double *jacobian,**x,*y,*d;

	if(detail)
	{
		fprintf(stderr,"    Monte Carlo Iteration = %ld\n",fit->iteration);
	}

	switch(fit->group)
	{
		default: h=(fit->iteration/20); if(h<50) h=50; break;
		case 1 : h=(int)sqrt(fit->iteration); break;
	}
	if(NULL==(jacobian=(double *)calloc(np,sizeof(double)))) uerror("SearchMonteCarlo","lack of memory");
	if(NULL==(g=(int *)calloc(h,sizeof(int)))) uerror("SearchMonteCarlo","lack of memory");
	if(NULL==(y=(double *)calloc(h,sizeof(double)))) uerror("SearchMonteCarlo","lack of memory");
	if(NULL==(x=(double **)calloc(h,sizeof(double *)))) uerror("SearchMonteCarlo","lack of memory");
	for(j=0;j<h;j++)
		if(NULL==(x[j]=(double *)calloc(np,sizeof(double)))) uerror("SearchMonteCarlo","lack of memory");

	differentiate2(np,STEPSCALE,r,dyda,d2yda2,beta,alpha,data,molecule,ne,elemental,instrument,fit);
	savera0(ra0,molecule);
	for(j=0;j<h;j++)
	{
		g[j]=1;
		y[j]=data->chisqr;
	}
	for(j=0;j<np;j++)
	{
		x[0][j]=*(r[j]);
		jacobian[j]=sqrt(fabs(alpha[j][j]));
	}
	if(detail)
	{
		fprintf(stderr,"    Jacobian: ");
		for(j=0;j<np;j++)
			fprintf(stderr,"%lf ",jacobian[j]);
		fprintf(stderr,"\n");
	}

	n=montecarlo(detail,np,p,r,h,g,y,x,jacobian,data,molecule,ne,elemental,instrument,fit);

	if(NULL==(d=(double *)calloc(n,sizeof(double)))) uerror("SearchMonteCarlo","lack of memory");
	for(j=0;j<np;j++)
		*(r[j])=x[0][j];
	singlepoint(0,0,data,molecule,ne,elemental,instrument,fit);
	savera0(ra1,molecule);
	dc=convergencedistance(detail,ra1,n,np,r,g,x,d,data,molecule,ne,elemental,instrument,fit);
	fprintf(stderr,"  DistanceConvergence= %lf from %d configurations\n",dc,n);
	for(i=0;i<n;i++)
	{
		fprintf(stderr,"    %5d: X2= %lf d= %lf(%d): ",1+i,y[i],d[i],g[i]);
		for(j=0;j<np;j++)
			fprintf(stderr,"%lf ",x[i][j]);
		fprintf(stderr,"\n");
	}
	convergencedistance(detail,ra0,n,np,r,g,x,d,data,molecule,ne,elemental,instrument,fit);
	fprintf(stdout,"  --------------------------------------");
	for(k=0;k<np;k++) fprintf(stdout,"----------"); fprintf(stdout,"\n");
	fprintf(stdout,"      -- Monte Carlo Results ---------------");
	for(k=0;k<np;k++) fprintf(stdout,"--------"); fprintf(stdout,"\n");
	fprintf(stdout,"  --------------------------------------");
	for(k=0;k<np;k++) fprintf(stdout,"----------"); fprintf(stdout,"\n");
	for(i=0;i<n;i++)
	{
		for(j=0;j<np;j++)
			*(r[j])=x[i][j];
		singlepoint(0,0,data,molecule,ne,elemental,instrument,fit);
		R=evaluateR(0,data,molecule);
		fprintf(stdout,"    %5d: R= %lf X2= %lf d= %lf(%d): ",1+i,R,y[i],d[i],g[i]);
		for(j=0;j<np;j++)
			fprintf(stdout,"%lf ",x[i][j]);
		if(d[i]<dc) fprintf(stdout, " <=== try me");
		fprintf(stdout,"\n");
	}
	fprintf(stdout,"  --------------------------------------");
	for(k=0;k<np;k++) fprintf(stdout,"----------"); fprintf(stdout,"\n");

	for(j=0;j<np;j++)
		p[j] = x[0][j];
	for(j=0;j<np;j++)
		*(r[j]) = p[j];
	singlepoint(0,0,data,molecule,ne,elemental,instrument,fit);

	free(d);
	for(j=0;j<h;j++)
		free(x[j]);
	free(x);
	free(y);
	free(g);
	free(jacobian);
	CalculateError(detail);
	return(i);
}

int montecarlo(int detail,int np,double *p,double **r,int h,int *g,double *y,double **x,double *jacobian
		,DIFFRACTION *data,MOLECULAR *molecule,int ne,ELEMENTAL *elemental,INSTRUMENTAL *instrument,FITTING *fit)
{
	int i,j,n=1;
	double *q;
	if(NULL==(q=(double *)calloc(np,sizeof(double)))) uerror("montecarlo","lack of memory");

	for(j=0;j<np;j++)
		p[j] = *(r[j]);
	for(i=0;i<fit->iteration;i++)
	{
		for(j=0;j<np;j++) 
		{
			q[j] = p[j] + fit->range[j]*2*(randomnumber()-0.5);
			*(r[j]) = q[j];
		}
		singlepoint(0,0,data,molecule,ne,elemental,instrument,fit);
		switch(fit->group)
		{
			default: insertxyvector(i,fit->iteration,&n,h,np,data->chisqr,q,y,x); break;
			case 1 : sortxyvector(i,fit->iteration,&n,h,np,data->chisqr,q,g,y,x,jacobian); break;
		}
		if(0)
		{
			fprintf(stderr,"    %8d, %12.8lf : ",i,data->chisqr);
			for(j=0;j<np;j++)
				fprintf(stderr,"%12.8lf ",*(r[j]));
			fprintf(stderr,"\n");
		}
	}
	free(q);
	return(n);
}

int savera0(double ra0[][MAX_NUM_BOND],MOLECULAR *molecule)
{
	MLS *mlsp=molecule->mls;
	int i,j,n=0;
	for(i=0;i<molecule->n;i++)
		for(j=0;j<mlsp[i].nl;j++)
		{
			ra0[i][j]=mlsp[i].ra[j];
			n++;
		}
	return(n);
}

double convergencedistance(int detail,double ra0[][MAX_NUM_BOND],int n,int np,double **r,int *g,double **x,double *d
		,DIFFRACTION *data,MOLECULAR *molecule,int ne,ELEMENTAL *elemental,INSTRUMENTAL *instrument,FITTING *fit)
{
	int i,j;
	double w,h=0,l=0;

	for(i=0;i<n;i++)
	{
		for(j=0;j<np;j++)
			*(r[j])=x[i][j];
		singlepoint(0,0,data,molecule,ne,elemental,instrument,fit);
		d[i]=advancedmlsdistance(ra0,molecule,elemental);
		w=g[i]/(data->chisqr*data->chisqr);
		l+=d[i]*w;
		h+=w;
	}
	return(l/h);
}

double advancedmlsdistance(double ra0[][MAX_NUM_BOND],MOLECULAR *molecule,ELEMENTAL *elemental)
{
	MLS *mlsp=molecule->mls;
	int i,j;
	double dr,r=0;
	for(i=0;i<molecule->n;i++)
		for(j=0;j<mlsp[i].nl;j++)
		{
			dr=(1/mlsp[i].ra[j]-1/ra0[i][j]);
			r+=dr*dr*mlsp[i].d[j]*elemental[mlsp[i].et1[j]].charge*elemental[mlsp[i].et1[j]].charge;
		}
	r=sqrt(r);
	return(r);
}

double mlsdistance(double ra0[][MAX_NUM_BOND],MOLECULAR *molecule)
{
	MLS *mlsp=molecule->mls;
	int i,j;
	double dr,r=0;
	if(UED3)
	{
		for(i=0;i<molecule->n;i++)
			for(j=0;j<mlsp[i].nl;j++)
			{
				dr=mlsp[i].ra[j]-ra0[i][j];
				r+=fabs(dr);
			}
	}
	else
	{
		for(i=0;i<molecule->n;i++)
			for(j=0;j<mlsp[i].nl;j++)
			{
				dr=mlsp[i].ra[j]-ra0[i][j];
				r+=dr*dr;
			}
		r=sqrt(r);
	}
	return(r);
}

double jacobiandistance(int n,double *q2,double *q1,double *j)
{
	int i;
	double rj,r2=0,j2=0;
	for(i=0;i<n;i++)
	{
		rj=(q2[i]-q1[i])*j[i];
		r2+=rj*rj;
		j2+=j[i]*j[i];
	}
	return(sqrt(r2/j2));
}

int copyxyvector(int np,double y1,double *x1,double *y2,double *x2)
{
	int i;
	*y2=y1;
	for(i=0;i<np;i++)
		x2[i]=x1[i];
	return(np);
}

int getprocess(int j,int m,int i,int k,int np,double chisqr,double *r,double d)
{
	int n;
	fprintf(stderr,"    [%d/%d] X2(%d<%d)= %lf r()= ",j,m,i,k,chisqr);
	for(n=0;n<np;n++)
		fprintf(stderr,"%lf ",r[n]);
	fprintf(stderr,"d= %lf\n",d);
	return(j);
}

int sortxyvector(int j,int m,int *n,int h,int np,double chisqr,double *r,int *g,double *y,double **x,double *jacobian)
{
	int i,k,l,p=*n;
	double d=0,d0=RADIUS;	// *sqrt(np);
	double dm=1.0;

	if(chisqr<y[*n-1]) 
	{
		for(p=0;p<*n-1;p++)
			if(chisqr<y[p]) break;
		for(i=p;i<*n;i++)
		{
			d=jacobiandistance(np,r,x[i],jacobian);
			if(d<dm) dm=d;
			if(d<d0)
			{
				// fprintf(stderr,"    X2(%d)= %lf r()= ",i,y[i]);
				// for(k=0;k<np;k++)
				//	fprintf(stderr,"%lf ",x[i][k]);
				for(l=i;l>p;l--)
				{
					copyxyvector(np,y[l-1],x[l-1],y+l,x[l]);
					g[l]=g[l-1];
				}
				copyxyvector(np,chisqr,r,y+p,x[p]);
				g[p]++;
				getprocess(j,m,p,i,np,chisqr,r,d);
				return(p);
			}
		}

		if(*n<h) (*n)++;
		for(k=(*n)-1;k>p;k--)
		{
			copyxyvector(np,y[k-1],x[k-1],y+k,x[k]);
			g[k]=g[k-1];
		}
		copyxyvector(np,chisqr,r,y+p,x[p]);
		g[p]=1;
		getprocess(j,m,p,p,np,chisqr,r,dm);
		return(p);
	}
	else if (FILLLIST)
	{
		if(*n<h)
		{
			copyxyvector(np,chisqr,r,y+*n,x[*n]);
			g[p]=1;
			getprocess(j,m,p,(*n)++,np,chisqr,r,dm);
		}
	}
	return(0);
}

int insertxyvector(int j,int m,int *n,int h,int np,double chisqr,double *r,double *y,double **x)
{
	int k,p=*n;

	if(chisqr<y[*n-1]) 
	{
		for(p=0;p<*n-1;p++)
			if(chisqr<y[p]) break;
		if(*n<h) (*n)++;
		for(k=(*n)-1;k>p;k--)
			copyxyvector(np,y[k-1],x[k-1],y+k,x[k]);
		copyxyvector(np,chisqr,r,y+p,x[p]);
		return(p);
	}
	else 
	{
		if(*n<h)
			copyxyvector(np,chisqr,r,y+*n,x[(*n)++]);
	}
	return(*n);
}

