#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "instrumental.h"
#include "programming.h"
#include "status.h"

#include "calculate.h"
#include "evaluate.h"
#include "setstructure.h"
#include "leastsquare.h"
#include "fitmarquardt.h"
#include "fitjacobieigen.h"
#include "montecarlo.h"
#include "outputs.h"
#include "errors.h"

#include "numericalfit.h"

double evaluatebeta(int np,double step,double **r,double *beta
		,DIFFRACTION *data,MOLECULAR *molecule,int ne,ELEMENTAL *elemental,INSTRUMENTAL *instrument,FITTING *fit)
{
	int i;
	double chi2,q;
	for(i=0;i<np;i++)
	{
		q=*(r[i]);

		*(r[i])=q-fit->step[i];
		chi2=singlepoint(0,0,data,molecule,ne,elemental,instrument,fit);

		*(r[i])=q+fit->step[i];
		chi2-=singlepoint(0,0,data,molecule,ne,elemental,instrument,fit);

		*(r[i])=q;
		beta[i]=0.25*chi2/fit->step[i];	// another 0.5 from definition of beta
	}
	chi2=singlepoint(0,0,data,molecule,ne,elemental,instrument,fit);
	return(chi2);
}

int evaluatedyda(int np,double step,double **r,double *beta,double **dyda
		,DIFFRACTION *data,MOLECULAR *molecule,int ne,ELEMENTAL *elemental,INSTRUMENTAL *instrument,FITTING *fit)
{
	int i,is,i0=data->cs,i1=data->ce;
	double chi2,q,divider;
	double chiw;
	static double dwda[MAX_NUM_DATA];

	for(i=0;i<np;i++)
	{
		q=*(r[i]);
		divider=0.5/fit->step[i];

		*(r[i])=q-fit->step[i];
		chi2=singlepoint(0,0,data,molecule,ne,elemental,instrument,fit);
		for(is=i0;is<i1;is++)
			dyda[i][is-i0]=data->chi[is]*divider; // smc is not needed unless camera, but

		*(r[i])=q+fit->step[i];
		chi2-=singlepoint(0,0,data,molecule,ne,elemental,instrument,fit);
		for(is=i0;is<i1;is++)
			dyda[i][is-i0]-=data->chi[is]*divider;

		*(r[i])=q;
		beta[i]=0.5*chi2*divider;
	}
	if(INDEX_FIT_CAMERA==fit->nonlinear[0])	// camera only can be 0th nonlinear parameter
	{
		q=*(r[0]);
		divider=0.5/fit->step[0];

		*(r[0])=q-fit->step[0];
		chi2=singlepoint(0,0,data,molecule,ne,elemental,instrument,fit);
		for(is=i0;is<i1;is++)
			dwda[is]=data->w[is];

		*(r[0])=q+fit->step[0];
		chi2=singlepoint(0,0,data,molecule,ne,elemental,instrument,fit);
		for(is=i0;is<i1;is++)
			dwda[is]-=data->w[is];

		*(r[0])=q;
		chi2=singlepoint(0,0,data,molecule,ne,elemental,instrument,fit);
		for(is=i0;is<i1;is++)
		{
			chiw=0.5*data->chi[is]/data->w[is];
			dyda[0][is-i0]+=dwda[is]*chiw*divider;
		}
	}
	else
		chi2=singlepoint(0,0,data,molecule,ne,elemental,instrument,fit);
	return(np);
}

double evaluatealpha(int np,double **alpha,double **dyda,DIFFRACTION *data)
{
	int i,j,k,i0=data->cs,range=data->ce-i0;

	for(i=0;i<np;i++)
		for(j=0;j<=i;j++)
			alpha[i][j]=0;

	for(i=0;i<np;i++) 
	{
		for(j=0;j<=i;j++) 
		{
			for(k=0;k<range;k++)
				alpha[i][j]+=data->w[i0+k]*dyda[i][k]*dyda[j][k];
			alpha[j][i]=alpha[i][j];
		}
	}

	return (data->chisqr);
}

double evaluatealphabeta(int np,double *beta,double **alpha,double **dyda,DIFFRACTION *data)
{
	int is,i,j,i0=data->cs,i1=data->ce;
	double wdyda;

	for(i=0;i<np;i++)
	{
		beta[i]=0;
		for(j=0;j<=i;j++)	// for(j=0;j<np;j++)
			alpha[i][j]=0;
	}

	for(i=0;i<np;i++) 
	{
		for(is=i0;is<i1;is++)
		{
			wdyda=data->w[is]*dyda[i][is-i0];
			beta[i]+=data->chi[is]*wdyda;
			for(j=0;j<=i;j++) 
				alpha[i][j]+=wdyda*dyda[j][is-i0];
		}
	}

	for(i=0;i<np;i++)
		for(j=0;j<i;j++)
			alpha[j][i]=alpha[i][j];

	return (data->chisqr);
}

int differentiate(int np,double step,double **r,double **dyda,double *beta,double **alpha
		,DIFFRACTION *data,MOLECULAR *molecule,int ne,ELEMENTAL *elemental,INSTRUMENTAL *instrument,FITTING *fit)
{
	evaluatedyda(np,STEPSCALE,r,beta,dyda,data,molecule,ne,elemental,instrument,fit);
	if(NUMERICALBETA)
		evaluatealpha(np,alpha,dyda,data);
		// evaluatebeta(np,STEPSCALE,r,beta,data,molecule,ne,elemental,instrument,fit);
	else
		evaluatealphabeta(np,beta,alpha,dyda,data);
	return(NUMERICALBETA);
}

int evaluated2yda2(int np,double step,double **r,double *beta,double **dyda,double **d2yda2
		,DIFFRACTION *data,MOLECULAR *molecule,int ne,ELEMENTAL *elemental,INSTRUMENTAL *instrument,FITTING *fit)
{
	int i,is,i0=data->cs,i1=data->ce;
	double chi2,q,divider,divider2;
	double chiw;
	static double dwda[MAX_NUM_DATA];

	//fprintf(stderr," d2yda2 update= %d\n",molecule->preset[0]);
	for(i=0;i<np;i++)
	{
		q=*(r[i]);
		divider=0.5/fit->step[i];
		divider2=1/(fit->step[i]*fit->step[i]);

		*(r[i])=q-fit->step[i];
		chi2=singlepoint(0,0,data,molecule,ne,elemental,instrument,fit);
		for(is=i0;is<i1;is++)
		{
			dyda  [i][is-i0] =data->chi[is]*divider; // smc is not needed unless camera, but
			d2yda2[i][is-i0] =data->chi[is]*divider2;
		}

		*(r[i])=q+fit->step[i];
		chi2-=singlepoint(0,0,data,molecule,ne,elemental,instrument,fit);
		for(is=i0;is<i1;is++) 
		{
			dyda  [i][is-i0]-=data->chi[is]*divider;
			d2yda2[i][is-i0]+=data->chi[is]*divider2;
		}

		*(r[i])=q;
		beta[i]=0.5*chi2*divider;
	}
	chi2=singlepoint(0,0,data,molecule,ne,elemental,instrument,fit);
	for(i=0;i<np;i++)
	{
		divider2=1/(fit->step[i]*fit->step[i]);
		for(is=i0;is<i1;is++)
			d2yda2[i][is-i0]-=2*data->chi[is]*divider2;
	}
	if(INDEX_FIT_CAMERA==fit->nonlinear[0])	// camera only can be 0th nonlinear parameter
	{
		q=*(r[0]);
		divider=0.5/fit->step[0];

		*(r[0])=q-fit->step[0];
		chi2=singlepoint(0,0,data,molecule,ne,elemental,instrument,fit);
		for(is=i0;is<i1;is++)
			dwda[is]=data->w[is];

		*(r[0])=q+fit->step[0];
		chi2=singlepoint(0,0,data,molecule,ne,elemental,instrument,fit);
		for(is=i0;is<i1;is++)
			dwda[is]-=data->w[is];

		*(r[0])=q;
		chi2=singlepoint(0,0,data,molecule,ne,elemental,instrument,fit);
		for(is=i0;is<i1;is++)
		{
			chiw=0.5*data->chi[is]/data->w[is];
			dyda[0][is-i0]+=dwda[is]*chiw*divider;
		}
	}
	return(np);
}

double evaluatealpha2(int np,double **alpha,double **dyda,double **d2yda2,DIFFRACTION *data)
{
	int i,j,k,i0=data->cs,range=data->ce-i0;
	double alpha2;

	for(i=0;i<np;i++)
		for(j=0;j<=i;j++)
			alpha[i][j]=0;

	for(i=0;i<np;i++) 
	{
		for(j=0;j<=i;j++) 
		{
			for(k=0;k<range;k++)
				alpha[i][j]+=data->w[i0+k]*dyda[i][k]*dyda[j][k];
			alpha[j][i]=alpha[i][j];
		}
		alpha2=0;
		for(k=0;k<range;k++)
			alpha2+=data->w[i0+k]*data->chi[i0+k]*d2yda2[i][k];
		alpha[i][i]+=fabs(alpha2);
	}

	return (data->chisqr);
}

double evaluatealpha2beta(int np,double *beta,double **alpha,double **dyda,double **d2yda2,DIFFRACTION *data)
{
	int is,i,j,k,i0=data->cs,i1=data->ce;
	double wdyda,alpha2;

	for(i=0;i<np;i++)
	{
		beta[i]=0;
		for(j=0;j<=i;j++)	// for(j=0;j<np;j++)
			alpha[i][j]=0;
	}

	for(i=0;i<np;i++) 
	{
		for(is=i0;is<i1;is++)
		{
			k=is-i0;
			wdyda=data->w[is]*dyda[i][k];
			beta[i]+=data->chi[is]*wdyda;
			for(j=0;j<=i;j++) 
				alpha[i][j]+=wdyda*dyda[j][k];
		}
		alpha2=0;
		for(is=i0;is<i1;is++)
			alpha2+=data->w[is]*data->chi[is]*d2yda2[i][is-i0];
		alpha[i][i]+=fabs(alpha2);
	}

	for(i=0;i<np;i++)
		for(j=0;j<i;j++)
			alpha[j][i]=alpha[i][j];

	return (data->chisqr);
}

int differentiate2(int np,double step,double **r,double **dyda,double** d2yda2,double *beta,double **alpha
                ,DIFFRACTION *data,MOLECULAR *molecule,int ne,ELEMENTAL *elemental,INSTRUMENTAL *instrument,FITTING *fit)
{
	if(fit->d2yda2)
	{
		evaluated2yda2(np,STEPSCALE,r,beta,dyda,d2yda2,data,molecule,ne,elemental,instrument,fit);
		if(NUMERICALBETA)
			evaluatealpha2(np,alpha,dyda,d2yda2,data);
		else
			evaluatealpha2beta(np,beta,alpha,dyda,d2yda2,data);
	}
	else
	{
		evaluatedyda(np,STEPSCALE,r,beta,dyda,data,molecule,ne,elemental,instrument,fit);
		if(NUMERICALBETA)
			evaluatealpha(np,alpha,dyda,data);
		else
			evaluatealphabeta(np,beta,alpha,dyda,data);
	}
	return(NUMERICALBETA);
}

int calculatenonlinears(int detail,DIFFRACTION *data,MOLECULAR *molecule,int ne,ELEMENTAL *elemental,INSTRUMENTAL *instrument,FITTING *fit)
{
	int i,j,ng=0,update=0;
	int is=data->cs,ie=data->ce;
	int range=ie-is,mb=1;
	int np=0,nc=0,nt=0,nr=0,nz=0,nv=0;
	int jf[MAX_MIXTURE]={0};
	int jz[MAX_MIXTURE*MAX_NUM_ZMAT]={-1};
	int iz[MAX_MIXTURE*MAX_NUM_ZMAT]={-1};
	int jv[MAX_MIXTURE*MAX_NUM_ZMAT]={-1};
	int iv[MAX_MIXTURE*MAX_NUM_ZMAT]={-1};
	int jr[MAX_MIXTURE*MAX_NUM_ZMAT]={-1};
	int ir[MAX_MIXTURE*MAX_NUM_ZMAT]={-1};
	double chi2=data->chisqr;
	double *p,*q,**r,**s,*beta,**alpha,**dyda,*oneda,**covar,**b,*da,**d2yda2=NULL;

	if(fit->camera) nc=1;
	if(fit->cameravariance) nc++;
	for(i=0;i<molecule->n;i++) 
		if(fit->temperature[i]) jf[nt++]=i;
	if(fit->coordinates) 
	{
		for(i=0;i<molecule->n;i++)
			if(INDEX_RIC_SUPPLIED==molecule->ric[i])
			{
				for(j=0;j<molecule->h[i].nr;j++)
					if(fit->internal[i][j]) { jr[nr]=j; ir[nr++]=i; }
			}
			else
			{
				for(j=0;j<molecule->zmx[i].nz;j++)
				{
					if(fit->coordinate[i][j]) { jz[nz]=j; iz[nz++]=i; }
					if(fit->potential[i][j])  { jv[nv]=j; iv[nv++]=i; }
				}
			}
	}
	
	np=nc+nt+nz+nv+nr;
	if(np<1) return(0);

	if(NULL==(p=(double *)calloc(np,sizeof(double)))) uerror("calculatenonlinears","lack of memory");
	if(NULL==(q=(double *)calloc(np,sizeof(double)))) uerror("calculatenonlinears","lack of memory");
	if(NULL==(r=(double **)calloc(np,sizeof(double*)))) uerror("calculatenonlinears","lack of memory");
	if(NULL==(s=(double **)calloc(np,sizeof(double*)))) uerror("calculatenonlinears","lack of memory");
	if(NULL==(beta=(double *)calloc(np,sizeof(double)))) uerror("calculatenonlinears","lack of memory");
	if(NULL==(alpha=(double **)calloc(np,sizeof(double*)))) uerror("calculatenonlinears","lack of memory");
	for(i=0;i<np;i++)
		if(NULL==(alpha[i]=(double *)calloc(np,sizeof(double)))) uerror("calculatenonlinears","lack of memory");
	if(NULL==(dyda=(double **)calloc(np,sizeof(double*)))) uerror("calculatenonlinears","lack of memory");
	for(i=0;i<np;i++)
		if(NULL==(dyda[i]=(double *)calloc(range,sizeof(double)))) uerror("calculatenonlinears","lack of memory");
	if(fit->d2yda2)
	{
		if(NULL==(d2yda2=(double **)calloc(np,sizeof(double*)))) uerror("calculatenonlinears","lack of memory");
		for(i=0;i<np;i++)
			if(NULL==(d2yda2[i]=(double *)calloc(range,sizeof(double)))) uerror("calculatenonlinears","lack of memory");
	}
	if(NULL==(oneda=(double *)calloc(np,sizeof(double)))) uerror("calculatenonlinears","lack of memory");
	if(NULL==(covar=(double **)calloc(np,sizeof(double*)))) uerror("calculatenonlinears","lack of memory");
	for(i=0;i<np;i++)
		if(NULL==(covar[i]=(double *)calloc(np,sizeof(double)))) uerror("calculatenonlinears","lack of memory");
	if(NULL==(b=(double **)calloc(mb,sizeof(double*)))) uerror("calculatenonlinears","lack of memory");
	for(i=0;i<mb;i++)
		if(NULL==(b[i]=(double *)calloc(np,sizeof(double)))) uerror("calculatenonlinears","lack of memory");
	if(NULL==(da=(double *)calloc(np,sizeof(double)))) uerror("calculatenonlinears","lack of memory");

	if(nc)
	{
		if(fit->camera)
		{
			fit->nonlinear[0]=INDEX_FIT_CAMERA;
			s[0]=r[0]=&(instrument->CameraLength);
			data->preset=STATUS_NEED_TO_MAKE_ALL;
		}
		if(fit->cameravariance)
		{
			fit->nonlinear[nc-1]=INDEX_FIT_CAMERAVARIANCE;
			s[nc-1]=r[nc-1]=&(instrument->CameraLengthVariance);
			data->preset=STATUS_NEED_TO_MAKE_ALL;
		}
	}
	if(nt)
	{
		for(i=0;i<nt;i++)
		{
			fit->nonlinear[i+nc]=INDEX_FIT_TEMPERATURE;
			s[i+nc]=r[i+nc]=&(molecule->temperature[jf[i]]);
			setpreset(&(molecule->preset[jf[i]]),STATUS_ZMX_NEED_TO_MAKE_AMPLITUDE);
		}
		setpreset(&(data->preset),STATUS_NEED_TO_MAKE_MOLECULAR);
	}
	if(nz)
	{
		for(i=0;i<nz;i++)
		{
			fit->nonlinear[i+nc+nt]=INDEX_FIT_COORDINATE+molecule->zmx[iz[i]].zt[jz[i]];
			s[i+nc+nt]=r[i+nc+nt]=&(molecule->zmx[iz[i]].r[jz[i]]);
			setpreset(&(molecule->preset[iz[i]]),STATUS_ZMX_NEED_TO_MAKE_ALL);
		}
		setpreset(&(data->preset),STATUS_NEED_TO_MAKE_MOLECULAR);
	}
	if(nv)
	{
		for(i=0;i<nv;i++)
		{
			fit->nonlinear[i+nc+nt+nz]=INDEX_FIT_POTENTIAL+molecule->zmx[iv[i]].zt[jv[i]];
			s[i+nc+nt+nz]=r[i+nc+nt+nz]=&(molecule->zmx[iv[i]].v0[jv[i]]);
			setpreset(&(molecule->preset[iv[i]]),STATUS_ZMX_NEED_TO_MAKE_ALL);
		}
		setpreset(&(data->preset),STATUS_NEED_TO_MAKE_MOLECULAR);
	}
	if(nr)
	{
		for(i=0;i<nr;i++)
		{
			//fit->nonlinear[i+nc+nt+nz+nv]=INDEX_FIT_COORDINATE+molecule->h[ir[i]].z[jr[i]].type;
			fit->nonlinear[i+nc+nt+nz+nv]=INDEX_FIT_COORDINATE;
			r[i+nc+nt+nz+nv]=&(molecule->h[ir[i]].r[jr[i]]);
			s[i+nc+nt+nz+nv]=&(molecule->h[ir[i]].z[jr[i]].r);
			setpreset(&(molecule->preset[ir[i]]),STATUS_ZMX_NEED_TO_MAKE_ALL);
		}
		setpreset(&(data->preset),STATUS_NEED_TO_MAKE_MOLECULAR);
	}
	preparesearch(detail,np,fit);
	for(i=0;i<np;i++) q[i]=*(r[i]);

	switch(fit->method)
	{
		default:
		case INDEX_FIT_MARQUARDT:
			ng=FitMarquardt(detail,np,mb,p,r,beta,alpha,b,dyda,d2yda2,covar,oneda,da
					,data,molecule,ne,elemental,instrument,fit);
			break;
		case INDEX_FIT_JACOBIEIGEN:
			ng=FitJacobiEigen(detail,np,mb,p,r,beta,alpha,b,dyda,d2yda2,covar,oneda,da
					,data,molecule,ne,elemental,instrument,fit);
			break;
		case INDEX_FIT_MONTECARLO:
		case INDEX_FIT_SEARCH:
			ng=SearchMonteCarlo(detail,np,mb,p,r,beta,alpha,b,dyda,d2yda2,covar,oneda,da
					,data,molecule,ne,elemental,instrument,fit);
			break;
	}

	if(ng&&(data->chisqr<chi2))
	{
		if(fit->update)
			update=1;
		else
		{
			if(detail)
			{
				fprintf(stderr,"      X2= %20.12lf <-- %20.12lf\n",data->chisqr,chi2);
				fprintf(stderr,"    -------------------------------------------------------\n");
				for(i=0;i<np;i++)
					fprintf(stderr,"    r[%d]= %20.12lf = %20.12lf <-- %20.12lf\n",i,*s[i],p[i],q[i]);
				fprintf(stderr,"    -------------------------------------------------------\n");
				updatesmscurves(0,CURRENTSMSFILENAME,data);
			}
			update=updateresult();
		}
	}
	else
	{
		fprintf(stderr,"    fitting failed: ng= %d, chi^2+= %lf\n",ng,data->chisqr-chi2);
	}
	if(update)
	{
		fprintf(stderr,"    result updated ....\n");
		for(i=0;i<np;i++)
			fprintf(stderr,"    r[%d]= %20.12lf = %20.12lf <-- %20.12lf\n",i,*s[i],p[i],q[i]);
		for(i=0;i<np;i++) *(r[i])=p[i];
		// for(i=0;i<molecule->n;i++)
			// if(INDEX_RIC_SUPPLIED==molecule->ric[i]) updatecoordinate(molecule->h+i);
	}
	else
	{
		fprintf(stderr,"    nonlinear fit failed or discarded ....\n");
		for(i=0;i<np;i++) *(r[i])=q[i];
	}

	singlepoint(0,1,data,molecule,ne,elemental,instrument,fit);

	if(detail)
	{
		fprintf(stderr,"      X2= %10lf <-- %10lf\n",data->chisqr,chi2);
		fprintf(stderr,"    -----------------------------------\n");
		for(i=0;i<np;i++)
			fprintf(stderr,"    r[%d]= %10lf <-- %10lf\n",i,*(r[i]),q[i]);
		fprintf(stderr,"    -----------------------------------\n");
		updatesmscurves(0,CURRENTSMSFILENAME,data);
	}

	free(da);
	for(i=0;i<mb;i++)
		free(b[i]);
	free(b);
	free(oneda);
	if(fit->d2yda2)
	{
		for(i=0;i<np;i++)
			free(d2yda2[i]);
		free(d2yda2);
	}
	for(i=0;i<np;i++)
		free(dyda[i]);
	free(dyda);
	for(i=0;i<np;i++)
		free(alpha[i]);
	free(alpha);
	free(beta);
	free(s);
	free(r);
	free(q);
	free(p);

	return(np);
}

int updateresult()
{
	char *ef;
	char buffer[STRINGBUFFERSIZE];
	char input[STRINGBUFFERSIZE];
	int update=0;

	fprintf(stderr,"    update ? ");
	if(NULL==(ef=fgets(buffer,STRINGBUFFERSIZE,stdin))) 
		sprintf(input,"no");
	else
		sscanf(buffer,"%s",input);
	if((!strcasecmp("yes",input))||(!strcasecmp("y",input)))
		update=1;
	return(update);
}
