#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include "programming.h"
#include "instrumental.h"
#include "status.h"
#include "errors.h"

#include "evaluate.h"
#include "empirical.h"
#include "readpredata.h"
#include "readzmatrix.h"
#include "outputs.h"
#include "profile.h"
#include "jacobi.h"

#include "readdata.h"

int findnucleartype(char *s,int ne,ELEMENTAL *elemental)
{
	int i,it=-1;
	char name[NUCLEICSYMBOLLENGTH] = {0};
	sscanf(s,"%[A-Za-z]",name);
	for(i=0;i<ne;i++)
	{
		if(!strcmp(name,elemental[i].symbol)) 
		{
		//	fprintf(stdout,"%s is %s at %d\n",s,name,i);
			it=i;
			break;
		}
	}
		
	if(it==-1)
	{
		fprintf(stderr,"unknown nuclear type for %s\n",s);
		exit(1);
	}
	else if(DEBUG) fprintf(stderr,"%s is %dth element\n",name,it);
	return(it);
}


// This function reads the experimental data from a *.dat file 
// containing 3 columns holding the radius (in pixel), the experimental intensity, 
// and the experimental standard deviation 

int readdata(int detail,DIFFRACTION *data, char *fn)
{
	int i,j,n=0;
	FILE *fp;
	char fnx[STRINGBUFFERSIZE];
	char buffer[STRINGBUFFERSIZE];
	double m,d;

	setpreset(&(data->preset),STATUS_NEED_TO_MAKE_S_DATA);  //NOT CLEAR
	for(i=0;i<MAX_NUM_DATA;i++)				// initialize the data arrays to zero
	{
		data->imx[i]=0;
		data->snx[i]=0;
	}
	sprintf(fnx,"%s.dat",fn);				// read the filename
	if(NULL==(fp=fopen(fnx,"rt"))) ioerror(fnx);		
	else
	{
		data->ss=1;					// to avoid division by zero  
		for(i=0;i<MAX_NUM_DATA;i++)	
		{
			if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) break;
			if(1>(n=sscanf(buffer,"%d%lf%lf",&j,&m,&d))) break;
			data->imx[j]=m;				// assign the experimental intensity 
			data->snx[j]=d;				// assign the experimental standard deviation
			// if(detail) fprintf(stderr,"j= %d m= %lf d= %lf\n",j,m,d);

			//if(0==i) data->ss=j;
			//if((0==data->ss)&&(0!=data->imx[j])) data->ss=j;
		}
		fclose(fp);
		data->se=j;					// final data point read
		n=i;						// number of data points read
		sprintf(data->filename,"%s",fn);		// assign the filename of the data file
	}

	if(detail) fprintf(stderr,"    read experiment data for %s of %d points (%d-%d)\n",fn,n,data->ss,data->se);
	return(n);
}


// This function read the experimental data from a *.xl0 file and a *.snr file
// each containing 2 columns holding 
// the radius (in pixel), the experimental intensity (*.xl0) 
// the radius (in pixel), the experimental standard deviation (*.snr)

int readexpdata(int detail,DIFFRACTION *data, char *fn)
{
	int i,j,n=0;
	FILE *fp;
	char fnx[STRINGBUFFERSIZE];
	char fnr[STRINGBUFFERSIZE];
	char buffer[STRINGBUFFERSIZE];
	double d;

	setpreset(&(data->preset),STATUS_NEED_TO_MAKE_S_DATA);
	for(i=0;i<MAX_NUM_DATA;i++)
	{
		data->imx[i]=0;
		data->snx[i]=0;
	}
	sprintf(fnx,"%s.xl0",fn);				// read the filename containing the experimental intensity
	sprintf(fnr,"%s.snr",fn);				// read the filename containing the experimental standard deviation
	if(NULL==(fp=fopen(fnx,"rt"))) ioerror(fnx);
	else
	{
		data->ss=1;					// to avoid division by zero
		for(i=0;i<MAX_NUM_DATA;i++)
		{
			if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) break;
			if(1>(n=sscanf(buffer,"%d,%lf",&j,&d))) break;
			data->imx[j]=d;
			// if(detail) fprintf(stderr,"j= %d d= %lf\n",j,d);

			//if(0==i) data->ss=j;
			//if((0==data->ss)&&(0!=data->imx[j])) data->ss=j;
		}
		fclose(fp);
		data->se=j;
		n=i;
		sprintf(data->filename,"%s",fn);
	}

	if(NULL==(fp=fopen(fnr,"rt"))) ioerror(fnr);
	else
	{
		for(i=0;i<MAX_NUM_DATA;i++)
		{
			if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) break;
			if(1>sscanf(buffer,"%d,%lf",&j,&d)) break;
			// if(detail) fprintf(stderr,"j= %d d= %lf\n",j,d);
			data->snx[j]=d;
		}
		fclose(fp);
	}
	if(detail) fprintf(stderr,"    read experiment data for %s of %d and %d points (%d-%d)\n",fn,n,i,data->ss,data->se);
	return(n);
}

int resolveconstitution(MLS *mls,char *atoms,int ne,ELEMENTAL *elemental)
{
	int i=0,j=0,k=0,l=0,nn=0,na=0,n=strlen(atoms);
	char *pos=atoms,buffer[STRINGBUFFERSIZE]="";
	for(i=0;i<n;i++)
	{
		if(0<(l=sscanf(atoms+i,"%[A-Za-z]",buffer)))
			j = findnucleartype(buffer,ne,elemental);
		else
			j = -1;
		if(0<j)
		{
			i+=l;
			if(0<(l=sscanf(atoms+i,"%[0-9]",buffer)))
			{	nn = atoi(buffer); i+=(l-1); }
			else
				nn = 1;
			mls->et0[na] = j;
			mls->ne0[na] = nn;
			na++;
		//	fprintf(stderr,"%d: %s %d\n",na,elemental[j].symbol,nn);
		}
	}
	mls->na=na;
	return(na);
}

// mls file contains re and lh from diffraction results

int readmlsdata(int detail,MOLECULAR *molecule,int ne,ELEMENTAL *elemental,char *fn)
{
	MLS *mls=&(molecule->mls[molecule->i]);
	int i,ni;
	FILE *fp;
	char *ef;
	char buffer[STRINGBUFFERSIZE];
	char atoms[STRINGBUFFERSIZE];
	char fnm[STRINGBUFFERSIZE];

	mls->nl=0;
//	if(data->preset>3) data->preset=3;
	sprintf(fnm,"%s.mls",fn);
	if(NULL==(fp=fopen(fnm,"rt"))) ioerror(fnm);
	else
	{
		if(NULL==(ef=fgets(buffer,STRINGBUFFERSIZE,fp))) ioerror(fnm);
		sscanf(buffer,"%d",&ni);
		if(NULL==(ef=fgets(buffer,STRINGBUFFERSIZE,fp))) ioerror(fnm);
		sscanf(buffer,"%s",mls->molecule);
		if(NULL==(ef=fgets(buffer,STRINGBUFFERSIZE,fp))) ioerror(fnm);
		sscanf(buffer,"%d%s\n",&(mls->nl),atoms);
		resolveconstitution(mls,atoms,ne,elemental);
		for(i=0;i<(mls->nl);i++)
		{
			if(NULL==(ef=fgets(buffer,STRINGBUFFERSIZE,fp))) ioerror(fnm);
			if (UED3&&0)
			{
				sscanf(buffer,"%s%s%d%lf%lf%lf\n",mls->a1[i],mls->a2[i],&(mls->d[i])
						,&(mls->re[i]),&(mls->lh[i]),&(mls->a[i]));
			}
			else
			{
				int j;
				double lh2=0;
				sscanf(buffer,"%s%s%d%lf%lf%lf\n",mls->a1[i],mls->a2[i],&(mls->d[i])
					,&(mls->ra[i]),&(mls->lm[i]),&(mls->a[i]));	// doesn't seem right
				mls->lh[i]=mls->lm[i];
				lh2=mls->lh[i]*mls->lh[i];
				// ra = rg-lh2/re = re + 1.5 a lh2 - lh2 / re
				// re = ra - 1.5 a lh2 + lh2 / re
				mls->re[i]=mls->ra[i];
				for(j=0;j<16;j++)
				{
					mls->re[i]=mls->ra[i]-1.5*mls->a[i]*lh2+lh2/mls->re[i];
					// fprintf(stderr,"     j= %d : re= %20.15lf, ra= %lf\n",j,mls->re[i],mls->ra[i]);
				}
			}

			mls->et1[i]=findnucleartype(mls->a1[i],ne,elemental);
			mls->et2[i]=findnucleartype(mls->a2[i],ne,elemental);
			mls->mu[i]=reducedmasset(elemental[mls->et1[i]].charge,elemental[mls->et2[i]].charge);
			mls->nu[i]=estimatefrequency(detail,mls->mu[i],mls->lh[i],300.0);
		}
		mls->temperature=300.0;
		fclose(fp);
		if(detail)
		{
			for(i=0;i<(mls->na);i++)
				fprintf(stderr,"    %s %d ",elemental[mls->et0[i]].symbol,mls->ne0[i]);
			fprintf(stderr,"\n");
			for(i=0;i<(mls->nl);i++)
				fprintf(stderr,"%5d: %s-%s (%d) re= %lf (lh= %lf) a= %lf <-- ra= %lf\n"
					,1+i,mls->a1[i],mls->a2[i],mls->d[i],mls->re[i],mls->lh[i],mls->a[i],mls->ra[i]);
		}
		if(molecule->n==molecule->i) 
		{
			molecule->n++;	// seems redundant, check ueda.c
			if(detail) fprintf(stderr,"    added mls data for %s of %d points\n",fn,mls->nl);
		}
		else
			if(detail) fprintf(stderr,"    changed mls data for %s of %d points\n",fn,mls->nl);
		molecule->kind[molecule->i]=INDEX_KIND_MLS;
		molecule->preset[molecule->i]=STATUS_ZMX_NEED_TO_MAKE_AMPLITUDE;
		//molecule->temperature[molecule->i]=ROOM_TEMPERATURE;
	}
	return(mls->na+mls->nl);
}

double estimate_lh_lm_a(int detail,double lm,double a)
{
	int i;
	double a2=a*a;
	double lm2=lm*lm;
	double lh2=lm2-1.5*a2*lm2;
	for(i=0;i<8;i++)
	{
		lh2=lm2-1.5*a2*lh2*lh2;
		if(detail) fprintf(stderr,"%5d lh= %le lm= %le a=%le\n",i,sqrt(lh2),sqrt(lm2),a);
	}
	return(sqrt(lh2));
}

#define SMALLDIFFERENCE 1.0E-8

double estimate_a_drg_lm(int detail,double drg,double lm)
{
	int i;
	double a=0,lm2=lm*lm;
	double lh2=lm2-1.5*a*a*lm2*lm2;
	double previous=a;
	for(i=0;i<16;i++)
	{
		lh2=lm2-1.5*a*a*lh2*lh2;
		a=(drg-2.25*a*a*a*lh2*lh2*lh2)/(3*lh2);
		//if(detail) fprintf(stderr,"%5d a= %le lh2= %le lm2=%le drg=%le\n",i,a,(lh2),(lm2),drg);
		if(detail) fprintf(stderr,"%5d a= %le lh= %le lm=%le drg=%le\n",i,a,sqrt(lh2),sqrt(lm2),drg);
		if(fabs(a-previous)<SMALLDIFFERENCE) break;
		previous=a;
	}
	return(a);
}

double evaluate_lh_lm_a(int detail,double lm,double a)
{
	// lm^2 = lh^2 + 1.5 a^2 lh^4
	// 0 = -lm2 + lh2 + 1.5 a2 lh2^2
	double lh=lm*M_SQRT2/sqrt(1+sqrt(1+6*a*a*lm*lm));
	if(detail) fprintf(stderr,"%5d lh= %le lm= %le a=%le\n",-1,lh,lm,a);
	return(lh);
}

int readampdata(int detail,MOLECULAR *molecule,int ne,ELEMENTAL *elemental,char *fn)
{
	MLS *mls=&(molecule->mls[molecule->i]);
	ZMAT *z=&(molecule->zmx[molecule->i]);
	ATOMIC *x=&(molecule->atomic[molecule->i]);
	int i,j,k,m,n=0;
	FILE *fp;
	char fna[STRINGBUFFERSIZE];
	char buffer[STRINGBUFFERSIZE];
	char a1[STRINGBUFFERSIZE];
	char a2[STRINGBUFFERSIZE];
	char tmp1[STRINGBUFFERSIZE];
	char tmp2[STRINGBUFFERSIZE];
	char tmp3[STRINGBUFFERSIZE];
	double re=0,la=0,lh=0,drg=0,a=1;

	int *et=NULL,na=0;
	char *atom = NULL;
	switch(molecule->kind[molecule->i])
	{
	case INDEX_KIND_ZMX:
		et=z->et;
		na=z->na;
		atom=z->atom;
		break;
	case INDEX_KIND_XYZ:
		et=x->et;
		na=x->na;
		atom=x->atom;
		break;
	default:
		return(0);
	}

	sprintf(fna,"%s.amp",fn);
	if(NULL==(fp=fopen(fna,"rt"))) ioerror(fna);
	else
	{
		if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) ioerror(fna);
		if(1>(m=sscanf(buffer,"%d%lf",&n,&(mls->temperature)))) { fclose(fp); return(0); }
		if(2>m) mls->temperature = molecule->temperature[molecule->i];
		fprintf(stderr," reading amplitudes of %d distances at %lf K\n",n,mls->temperature);
		for(m=0;m<mls->nl;m++)
		{
			//if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) ioerror(fna);
			if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) break;
			//sscanf(buffer,"%s%s",a1,a2);
			sscanf(buffer,"%s%s%s%lf%s%lf%s%lf",a1,a2,tmp1,&re,tmp2,&drg,tmp3,&la);
			i=findzreference(a1,atom,na);
			j=findzreference(a2,atom,na);
			if(i>j) { k=j; j=i; i=k; }
			k=mindex(i,j);
			//if(fabs(re-mls->re[k])>0.5)
			if(fabs(re-mls->re[k])>1.0)	// needed to adjust this criterion to allow large amplitude motions in Tryptophan
			{
				//fprintf(stderr,"%d (%d,%d) %lf %lf: %s",k,i,j,re,mls->re[k],buffer);
				ioerror("amplitude doesn't match zmatrix");
			}
			else
			{
				mls->d[k]=1;
				mls->et1[k]=et[i];
				mls->et2[k]=et[j];
				strcpy(mls->a1[k],elemental[et[i]].symbol);
				strcpy(mls->a2[k],elemental[et[j]].symbol);
				//mls->re[k]=re;
				a=estimate_a_drg_lm(0,drg,la);
				// lh=estimate_lh_lm_a(0,la,a);
				lh=evaluate_lh_lm_a(0,la,a);
				if(!finite(lh))
				{
					fprintf(stderr,"LH= %lf:",lh);
					a=estimate_a_drg_lm(1,drg,la);
					// lh=estimate_lh_lm_a(1,la,a);
					lh=evaluate_lh_lm_a(1,la,a);
				}
				mls->a[k]=a;
				mls->lh[k]=lh;
				mls->lm[k]=la;
				mls->rg[k]=mls->re[k]+drg;
				mls->ra[k]=mls->rg[k]-lh*lh/mls->re[k];;
				mls->mu[k]=reducedmasset(elemental[et[i]].charge,elemental[et[j]].charge);
				mls->T[k]=mls->temperature;
				mls->nu[k]=estimatefrequency(0,mls->mu[k],mls->lh[k],mls->temperature);
			}
			fprintf(stderr," %s %s re= %lf drg= %+lf lm= %lf | lh= %lf a= %+lf | k= %d (%d,%d) me= %lf\n"
				,a1,a2,re,drg,la,lh,a,k,i,j,mls->re[k]);
		}
		fclose(fp);
		if(detail)
		{
			for(i=0;i<(mls->na);i++)
				fprintf(stderr,"    %-2s %d ",elemental[mls->et0[i]].symbol,mls->ne0[i]);
			fprintf(stderr,"\n");
			for(i=0;i<(mls->nl);i++)
			{
				fprintf(stderr,"%5d: %s-%s (%d) ra= %lf lm= %lf a= %+lf  "
					,1+i,mls->a1[i],mls->a2[i],mls->d[i], mls->ra[i],mls->lm[i],mls->a[i]);
				fprintf(stderr,"drg= %+lf rg= %lf re= %lf lh= %lf\n"
					,mls->rg[i]-mls->re[i],mls->rg[i],mls->re[i],mls->lh[i]);
			}
		}
		molecule->amp[molecule->i]=INDEX_MLS_AMPLITUDE;
		molecule->preset[molecule->i]=STATUS_ZMX_NEED_TO_MAKE_AMPLITUDE;
	}
	return(mls->nl);
}


// This function initializes / defines all the parameter for the fitting routines 
// (some of these can be manually changed during the programs operation)

int initialize(DIFFRACTION *data,MOLECULAR *mole,INSTRUMENTAL *inst,FITTING *fit)
{
	int i,j;
	fprintf(stderr,"you gotta work on initialize\n");
	data->ss=0;
	data->se=MAX_NUM_DATA;
	data->cs=DEFAULT_SPIX_START;
	data->ce=DEFAULT_SPIX_END;
	data->np=DEFAULT_NUM_POLY;
	data->scale=1;
	data->preset=0;					// WHAT DOES THIS DO
	data->atomicbackground=0;
	inst->polynomials=RECIPROCAL;
	strcpy(inst->ReferenceAtom,"Xe");		// defines the reference atom to be Xe
	inst->Reference=DIVIDEDBYREFNUCLEI;
	inst->sMs=LEVELING;
	strcpy(inst->fi,"C");
	strcpy(inst->fj,"C");
	inst->ElectronBeamSize=ELECTRONBEAMSIZE;	
	inst->KineticEnergy=ElectronKineticEnergy;	
	inst->CameraLength=CameraDistance;		
	inst->CameraLengthVariance=0;
	inst->PixelSize=PIXELSIZE;
	inst->OverallTemperature=1;
	inst->dampingk=DAMPINGPARAMETER;
	inst->noise=0;
	data->noise=&(inst->noise);
	inst->e.width=3;
	buildeprofile(&(inst->e));
	mole->n=0;
	mole->i=-1;
	mole->j=0;
	mole->fsum=1;
	for(i=0;i<MAX_MIXTURE;i++)			// for each species, define parameters for structural definition 
	{
		mole->kind[i]=INDEX_KIND_MLS;
		mole->amp[i]=INDEX_MLS_EMPIRICAL;
		mole->ric[i]=INDEX_RIC_NONE;
		mole->preset[i]=0;
		mole->fraction[i]=0;
		mole->temperature[i]=ROOM_TEMPERATURE; 
		mole->mls[i].limit=100;
		mole->h[i].na=0;
		mole->h[i].nr=0;
		mole->h[i].bi=NULL;
	}
	mole->fraction[0]=1;				// set the fraction of the first species to one


	// enable or disable specific fitting parameters	
	
	fit->camera=0;
	fit->background=1;
	fit->atomicbackground=0;
	fit->scalefactor=1;
	fit->fractions=0;
	for(i=0;i<MAX_MIXTURE;i++)
	{
		fit->fraction[i]=1;
		fit->temperature[i]=0;
		for(j=0;j<MAX_NUM_ZMAT;j++)
			fit->coordinate[i][j]=0;
	}
	fit->linears=1;
	fit->nonlinears=0;
	fit->coordinates=0;
	fit->method=INDEX_FIT_MARQUARDT;
	fit->d2yda2=1;
	fit->update=0;
	fit->group=1;
	fit->alpha=10;
	fit->iteration=100L;
	fit->coupled[0]=-1;
	fit->coupled[1]=-1;

	
	// set parameters for the Jacobian fitting routine
	
	JACOBI_DEBUG=1;
	JACOBI_METHOD=2;
	JACOBI_LINEAR_SMALL=10;
	JACOBI_NONLINEAR_SMALL=10;
	JACOBI_NONLINEARS=0;
	return(0);
}

int readconfig(DIFFRACTION *data,MOLECULAR *mole,INSTRUMENTAL *inst,FITTING *fit)
{
	fprintf(stderr,"you gotta work on readconfig\n");
	return(0);
}
