#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "programming.h"
#include "instrumental.h"
#include "calculate.h"
#include "mathematical.h"
#include "errors.h"

#include "readpredata.h"


// This function reads the real inelastic and the complex elastic scattering factors from
// the files Si.dat and fn.dat and assigns the reading to the stucture array ELEMENTAL.

int readelements(int detail,ELEMENTAL *elemental)
{
	int i,j,k;
	FILE *fp;
	char buffer[STRINGBUFFERSIZE];
	char name[STRINGBUFFERSIZE];
	char *fnfn=ELEMENTfnFILENAME;
	char *fnsi=ELEMENTsiFILENAME;
	double t[1+2*SCATTERINGENERGIES];

	if(NULL==(fp=fopen(fnsi,"rt"))) ioerror(fnsi);				// open Si.dat
	else
	{
		for(i=0;i<MAX_NUCLEI;i++)		
		{
			for(j=0,k=0;j<INELASTICSCATTERINGPREDATA;j++)
			{
				if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) break;		
				if(SCATTERINGENERGIES>sscanf(buffer,"%s%d%lf%lf",name,&k,t,t+1)) 
					uerror("readelements",fnsi);
				else
					sscanf(buffer,"%s%d%lf%lf"	
					,elemental[k].symbol,&elemental[k].charge,&elemental[k].si[j],&elemental[k].S[j]);
			}
			if(detail>1) fprintf(stderr,"read inelastic factor for %s\n",name);
			for(j=0;j<SCATTERINGENERGIES;j++)					// the inelastic scattering factors are assumed independent on electron kinetic energy
				elemental[k].E[j]=elemental[0].E[j];
		}
		fclose(fp);
	}
	if(NULL==(fp=fopen(fnfn,"rt"))) ioerror(fnfn);				// open fn.dat
	else
	{
		if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) ioerror(fnsi);
		if(SCATTERINGENERGIES>sscanf(buffer,"%lf%lf%lf%lf",&elemental[0].E[0],&elemental[0].E[1],&elemental[0].E[2],&elemental[0].E[3])) // assign the scattering energies
			uerror("readelements","KE");
		else
			if(detail)
				for(j=0;j<SCATTERINGENERGIES;j++)
					fprintf(stderr,"read scattering factor for %lf keV\n",elemental[0].E[j]);
		if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) ioerror(fnsi);		// read a blank line
		for(i=0;i<MAX_NUCLEI;i++)
		{
			for(j=0,k=0;j<ELASTICSCATTERINGPREDATA;j++)
			{
				if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) break;
				if(11>sscanf(buffer,"%s%d%lf%lf%lf%lf%lf%lf%lf%lf%lf"
					,name,&k,t,t+1,t+2,t+3,t+4,t+5,t+6,t+7,t+8)) 
					uerror("readelements",fnfn);
				else		// read the elastic scattering amplitudes and phases at different kinetic energies
					sscanf(buffer,"%s%d%lf%lf%lf%lf%lf%lf%lf%lf%lf"
					,elemental[k].symbol,&elemental[k].charge,&elemental[k].se[j]
					,&elemental[k].f[0][j],&elemental[k].n[0][j]
					,&elemental[k].f[1][j],&elemental[k].n[1][j]
					,&elemental[k].f[2][j],&elemental[k].n[2][j]
					,&elemental[k].f[3][j],&elemental[k].n[3][j]);		
			}
			if(detail>1) fprintf(stderr,"read elastic factors for %s\n",name);
		}
		fclose(fp);
	}
	fprintf(stderr,"    read calibration data for %d elements\n",i);
	return(i);
}


// This function prints the contents of the structure array elemental

int confirmelements(int ne,ELEMENTAL *elemental)
{
	int i;
	for(i=0;i<ne;i++)
	{
		fprintf(stderr,"%10d %-5s %d %lf\n",i
				,elemental[i].symbol
				,elemental[i].charge
				,elemental[i].radius);			// this element of the structure is not assigned
	}
	return(ne);
}

int estimateradius(int ne,ELEMENTAL *elemental)
{
	int i;
	for(i=0;i<ne;i++)
	{
		switch(elemental[i].charge)
		{
			case  1: elemental[i].radius=0.40; break;
			case  6: elemental[i].radius=0.80; break;
			case  7: elemental[i].radius=0.80; break;
			case  8: elemental[i].radius=0.80; break;
			case  9: elemental[i].radius=0.80; break;
			case 15: elemental[i].radius=1.00; break;
			case 16: elemental[i].radius=1.00; break;
			case 17: elemental[i].radius=1.00; break;
			case 18: elemental[i].radius=1.00; break;
			case 25: elemental[i].radius=1.00; break;
			case 26: elemental[i].radius=1.00; break;
			case 35: elemental[i].radius=1.10; break;
			case 53: elemental[i].radius=1.40; break;
			case 54: elemental[i].radius=1.40; break;
			default: elemental[i].radius=1.50;
		}
	}
	return(ne);
}

int interpolateenergy(double Ex,ELEMENTAL *elemental)
{
	int i,j,k;
	double E0=0;
	double l0=0;
	double lx=1/waveconstant(Ex);
	double l[SCATTERINGENERGIES];
	double E[SCATTERINGENERGIES];
	double v[SCATTERINGENERGIES];
	double p[SCATTERINGENERGIES];
	int np=SCATTERINGENERGIES;
	for(k=0;k<SCATTERINGENERGIES;k++)
	{
		E[k] = elemental[0].E[k] - E0;
		l[k] = 1/waveconstant(E[k]) - l0;
	}
	fprintf(stderr,"interpolating scattering factors at %lf keV\n",Ex);
	for(i=0;i<MAX_NUCLEI;i++)
	{
		for(j=0;j<ELASTICSCATTERINGPREDATA;j++)
		{
			for(k=0;k<SCATTERINGENERGIES;k++)
				v[k]=elemental[i].f[k][j];
			polynomialfit(SCATTERINGENERGIES,v,E,np,p);
			elemental[i].fe[j]=polynomialvalue(Ex-E0,np,p);
			elemental[i].logfe[j]=FSTFN(polynomialvalue(Ex-E0,np,p));
			for(k=0;k<SCATTERINGENERGIES;k++)
				v[k]=elemental[i].n[k][j];
			polynomialfit(SCATTERINGENERGIES,v,l,np,p);
			elemental[i].ne[j]=polynomialvalue(lx-l0,np,p);
		}
		uniformnaturalcubicspline(ELASTICSCATTERINGPREDATA,elemental[i].fe,elemental[i].f2);
		uniformnaturalcubicspline(ELASTICSCATTERINGPREDATA,elemental[i].ne,elemental[i].n2);
		uniformnaturalcubicspline(INELASTICSCATTERINGPREDATA,elemental[i].S,elemental[i].S2);
	}
	return i;
}

void printinterpolated(int i,ELEMENTAL *elemental)
{
	int j;
	for(j=0;j<ELASTICSCATTERINGPREDATA;j++)
		fprintf(stdout,"%16.10lf %16.10lf %16.10lf\n",elemental[i].se[j],elemental[i].fe[j],elemental[i].ne[j]);
	for(j=0;j<INELASTICSCATTERINGPREDATA;j++)
		fprintf(stdout,"%16.10lf %16.10lf\n",elemental[i].si[j],elemental[i].S[j]);
}
