#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <ctype.h>
#include "programming.h"
#include "instrumental.h"
#include "status.h"
#include "atomictable.h"

#include "readpredata.h"
#include "readdata.h"
#include "readzmatrix.h"
#include "empirical.h"
#include "outputs.h"
#include "errors.h"
#include "jacobi.h"

void limitcheck(int i,int m,char *s)
{
	if(i>m)
	{
		fprintf(stderr,"& The number of %s (%d) exceeds the limit of %d\n",s,i,m);
		fprintf(stderr,"& See programing.h for parameters\n");
		exit(1);
	}
	return;
}

void zmatrixerror(char *s)
{
	fprintf(stderr,"zmatrix in %s is not valid\n",s);
	return;
}

int ztypechar(char *s)
{
	int zt=0;
	switch (s[0])
	{
		case 'r' : zt=INDEX_ZMX_BONDLENGTH; break;
		case 'l' : zt=INDEX_ZMX_BONDLENGTH; break;
		case 'a' : zt=INDEX_ZMX_BENDINGANGLE; break;
		case 'b' : zt=INDEX_ZMX_BENDINGANGLE; break;
		case 'd' : zt=INDEX_ZMX_DIHEDRAL; break;
		case 't' : zt=INDEX_ZMX_DIHEDRAL; break;
		default  : fprintf(stderr," unknown coordinate name type for %s\n",s);
			   exit(1);
	}
	return (zt);
}

int findzreference(char *reference, char list[][ATOMICINDEXNAMELENGTH],int n)
{
	int i;
	for(i=0;i<n;i++)
		if(!strcasecmp(reference,list[i])) 
			return(i);
	zmatrixerror(reference);
	return(-1);
}

int readxyz(int detail,MOLECULAR *molecule,int ne,ELEMENTAL *elemental,char *fn)
{
	ATOMIC *atom=&(molecule->atomic[molecule->i]);
	int i;
	FILE *fp;
	char *ef;
	char buffer[STRINGBUFFERSIZE];
	char fnc[STRINGBUFFERSIZE];

	// if(data->preset>3) data->preset=3;
	sprintf(fnc,"%s.xyz",fn);
	if(NULL==(fp=fopen(fnc,"rt"))) ioerror(fnc);
	else
	{
		if(NULL==(ef=fgets(buffer,STRINGBUFFERSIZE,fp))) ioerror(fnc);
		sscanf(buffer,"%s",atom->molecule);
		if(NULL==(ef=fgets(buffer,STRINGBUFFERSIZE,fp))) ioerror(fnc);
		sscanf(buffer,"%d",&(atom->na));
		limitcheck(atom->na,MAX_NUM_ATOM,"MAX_NUM_ATOM");
		for(i=0;i<atom->na;i++)
		{
			if(NULL==(ef=fgets(buffer,STRINGBUFFERSIZE,fp))) ioerror(fnc);
			sscanf(buffer,"%s%lf%lf%lf",atom->atom[i],atom->q[i],atom->q[i]+1,atom->q[i]+2);
			atom->et[i]=findnucleartype(atom->atom[i],ne,elemental);
			// atom->eZ[i]=elemental[z->et[i]].charge;
		}
		fclose(fp);
		if(molecule->n==molecule->i)
		{
			molecule->n++;	// looks redundant
			if(detail) fprintf(stderr,"    read xyz %d coordinates for %s from %s\n"
					,atom->na,atom->molecule,fnc);
		}
		else
			if(detail) fprintf(stderr,"    changed xyz %d coordinates for %s from %s\n"
					,atom->na,atom->molecule,fnc);
		molecule->kind[molecule->i]=INDEX_KIND_XYZ;
		molecule->preset[molecule->i]=STATUS_ZMX_NEED_TO_MAKE_MLS;
	}
	return(atom->na);
}

int buildzmx(int detail,MOLECULAR *molecule)
{
	ZMAT *z=&(molecule->zmx[molecule->i]);
	ATOMIC *atom=&(molecule->atomic[molecule->i]);
	int i;

	for(i=0;i<z->na;i++)
	{
		switch(i)
		{
			default:
				z->r[z->zindex[i][2]]=dihedralangle(atom->q[i]
						,atom->q[z->ref[i][0]],atom->q[z->ref[i][1]],atom->q[z->ref[i][2]]);
			case 2 :
				z->r[z->zindex[i][1]]=bendingangle(atom->q[i]
						,atom->q[z->ref[i][0]],atom->q[z->ref[i][1]]);
			case 1 :
				z->r[z->zindex[i][0]]=bondlength(atom->q[i],atom->q[z->ref[i][0]]);
			case 0 :
				break;
		}

	}
	return(z->nz);
}

int evaluatezmx4xyz(int detail,MOLECULAR *molecule)
{
	ZMAT *z=&(molecule->zmx[molecule->i]);
	ATOMIC *atom=&(molecule->atomic[molecule->i]);
	int i=0;

	if(atom->na==z->na)
	{
		for(i=0;i<z->na;i++)
		{
			if(atom->et[i]!=z->et[i])
			{
				if(detail) fprintf(stderr,"    %dth elements are not same %d,%d\n",i+1,atom->et[i],z->et[i]);
				return(0);
			}
		}
		if(0<(i=buildzmx(detail,molecule)))
		{
			molecule->kind[molecule->i]=INDEX_KIND_ZMX;
			molecule->preset[molecule->i]=STATUS_ZMX_NEED_TO_MAKE_ALL;
			sprintf(z->molecule,"%s",atom->molecule);
			if(detail) fprintf(stderr,"    converted xyz to zmx for %s\n",z->molecule);
		}
		else
			fprintf(stderr,"     conversion failed for %s\n",z->molecule);
	}
	else
		fprintf(stderr,"    numbers of atoms don't match %d != %d\n",atom->na,z->na);
	return(i);
}

int assignzmxmass(int detail,MOLECULAR *molecule,int ne,ELEMENTAL *elemental)
{
	ZMAT *z=&(molecule->zmx[molecule->i]);
	ATOMIC *atom=&(molecule->atomic[molecule->i]);
	int i;
	double w;
	for(i=0;i<z->na;i++)
	{
		w=tablemass[z->eZ[i]];
		atom->m[i][0]=w;
		atom->m[i][1]=w;
		atom->m[i][2]=w;
	}
	return(z->na);
}

int readzmx(int detail,MOLECULAR *molecule,int ne,ELEMENTAL *elemental,char *fn)
{
	ZMAT *z=&(molecule->zmx[molecule->i]);
	int i,j,k,nz=0,ns,ni,nc,zij;
	FILE *fp;
	char buffer[STRINGBUFFERSIZE];
	char name[STRINGBUFFERSIZE];
	char c[12][STRINGBUFFERSIZE];
	char fnz[STRINGBUFFERSIZE];
	char zmxtype[][16]={"unused or wrong","bond length","bending angle","dihedral angle"};
	int zmxindex[]={ INDEX_ZMX_BONDLENGTH, INDEX_ZMX_BENDINGANGLE, INDEX_ZMX_DIHEDRAL };


	// if(data->preset>3) data->preset=3;
	sprintf(fnz,"%s.zmx",fn);
	if(NULL==(fp=fopen(fnz,"rt"))) ioerror(fnz);
	else
	{
		if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) ioerror(fnz);
		sscanf(buffer,"%s",z->molecule);
		if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) ioerror(fnz);
		sscanf(buffer,"%d",&(z->na));
		limitcheck(z->na,MAX_NUM_ATOM,"MAX_NUM_ATOM");
		for(i=0;i<z->na;i++)
		{
			if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) ioerror(fnz);
			sscanf(buffer,"%s",name);
			z->et[i]=findnucleartype(name,ne,elemental);
			z->eZ[i]=elemental[z->et[i]].charge;
		}
		if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) ioerror(fnz);
		nz=MAX_NUM_ZMAT;
		for(i=0;i<nz;i++)
		{
			if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) break;
			z->ien[i]=0;
			if(2>(ns=sscanf(buffer,"%s%s%s%s%s%s%s%s%s%s%s%s"
				,c[0],c[1],c[2],c[3],c[4],c[5],c[6],c[7],c[8],c[9],c[10],c[11])))
				break;
			else
			{
				sscanf(c[0],"%s",z->z[i]);
				if(!strcmp("=",c[1])) 
					z->zt[i]=linkdependency(detail,i,i,ns,c,z);
				else
				{
					sscanf(buffer,"%s%lf",z->z[i],&(z->r[i]));
			 		z->zt[i]=INDEX_ZMX_UNUSED;
				}
			}
		}
		nz=i;
		if(nz<1) ioerror(fnz);

		//evaluatecoordinates(z);
			
		rewind(fp);
		fgets(buffer,STRINGBUFFERSIZE,fp);
		fgets(buffer,STRINGBUFFERSIZE,fp);
		for(i=0;i<z->na;i++)
		{
			if(NULL==(fgets(buffer,STRINGBUFFERSIZE,fp))) ioerror(fnz);
			ni=sscanf(buffer,"%s%s%s%s%s%s%s",z->atom[i]
					,z->reference[i][0],z->coordinate[i][0]
					,z->reference[i][1],z->coordinate[i][1]
					,z->reference[i][2],z->coordinate[i][2]); 
			nc=(ni-1)/2;
//			if(detail) fprintf(stderr,"%s(%d) ",z->atom[i],i);
			for(j=0;j<nc;j++)
			{
				if(isdigit(z->reference[i][j][0]))
					z->ref[i][j]=atoi(z->reference[i][j])-1;
				else
					z->ref[i][j]=findzreference(z->reference[i][j],z->atom,i);
				zij=findzreference(z->coordinate[i][j],z->z,nz);
				z->zindex[i][j]=zij;
				switch(z->zt[zij])
				{
					case INDEX_ZMX_UNUSED :
						z->zt[zij]=zmxindex[j];
						break;
					case INDEX_ZMX_FUNCTION :
						fprintf(stderr," %s is function of",z->z[zij]);
						fprintf(stderr," %s",z->z[z->b[zij][0]]);
						fprintf(stderr,"\n");
						break;
					case INDEX_ZMX_DEPENDENT :
						z->zt[zij]=-zmxindex[j];
						fprintf(stderr," %s is %s dependent on",z->z[zij]
							,zmxtype[zmxindex[j]]);
						for(k=1;k<z->nd[zij];k++) 
							fprintf(stderr," %s",z->z[z->b[zij][k]]);
						fprintf(stderr,"\n");
						break;
					case INDEX_ZMX_DEPENDENT_LENGTH :
					case INDEX_ZMX_DEPENDENT_ANGLE :
					case INDEX_ZMX_DEPENDENT_DIHEDRAL :
						if(-zmxindex[j]!=z->zt[zij])
						{
							fprintf(stderr,"    %s was assigned as %s(%d) and then as %s(%d)\n"
									,z->coordinate[i][j],zmxtype[z->zt[zij]],z->zt[zij],zmxtype[zmxindex[j]],zmxindex[j]);
							z->nz=0;
							fclose(fp);
							return(0);
						}
						else 
							fprintf(stderr,"    %s is recycled as %s(%d)\n"
									,z->coordinate[i][j],zmxtype[-z->zt[zij]],z->zt[zij]);
						break;
					default:
						if(zmxindex[j]!=z->zt[zij])
						{
							fprintf(stderr,"    %s was assigned as %s(%d) and then as %s(%d)\n"
									,z->coordinate[i][j],zmxtype[z->zt[zij]],z->zt[zij],zmxtype[zmxindex[j]],zmxindex[j]);
							z->nz=0;
							fclose(fp);
							return(0);
						}
						break;
				}
			}
		}
		fclose(fp);
		if(molecule->n==molecule->i)
		{
			molecule->n++;	// looks redundant
			if(detail) fprintf(stderr,"    read %d coordinates for %s from %s\n",nz,z->molecule,fnz);
		}
		else
			if(detail) fprintf(stderr,"    changed zmatrix %d coordinates for %s from %s\n",nz,z->molecule,fnz);
		molecule->kind[molecule->i]=INDEX_KIND_ZMX;
		molecule->preset[molecule->i]=STATUS_ZMX_NEED_TO_MAKE_ALL;
		z->nz=nz;
		z->nr=0;
	}
	return(z->nz);
}

double dsqr(double x)
{
	return(x*x);
}

double dsin(double x)
{
	return(sin(x*PI/180));
}

double dcos(double x)
{
	return(cos(x*PI/180));
}

double evaluatedependent(int i,ZMAT *z)
{
	int j,n=z->nd[i];
	double r=z->r[i]; 
        switch(z->zt[i])
        {
                default:
                        break;
                case INDEX_ZMX_FUNCTION :
			z->r[z->b[i][0]]=evaluatedependent(z->b[i][0],z);
			r=(z->function[i])(z->r[z->b[i][0]]);
                        break;
                case INDEX_ZMX_DEPENDENT :
                case INDEX_ZMX_DEPENDENT_LENGTH :
                case INDEX_ZMX_DEPENDENT_ANGLE :
                case INDEX_ZMX_DEPENDENT_DIHEDRAL :
			r=z->a[i][0]; 
			for(j=1;j<n;j++)
			{
				z->r[z->b[i][j]]=evaluatedependent(z->b[i][j],z);
				r+=z->a[i][j]*z->r[z->b[i][j]];
			}
			break;
	}
	return(r);
}

int evaluatecoordinates(ZMAT *z)
{
	int i;
	for(i=0;i<z->nz;i++)
		switch(z->zt[i])
		{
                	default:
                        	break;
                	case INDEX_ZMX_FUNCTION :
			case INDEX_ZMX_DEPENDENT :
			case INDEX_ZMX_DEPENDENT_LENGTH :
			case INDEX_ZMX_DEPENDENT_ANGLE :
			case INDEX_ZMX_DEPENDENT_DIHEDRAL :
				evaluatedependent(i,z);
				break;
		}
	return(z->nz);
}

int linkdependency(int detail,int i,int n,int ns,char c[][STRINGBUFFERSIZE],ZMAT *z)
{
	int j,k,na=(ns+2)/4;
	double a1;

	z->nd[i]=na;
	if(0)
	{
		fprintf(stderr,"%d: ",na);
		for(i=0;i<ns;i++) fprintf(stderr,"%s ",c[i]);
		fprintf(stderr,"\n");
	}
	if(MAX_NUM_DEPENDENT<na) { uedmessage(" too many dependency"); return(INDEX_ZMX_WRONG); }
	if(1>sscanf(c[2],"%lf",&(z->a[i][0]))) 
	{
		z->zt[i]=INDEX_ZMX_FUNCTION;
		if(0>(z->b[i][0]=findzreference(c[3],z->z,i)))
			{ uedmessage(" r should refer zmx"); return(INDEX_ZMX_WRONG); } 
		if(!strcasecmp("sqr",c[2])) (z->function[i])=&dsqr;
		else if(!strcasecmp("sin",c[2])) (z->function[i])=&dsin;
		else if(!strcasecmp("cos",c[2])) (z->function[i])=&dcos;
		else (z->function[i])=&dsqr;
		z->r[i]=evaluatedependent(i,z);
		fprintf(stderr," %s = %lf = %s[%s(%lf)]\n" ,z->z[i],z->r[i],"f",z->z[z->b[i][0]],z->r[z->b[i][0]]);
		return(INDEX_ZMX_FUNCTION);
	}
	else
	{
		for(j=1;j<na;j++)
		{
			k=4*j-1;
			if('+'==c[k][0]) z->a[i][j]=1;
			else if('-'==c[k][0]) z->a[i][j]=-1;
			else { z->a[i][j]=0; uedmessage(" use either + or -"); return(INDEX_ZMX_WRONG); }

			if(0>(z->b[i][j]=findzreference(c[++k],z->z,i)))
				{ uedmessage(" b should refer zmx"); return(INDEX_ZMX_WRONG); } // k=4*j
			if(1>sscanf(c[(++k)+1],"%lf",&a1)) { uedmessage(" a1 should be number"); } // k=4*j+1

			if('*'==c[k][0]) z->a[i][j]*=a1;
			else if('/'==c[k][0]) z->a[i][j]/=a1;
			else { uedmessage(" either * or / should come at 5th"); }
		}
		z->r[i]=evaluatedependent(i,z);
		fprintf(stderr," %s = %lf = %lf" ,z->z[i],z->r[i],z->a[i][0]);
		for(j=1;j<na;j++)
			fprintf(stderr," + %s(%lf) * %lf",z->z[z->b[i][j]],z->r[z->b[i][j]],z->a[i][j]);
		fprintf(stderr,"\n");
	}
	return(INDEX_ZMX_DEPENDENT);
}

double zvalue(ZMAT *z,int i,int t)
{
	int c,k;
	double r=0;
	c=z->zindex[i][t];
	k=z->zt[c];
	switch(k)
	{
		default:
			r=z->r[c];
			break;
		case INDEX_ZMX_FUNCTION           :
		case INDEX_ZMX_DEPENDENT          : 
		case INDEX_ZMX_DEPENDENT_LENGTH   : 
		case INDEX_ZMX_DEPENDENT_ANGLE    : 
		case INDEX_ZMX_DEPENDENT_DIHEDRAL : 
			r=evaluatedependent(c,z);
	}
	switch(k)
	{
		case INDEX_ZMX_BENDINGANGLE       : r*=PI/180; break;
		case INDEX_ZMX_DIHEDRAL           : r*=PI/180; break;
		case INDEX_ZMX_DEPENDENT_ANGLE    : r*=PI/180; break;
		case INDEX_ZMX_DEPENDENT_DIHEDRAL : r*=PI/180; break;
			break;
		default: break;
	}
	return(r);
}

double dihedralangle(double *qi, double *qj, double *qk, double *ql)
{
	double a,d,s;
	double dij[3],djk[3],dkl[3];
	double eijk[3],ejkl[3],eijkl[3];
	displacement(dij,qi,qj);
	displacement(djk,qj,qk);
	displacement(dkl,qk,ql);
	crossvector(eijk,dij,djk);
	crossvector(ejkl,djk,dkl);
	unitvector(eijk);
	unitvector(ejkl);
	d=dotvector(eijk,ejkl);
	a=acos(d)*180/PI;
	crossvector(eijkl,eijk,ejkl);
	s=dotvector(eijkl,djk);
	if(s>0) a*=-1;
	return(a);
}

double bendingangle(double *qi, double *qj, double *qk)
{
	double a,d;
	double eij[3],ejk[3];
	displacement(eij,qi,qj);
	displacement(ejk,qj,qk);
	unitvector(eij);
	unitvector(ejk);
	d=-dotvector(eij,ejk);
	a=acos(d)*180/PI;
	return(a);
}

double dotvector(double *ei, double *ej)
{
	int i;
	long double sum=0;
	for(i=0;i<3;i++)
		sum+=ei[i]*ej[i];
	if(sum>+1) sum=+1;
	if(sum<-1) sum=-1;
	return((double)sum);
}

double bondlength(double *qi, double *qj)
{
	int i;
	double x,norm=0;
	for(i=0;i<3;i++)
	{
		x=qi[i]-qj[i];
		norm+=x*x;
	}
	norm=sqrt(norm);
	return(norm);
}

double unitvector(double *e)
{
	int i;
	long double norm=0;
	for(i=0;i<3;i++)
		norm+=e[i]*e[i];
	norm=sqrtl(norm);
	if(norm>TOLERANCE_ZERO)
		for(i=0;i<3;i++)
			e[i]/=norm;
	else
	{
		fprintf(stderr,"something wrong. coincident atoms in normalvector\n");
		exit(1);
	}
	return((double)norm);
}

void displacement(double *ejk, double *qj, double *qk)
{
	int i;
	for(i=0;i<3;i++)
		ejk[i]=qj[i]-qk[i];
	return;
}

void crossvector(double *e, double *qjk, double *qkl)
{
	e[0]=qjk[1]*qkl[2]-qjk[2]*qkl[1];
	e[1]=qjk[2]*qkl[0]-qjk[0]*qkl[2];
	e[2]=qjk[0]*qkl[1]-qjk[1]*qkl[0];
	return;
}

void putatom(double *qi, double *qj,double r, double a, double t, double *ex, double *ey, double *ez)
{
	int i;
	double rsina=r*sin(a);
	double x=rsina*cos(t);
	double y=rsina*sin(t);
	double z=-r*cos(a);
	for(i=0;i<3;i++)
		qi[i]=qj[i]+x*ex[i]+y*ey[i]+z*ez[i];
	return;
}

int buildmolecule(int detail,MOLECULAR *molecule)
{
	ZMAT *z=&(molecule->zmx[molecule->i]);
	ATOMIC *atom=&(molecule->atomic[molecule->i]);
	int i,j,k,l,na=z->na;
	double r,a,t;
	double rg=0;
	static double ejk[3],ekl[3],ex[3],ey[3];

	if(na>0)
	{
		atom->q[0][0]=0;
		atom->q[0][1]=0;
		atom->q[0][2]=0;
	}
	if(na>1)
	{
		r=zvalue(z,1,0);
		atom->q[1][0]=r;
		atom->q[1][1]=0;
		atom->q[1][2]=0;
	}
	if(na>2)
	{
		r=zvalue(z,2,0);
		a=zvalue(z,2,1);
		j=z->ref[2][0];
		k=z->ref[2][1];
		if(j==0) r=-r;
		atom->q[2][0]=atom->q[j][0]-r*cos(a);
		atom->q[2][1]=atom->q[j][1]+r*sin(a);
		atom->q[2][2]=atom->q[j][2];
	}
	for(i=3;i<na;i++)
	{
		//fprintf(stderr,"i= %d\n",i);
		r=zvalue(z,i,0);
		a=zvalue(z,i,1);
		t=zvalue(z,i,2);
		j=z->ref[i][0];
		k=z->ref[i][1];
		l=z->ref[i][2];
		displacement(ejk,atom->q[j],atom->q[k]);
		displacement(ekl,atom->q[k],atom->q[l]);
		crossvector(ey,ekl,ejk);
		crossvector(ex,ey,ejk);
		unitvector(ey);
		unitvector(ex);
		unitvector(ejk);
		putatom(atom->q[i],atom->q[j],r,a,t,ex,ey,ejk);
	}

	atom->na=na;
//	fprintf(stderr," &&&&& na= %d &&&&&\n",atom->na);
	t=0;
	for(j=0;j<3;j++)
		ex[j]=0;
	for(i=0;i<na;i++)
	{
		atom->et[i]=z->et[i];
		t+=tablemass[z->eZ[i]];
		for(j=0;j<3;j++)
			ex[j]+=tablemass[z->eZ[i]]*atom->q[i][j];
	}
	for(j=0;j<3;j++)
		ex[j]/=t;
	rg=0;
	for(i=0;i<na;i++)
		for(j=0;j<3;j++)
		{
			atom->q[i][j]-=ex[j];
			rg+=tablemass[z->eZ[i]]*atom->q[i][j]*atom->q[i][j];
		}
	rg=sqrt(rg/t);
	if(detail) fprintf(stderr," (mass-weighted) radius of gyration is %lf A\n",rg);
	sprintf(atom->molecule,"%s",z->molecule);
	if(detail) fprintf(stderr,"    built cartesian coordinates of %d atoms for %s\n",na,atom->molecule);
	return(na);
}

int rotatemolecule(int detail,MOLECULAR *molecule)
{
	ZMAT *z=&(molecule->zmx[molecule->i]);
	ATOMIC *atom=&(molecule->atomic[molecule->i]);
	int i,j,k,l=0,na=atom->na;
	double t,ex[3],**mi,**e,q[MAX_NUM_ATOM][3];

	if(NULL==(mi=(double **)calloc(3,sizeof(double *)))) uerror("rotatemolecule","lack of memory");
	for(i=0;i<3;i++)
		if(NULL==(mi[i]=(double *)calloc(3,sizeof(double)))) uerror("rotatemolecule","lack of memory");
	if(NULL==(e=(double **)calloc(3,sizeof(double *)))) uerror("rotatemolecule","lack of memory");
	for(i=0;i<3;i++)
		if(NULL==(e[i]=(double *)calloc(3,sizeof(double)))) uerror("rotatemolecule","lack of memory");

	fprintf(stderr,"rotating molecule\n");
	t=0;
	for(j=0;j<3;j++)
		ex[j]=0;
	for(i=0;i<na;i++)
	{
		atom->et[i]=z->et[i];
		t+=tablemass[z->eZ[i]];
		for(j=0;j<3;j++)
			ex[j]+=tablemass[z->eZ[i]]*atom->q[i][j];
	}
	for(j=0;j<3;j++)
		ex[j]/=t;
	for(i=0;i<na;i++)
		for(j=0;j<3;j++)
			atom->q[i][j]-=ex[j];

	for(j=0;j<3;j++)
		for(k=0;k<3;k++)
			e[j][k]=mi[j][k]=0;
	for(i=0;i<na;i++)
		for(j=0;j<3;j++)
		{
			for(k=0;k<3;k++)
				mi[j][j]+=tablemass[z->eZ[i]]*atom->q[i][k]*atom->q[i][k];
			for(k=0;k<3;k++)
				mi[j][k]-=tablemass[z->eZ[i]]*atom->q[i][j]*atom->q[i][k];
		}

	for(j=0;j<3;j++)
	{
		for(k=0;k<3;k++)
			fprintf(stderr,"%20.10lf",mi[j][k]);
		fprintf(stderr,"\n");
	}
	jacobidiagonalize(3,mi,ex,e,&l);
	jacobisort(3,ex,e);
	for(j=0;j<3;j++)
	{
		fprintf(stderr,"%20.10lf :: ",ex[j]);
		for(k=0;k<3;k++)
			fprintf(stderr,"%20.10lf",e[j][k]);
		fprintf(stderr,"\n");
	}
	for(i=0;i<na;i++)
		for(j=0;j<3;j++)
		{
			q[i][j]=0;
			for(k=0;k<3;k++)
				q[i][j]+=atom->q[i][k]*e[k][j];
		}
	for(i=0;i<na;i++)
		for(j=0;j<3;j++)
			atom->q[i][j]=q[i][j];
	return(na);

	for(i=0;i<3;i++) free(e[i]);
	free(e);
	for(i=0;i<3;i++) free(mi[i]);
	free(mi);

}

double reducedmass(double m1, double m2)
{
	return(m1*m2/(m1+m2)*1.2);	// 1.2 is empirical correction to fit better
}

double reducedmasset(int c1, int c2)
{
	return(reducedmass(tablemass[c1],tablemass[c2]));
}

int buildmls(int detail,MOLECULAR *molecule, int ne, ELEMENTAL *ele)
{
	MLS *mls=&(molecule->mls[molecule->i]);
	// ZMAT *z=&(molecule->zmx[molecule->i]);
	ATOMIC *atom=&(molecule->atomic[molecule->i]);
	double r;
	int i,j,k=0,na=atom->na;

	mls->na=na;
//	fprintf(stderr," &&&&& mls na= %d &&&&&\n",mls->na);
	for(i=0;i<na;i++)
	{
		mls->et0[i]=atom->et[i];
		mls->ne0[i]=1;
	}
	mls->nl=na*(na-1)/2;
	// for(i=0;i<na-1;i++) for(j=i+1;j<na;j++)
	for(j=1;j<na;j++)
		for(i=0;i<j;i++)
		{
			r=bondlength(atom->q[i],atom->q[j]);
			if(r<mls->limit)
			{
				mls->d[k]=1;
				mls->et1[k]=atom->et[i];
				mls->et2[k]=atom->et[j];
				strcpy(mls->a1[k],ele[atom->et[i]].symbol);
				strcpy(mls->a2[k],ele[atom->et[j]].symbol);
				mls->re[k]=bondlength(atom->q[i],atom->q[j]);
				mls->mu[k]=reducedmasset(ele[atom->et[i]].charge,ele[atom->et[j]].charge);
				mls->T[k]=300;
				if(detail>DETAIL_BUILD_MLS) 
					fprintf(stderr," %5d : %5d %5d %5d %s %s %lf\n",mindex(i,j),k,i,j,mls->a1[k],mls->a2[k],mls->re[k]);
				k++;
			}
			else if(FRAGMENTALMLS)
			{
				mls->d[k]=0;
				mls->et1[k]=atom->et[i];
				mls->et2[k]=atom->et[j];
				strcpy(mls->a1[k],ele[atom->et[i]].symbol);
				strcpy(mls->a2[k],ele[atom->et[j]].symbol);
				mls->re[k]=bondlength(atom->q[i],atom->q[j]);
				mls->mu[k]=reducedmasset(ele[atom->et[i]].charge,ele[atom->et[j]].charge);
				mls->T[k]=300;
				if(detail>DETAIL_BUILD_MLS) 
					fprintf(stderr," %5d : %5d %5d %5d %s %s %lf\n",mindex(i,j),k,i,j,mls->a1[k],mls->a2[k],mls->re[k]);
				k++;
			}
			else
			{
				if(detail>DETAIL_BUILD_MLS) 
					fprintf(stderr,"    distance between %d and %d is %lf. it looks like fragments\n",i,j,r);
			}
		}
	mls->nl=k;
	sprintf(mls->molecule,"%s",atom->molecule);
	if(detail)
	{
		fprintf(stderr,"    built mls data of %d bonds from %s xyz coordinate\n",k,mls->molecule);
		if(k!=mls->nl) 
			fprintf(stderr," fragmentation ? the numbers of bond lengths do not match ? %d or %d\n",k,mls->nl);
	}
	return(k);
}

int updatemls(int detail,MOLECULAR *molecule, int ne, ELEMENTAL *ele)
{
	MLS *mls=&(molecule->mls[molecule->i]);
	// ZMAT *z=&(molecule->zmx[molecule->i]);
	ATOMIC *atom=&(molecule->atomic[molecule->i]);
	double r;
	int i,j,k=0,na=atom->na;

	for(j=1;j<na;j++)
		for(i=0;i<j;i++)
		{
			r=bondlength(atom->q[i],atom->q[j]);
			if(r<mls->limit)
			{
				mls->re[k]=bondlength(atom->q[i],atom->q[j]);
				k++;
			}
			else if(FRAGMENTALMLS)
			{
				mls->re[k]=bondlength(atom->q[i],atom->q[j]);
				k++;
			}
			else
			{
				if(detail>DETAIL_BUILD_MLS) 
					fprintf(stderr,"    distance between %d and %d is %lf. it looks like fragments\n",i,j,r);
			}
		}
	if(detail)
	{
		fprintf(stderr,"    built mls data of %d bonds from %s xyz coordinate\n",k,mls->molecule);
		if(k!=mls->nl) 
			fprintf(stderr," fragmentation ? the numbers of bond lengths do not match ? %d or %d\n",k,mls->nl);
	}
	return(k);
}
