#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "programming.h"
#include "intrinsic.h"
#include "status.h"

#include "errors.h"
#include "readzmatrix.h"
#include "evaluate.h"
#include "numericalfit.h"
#include "interface.h"
#include "transform.h"
#include "coordinates.h"
#include "setstructure.h"

int updatereference(INTERNAL *h)
{
	fDCOPY(h->nc,h->cc,1,*(h->q),1);
	return(h->nc);
}

int updatecoordinate(INTERNAL *h)
{
	calculatebtmatrix(h->nr,h->nc,h->z,h->r,h->cc,h->bt);
	return(h->nr);
}

int instantaneousinternal(double *r,INTERNAL *h)
{
	calculatebtmatrix(h->nr,h->nc,h->z,r,h->cc,h->bt);
	calculatemibtmatrix(h->nc,h->nr,h->bt,h->m,h->mb);	// M^-1 B^T = (B M^-1^T)^T = (B M^-1)^T
	calculategtmatrix(h->nc,h->nr,h->bt,h->mb,h->g);	// G= B (M^-1 B^T)
	//calculategimatrix(h->nr,h->g,h->gi);			// G^-1 = V^T E^-1 V ?
	calculategsmatrix(h->nr,h->g,h->gi,h->gs,h->gq);	// G^-1 = V^T E^-1 V ?
	calculatebimatrix(h->nc,h->nr,h->mb,h->gi,h->bi);	// B^-1 = (M^-1 B^T) G^-1
	calculatepimatrix(h->nc,h->nr,h->bt,h->bi,h->pi);	// pI   = (B^T)^T B^-1
	return(h->nr);
}

double excitemolecule(double *r,INTERNAL *h)
{
	int detail=0;
	int i,j,nr=h->nr,nc=h->nc;
	double step=1,rd1=0,rd2=0,*rd=NULL,*ud=NULL;
	if(NULL==(rd=(double *)calloc(nr,sizeof(double)))) uerror("excitemolecule","rd");
	if(NULL==(ud=(double *)calloc(nr,sizeof(double)))) uerror("excitemolecule","ud");

	fDCOPY(h->nc,*(h->q),1,h->cc,1);
	for(i=-2;i<2*nr;i++)
	{
		instantaneousinternal(rd,h);
		for(j=0;j<nr;j++) 
			if(DIHEDRAL_ANGLE==h->z[j].type)
				rd[j]=dihedraldifference(r[j]-h->z[j].r);
			else
				rd[j]=r[j]-h->z[j].r;
		rd2=fDDOT(nr,rd,1,rd,1);
		if(detail) fprintf(stderr," rd2= %le :",rd2);
		fDGEMV('N',nr,nr,1.0,h->gq,nr,rd,1,0.0,ud,1);
		rd2=fDDOT(nr,ud,1,ud,1);
		if(detail) fprintf(stderr," ud2= %le :",rd2);
		fDGEMV('N',nr,nr,1.0,h->gs,nr,ud,1,0.0,rd,1);
		rd2=fDDOT(nr,rd,1,rd,1);
		if(detail) fprintf(stderr," rd2= %le :",rd2);
		if(detail) fprintf(stderr," rd=  ");
		if(detail) for(j=0;j<nr;j++)
			fprintf(stderr,"%6.3lf",rd[j]);
		if(detail) fprintf(stderr,"\n");
		if(rd2<1.0E-16) break;
		if(fabs(rd2-rd1)<1.0E-16) break;
		if(i<nr) step=(i+3.)/(nr+4.); else step=1;
		fDGEMV('N',nc,nr,step,h->bi,nc,rd,1,1.0,h->cc,1);
		if(detail) fprintf(stderr," x=  ");
		if(detail) for(j=0;j<nc;j++)
			fprintf(stderr,"%6.3lf",h->cc[j]);
		if(detail) fprintf(stderr,"\n");
		rd1=rd2;
	}

	//instantaneousinternal(rd,h);
	//fprintf(stderr," R=  ");
	//for(j=0;j<nr;j++)
		//fprintf(stderr,"%6.3lf",rd[j]);
	//fprintf(stderr,"\n");
	free(ud);
	free(rd);
	return(rd2);
}

int buildstructure(int detail,MOLECULAR *molecule)
{
	int i=molecule->i;
	double rd2=0;
	INTERNAL *h=&(molecule->h[i]);
	rd2=excitemolecule(h->r,h);
	if(detail) fprintf(stderr," buildstructure built cartesian coordinates with rd2= %le\n",rd2);
	//updatereference(h);
	buildzmx(detail,molecule);
	//for(i=0;i<h->nr;i++) fprintf(stderr,"%8.4lf",h->r[i]); fprintf(stderr,"\n");
	//for(i=0;i<h->nr;i++) fprintf(stderr,"%8.4lf",h->z[i].r); fprintf(stderr,"\n");
	return(0);
}

int showstructure(int detail,FILE *fpi,DIFFRACTION *data,MOLECULAR *molecule
		,int ne,ELEMENTAL *elemental,INSTRUMENTAL *instrument,FITTING *fit)
{
	INTERNAL *h=&(molecule->h[molecule->i]);
	ATOMIC *a=&(molecule->atomic[molecule->i]);
	ZMAT *z=&(molecule->zmx[molecule->i]);
	int nr=h->nr,mi=molecule->i;
	int zindex,rindex,i,j,nc,set=0;
	char f[][10]={"fixed    ","to fit   ","dependent"};
	char input[STRINGBUFFERSIZE];
	char *ef,*c[MAX_KEYWORDS];

	if(INDEX_RIC_SUPPLIED!=molecule->ric[mi]) return(0);

	for(i=0;i<MAX_KEYWORDS;i++)
		if(NULL==(c[i]=(char *)calloc(KEYWORDLENGTH,sizeof(char)))) uerror("showstructure","c[]");

	while(1)
	{
		fprintf(stderr,"    [%d/%d] %s %lfK\n"
				,1+mi,molecule->n,h->molecule,molecule->temperature[mi]);
		fprintf(stderr,"  ---------------------------------------------------------------\n");
		for(i=0;i<h->na;i++)
			fprintf(stderr,"%6d  %-7s %12.8lf %12.8lf %12.8lf\n",1+i,z->atom[i]
					,h->q[i][0],h->q[i][1],h->q[i][2]);
		fprintf(stderr,"  ---------------------------------------------------------------\n");
		for(i=0;i<nr;i++)
		{
			fprintf(stderr,"%6d ",i+1);
			fprintf(stderr,"%15.8lf ",h->z[i].r*h->z[i].f);
			switch(h->z[i].type)
			{
				case BOND_RECIPROCAL:
				case BOND_RECIPROCAL2:
				case BOND_RECIPROCAL6:
				case BOND_STRETCHING: fprintf(stderr,"%15.8lf ",h->r[i]*h->z[i].f); break;
				case BENDING2_ANGLE:
				case LINEAR_BENDING:
				case PLANAR_BENDING:
				case DIHEDRAL_ANGLE:
				case BENDING_ANGLE: fprintf(stderr,"%15.6lf ",h->r[i]*h->z[i].f); break;
				case TANGENT_PI_DIHEDRAL2:
				case TANGENT_DIHEDRAL2: fprintf(stderr,"%15.10lf ",h->r[i]*h->z[i].f); break;
				default: fprintf(stderr,"%15.8lf ",h->r[i]*h->z[i].f); break;
			}
			fprintf(stderr,"%4d,",h->z[i].type);
			for(j=0;j<h->z[i].indices;j++)
				fprintf(stderr," %4s",z->atom[h->z[i].id[j]]);
			for(j=h->z[i].indices;j<5;j++)
				fprintf(stderr," %4s","");
			//for(j=0;j<h->z[i].indices;j++)
			//	fprintf(stderr," %4d",h->z[i].id[j]+1);
			//for(j=h->z[i].indices;j<5;j++)
			//	fprintf(stderr," %4s","");
			fprintf(stderr,"%10s",f[fit->internal[mi][i]]);
			fprintf(stderr," \n");
		}
		if(nr) fprintf(stderr,"  ---------------------------------------------------------------\n");
		fprintf(stderr,"    which ? ");
		// fscanf(fp,"%s",input);
		if(NULL==(fgets(input,STRINGBUFFERSIZE,fpi))) break;
		for(i=0;i<MAX_KEYWORDS;i++) sprintf(c[i],"%s","");
		nc=0;
		switch(MAX_KEYWORDS)
		{
			case 2 : nc=sscanf(input,"%s%s",c[0],c[1]); break;
			case 8 : nc=sscanf(input,"%s%s%s%s%s%s%s%s",c[0],c[1],c[2],c[3],c[4],c[5],c[6],c[7]); break;
			default: 
				 fprintf(stderr," code error in reading zmatrix\n");
				 exit(1);
		}
		if(!strcasecmp("done",c[0])) break;
		else if(!strcasecmp("go",c[0])) 
		{
			if(set)
			{
				setpreset(&(data->preset),STATUS_NEED_TO_MAKE_MOLECULAR);
				set=0;
			}
			singlepoint(0,1,data,molecule,ne,elemental,instrument,fit);
			calculatenonlinears(detail,data,molecule,ne,elemental,instrument,fit);
		}
		else if(!strcasecmp("fit",c[0])) 
		{
			for(i=1;i<nc;i++)
			{
				zindex=atoi(c[i])-1;
				if(!strcasecmp("temperature",c[i]))
					fit->temperature[mi]=1;
				else if(0<=zindex&&zindex<nr)
				{
					switch(h->z[zindex].type)
					{
						case BOND_STRETCHING:
						case BOND_RECIPROCAL:
						case BOND_RECIPROCAL2:
						case BOND_RECIPROCAL6:

						case BENDING_ANGLE:
						case BENDING2_ANGLE:
						case LINEAR_BENDING:
						case PLANAR_BENDING:
						case DIHEDRAL_ANGLE:

						case TANGENT_PI_DIHEDRAL2:
						case TANGENT_DIHEDRAL2:
							fit->internal[mi][zindex]=1;
							break;

						case NONE:
							fit->internal[mi][zindex]=0;
							break;
						default:
							uerror("showstructure","type");
					}
				}
			}
		}
		else if(!strcasecmp("fix",c[0])) 
		{
			for(i=1;i<nc;i++)
			{
				zindex=atoi(c[i])-1;
				if(!strcasecmp("temperature",c[i]))
					fit->temperature[mi]=0;
				else if(0<=zindex&&zindex<nr)
					fit->internal[mi][zindex]=0;
			}
		}
		else if(!strcasecmp("set",c[0])) 
		{
			zindex=atoi(c[1])-1;
			if(!strcasecmp("temperature",c[1])) 
			{
				molecule->temperature[mi]=atof(c[2]);
				setpreset(&(molecule->preset[mi]),STATUS_ZMX_NEED_TO_MAKE_AMPLITUDE);
				set++;
			}
			else if(!strcasecmp("molecule",c[1])) 
			{
				mi=atoi(c[2])-1;
				if((0<=mi)&&(mi<molecule->n)) molecule->i=mi;
				else mi=molecule->i;
				h=&(molecule->h[molecule->i]);
				z=&(molecule->zmx[molecule->i]);
				a=&(molecule->atomic[molecule->i]);
				nr=h->nr;
				fprintf(stderr,"    changed molecule to %d\n",1+mi);
			}
			else if(0<=zindex&&zindex<h->nr)
			{
				//if(0>(rindex=atoi(c[2])-1))
					h->r[zindex]=atof(c[2])/h->z[zindex].f;
				//else
				//	h->r[zindex]=z->r[rindex];
				setpreset(&(molecule->preset[mi]),STATUS_ZMX_NEED_TO_MAKE_ALL);
				set++;
			}
			// setpreset(&(data->preset),STATUS_NEED_TO_MAKE_MOLECULAR); // will be dealt in ueda.c
		}
	}
	// fgets(input,STRINGBUFFERSIZE,fpi);
	for(i=0;i<MAX_KEYWORDS;i++)
		free(c[i]);
	if(set)
	{
		setpreset(&(data->preset),STATUS_NEED_TO_MAKE_MOLECULAR);
		setmoleculemakesmt(detail,1,data,molecule,ne,elemental,instrument);
	}
	return(set);
}

