


import ctypes as ct
from numpy.ctypeslib import load_library, ndpointer
import numpy as np
from os import environ


__all__ = ['energy', 'can_pair', 'to_strseq',
        'to_intseq', 'to_thepairs', 'to_dpp']

class NupackException(Exception):
    def __init__(self, msg):
        self.msg = msg
    def __str__(self):
        return 'Nupack Exception: ' + repr(msg)

libdirs = []
if 'NUPACKHOME' in environ:
    libdirs += environ['NUPACKHOME']

libdirs += ['/usr/local/lib',
        '/usr/lib']

success = False
for dname in libdirs:
    try:
        _libnupack = load_library('libnupackpfunc_s', dname)
        success = True
    except Exception as e:
        pass

if not success:
    raise NupackException('Failed to find NUPACK library. Ensure you have a shared compilation of it')

_libnupack.naEnergyPairsOrParensFullWithSym.argtypes = [
            ndpointer(dtype=np.int32, ndim=1),  # thepairs
            ct.POINTER(ct.c_char),   # parens
            ndpointer(dtype=np.int32, ndim=1),  # sequence
            ct.c_int,         # nucleic acid type
            ct.c_int,         # dangles
            ct.c_longdouble,  # temperature
            ct.c_int,         # symmetry
            ct.c_longdouble,  # sodium concentration
            ct.c_longdouble,  # magnesium concentration
            ct.c_int          # uselongsalt
        ]

_libnupack.naEnergyPairsOrParensFullWithSym.restype = ct.c_longdouble

nuc_lookup = {
        'N': 0,
        'A': 1,
        'C': 2,
        'G': 3,
        'U': 4,
        'T': 4,
        'R': 5,
        'M': 6,
        'S': 7,
        'W': 8,
        'K': 9,
        'Y': 10,
        'V': 11,
        'H': 12,
        'D': 13,
        'B': 14,
        '+': 15,
        }

rna_rev_lookup = {
        }
dna_rev_lookup = {
        }

can_pair = np.array([
    [True, True, True, True, True],
    [True, False, False, False, True],
    [True, False, False, True, False],
    [True, False, True, False, True],
    [True, True, False, True, False]
])

for k, v in nuc_lookup.items():
    if k != 'T':
        rna_rev_lookup[v] = k
    if k != 'U':
        dna_rev_lookup[v] = k

def to_dpp(thepairs, breaks=[]):
    dpp_list = []
    for i, j in enumerate(thepairs):
        if i in breaks:
            dpp_list.append('+')
        if j > i:
            dpp_list.append('(')
        elif j < 0:
            dpp_list.append('.')
        else:
            dpp_list.append(')')
    return ''.join(dpp_list)

def to_strseq(sequence, material='rna'):
    if len(sequence) == 0:
        return ""
    else:
        if material in ['dna', 'dna1998']:
            return ''.join([dna_rev_lookup[n] for n in sequence])
        else:
            return ''.join([rna_rev_lookup[n] for n in sequence])

def to_intseq(sequence):
    if type(sequence) == list:
        if len(sequence) > 0 and type(sequence[0]) == int:
            return np.array(sequence, dtype=np.int32)
        elif len(sequence) > 0 and type(sequence[0]) == list:
            if len(sequence[0]) > 0 and type(sequence[0][0]) == int:
                fullseq = sequence[0]
                for seq in sequence[1:]:
                    fullseq.append(nuc_lookup['+'])
                    fullseq += seq
                fullseq.append(-1)
                return np.array(fullseq, dtype=int32)
            else:
                raise NupackException('Unknown sequence specification')
        elif len(sequence) > 0 and type(sequence[0]) == str:
            fullseq = [nuc_lookup[n] for n in sequence[0]]
            for seq in sequence[1:]:
                fullseq.append(nuc_lookup['+'])
                fullseq += [nuc_lookup[n] for n in seq]
            fullseq.append(-1)
            return np.array(fullseq, dtype=int32)
        elif len(sequence) == 0:
            return np.array([-1], dtype=int32)
    elif type(sequence) == str:
        fullseq = [nuc_lookup[n] for n in sequence]
        fullseq.append(-1)
        return np.array(fullseq, dtype=np.int32)

    return np.array(sequence, dtype=np.int32)

def dpp_to_thepairs(structure):
    struc = []
    stack = []
    breaks = []
    i = 0
    for c in structure:
        if c == '(':
            struc.append(-5)
            stack.append(i)
            i += 1
        elif c == ')':
            j = stack[-1]
            stack.pop()
            struc.append(j)
            struc[j] = i
            i += 1
        elif c == '.':
            struc.append(-1)
            i += 1
        elif c == '+':
            breaks.append(i)

    return np.array(struc, dtype=np.int32), np.array(breaks)

def to_thepairs(structure):
    if type(structure) == list:
        if len(structure) > 0 and type(structure[0]) == int:
            return np.array(structure, dtype=np.int32)
        elif len(structure) > 0 and type(structure[0]) == str:
            structure = ''.join(structure)
        elif len(structure) == 0:
            structure = ""

    if type(structure) == str:
        return dpp_to_thepairs(structure)[0]

    return np.array(structure, dtype=np.int32)

MATERIAL_DICT = {
            'dna': 0,
            'dna1998': 0,
            'rna': 1,
            'rna1995': 1,
            'rna1999': 2,
        }

def get_material_int(matstr):
    if type(matstr) == str:
        if matstr in MATERIAL_DICT:
            return MATERIAL_DICT[matstr]
        else:
            raise NupackException('Unrecognized argument for material: '
                    + repr(matstr))
    elif type(matstr) == int:
        if matstr >= 0 and matstr <= 2:
            return matstr
        else:
            raise NupackException('Unrecognized argument for material: '
                    + repr(matstr))
    else:
        raise NupackException('Unrecognized argument for material: '
                + repr(matstr))

DANGLE_DICT = {
        'none': 0,
        'some': 1,
        'all': 2,
        }

def get_dangle_int(danglestr):
    if type(danglestr) == str:
        if danglestr in DANGLE_DICT:
            return DANGLE_DICT[danglestr]
        else:
            raise NupackException('Unrecognized argument for dangle: '
                    + repr(danglestr))
    elif type(danglestr) == int:
        if danglestr in DANGLE_DICT.values():
            return danglestr
        else:
            raise NupackException('Unrecognized argument for dangle: '
                    + repr(danglestr))

DEFAULT_TEMPERATURE = 310.15
ZERO_C_IN_K = 273.15

def energy(sequence, structure, material='rna1995', temperature=DEFAULT_TEMPERATURE,
        sodium=1.0, magnesium=0.0, symmetry=1, dangles='none', longsalt=False):

    struc = to_thepairs(structure)
    seq = to_intseq(sequence)
    matint = get_material_int(material)
    dangleint = get_dangle_int(dangles)
    longsalt = int(longsalt)

    val = _libnupack.naEnergyPairsOrParensFullWithSym(
                struc,
                None,
                seq,
                matint,
                dangleint,
                temperature - ZERO_C_IN_K,
                symmetry,
                sodium,
                magnesium,
                longsalt
            )
    return float(val)

def test_energy():
    seq = 'GGGGAAAACCCC'
    struc = '((((....))))'
    ene = energy(seq, struc)
    print(seq, struc, ene)

if __name__ == '__main__':
    test_energy()
