#!/usr/bin/env python
import sys
from optparse import OptionParser
import string

# strings for optionparser
usage_str = "usage: %prog [options] [N_SYSTEMS]"
version_str = "%prog 0.9"


header = """
material = dna
temperature = 25

# target structures
"""

structures = """
# system {tag}
structure hairpin_{tag}_1       = U{toelength} D{stemlength} U{toelength}
structure hairpin_{tag}_2       = D{stemlength} U{toelength} U{toelength}
structure initiator_{tag}_1     = U{toelength} U{stemlength}
structure initiator_{tag}_2     = U{toelength} U{stemlength}
structure detect_{tag}_1        = D{toelength} D{stemlength} + U{toelength} U{stemlength}
structure detect_{tag}_2        = U{toelength} U{stemlength} D{toelength} D{stemlength} +
structure z_hairpin_toe_{tag}_1 = U{toelength}
structure z_hairpin_toe_{tag}_2 = U{toelength}
"""

self_cross = """
structure cross_{tag}_{tag}_1_2 = U{toelength} + U{stemlength} U{toelength}
structure cross_{tag}_{tag}_2_1 = U{toelength} + U{stemlength} U{toelength}
"""

cross_structures = """
structure cross_{tag1}_{tag2}_1_1 = U{toelength} + U{stemlength} U{toelength}
structure cross_{tag1}_{tag2}_1_2 = U{toelength} + U{stemlength} U{toelength}
structure cross_{tag1}_{tag2}_2_1 = U{toelength} + U{stemlength} U{toelength}
structure cross_{tag1}_{tag2}_2_2 = U{toelength} + U{stemlength} U{toelength}
"""

domains = """
# domains for system {tag}
domain a_{tag} = N{toelength}
domain b_{tag} = N{stemlength}
domain c_{tag} = N{toelength}
"""

strand_assign = """
strand hp_{tag}_1       = a_{tag} b_{tag} c_{tag}* b_{tag}*
strand hp_{tag}_2       = b_{tag}* a_{tag}* b_{tag} c_{tag}
strand init_{tag}_1     = b_{tag}* a_{tag}*
strand init_{tag}_2     = c_{tag}* b_{tag}*
strand z_hp_toe_{tag}_1 = a_{tag}
strand z_hp_toe_{tag}_2 = c_{tag}
"""

seq_assign = """
# sequence assignment for system {tag}
hairpin_{tag}_1.seq         = hp_{tag}_1
hairpin_{tag}_2.seq         = hp_{tag}_2
initiator_{tag}_1.seq       = init_{tag}_1
initiator_{tag}_2.seq       = init_{tag}_2
detect_{tag}_1.seq          = init_{tag}_1 hp_{tag}_1
detect_{tag}_2.seq          = hp_{tag}_2 init_{tag}_2
z_hairpin_toe_{tag}_1.seq   = z_hp_toe_{tag}_1
z_hairpin_toe_{tag}_2.seq   = z_hp_toe_{tag}_2
"""

self_cross_seq_assign = """
cross_{tag}_{tag}_1_2.seq = z_hp_toe_{tag}_1 init_{tag}_2
cross_{tag}_{tag}_2_1.seq = z_hp_toe_{tag}_2 init_{tag}_1
"""

cross_seq_assign = """
cross_{tag1}_{tag2}_1_1.seq = z_hp_toe_{tag1}_1 init_{tag2}_1
cross_{tag1}_{tag2}_1_2.seq = z_hp_toe_{tag1}_1 init_{tag2}_2
cross_{tag1}_{tag2}_2_1.seq = z_hp_toe_{tag1}_2 init_{tag2}_1
cross_{tag1}_{tag2}_2_2.seq = z_hp_toe_{tag1}_2 init_{tag2}_2
"""

self_cross_names = """ cross_{tag}_{tag}_1_2 cross_{tag}_{tag}_2_1 """
cross_names = """ cross_{tag1}_{tag2}_1_1 cross_{tag1}_{tag2}_1_2 cross_{tag1}_{tag2}_2_1 cross_{tag1}_{tag2}_2_2 """

initiator_names = """ initiator_{tag}_1 initiator_{tag}_2 """
toenames = """ z_hairpin_toe_{tag}_1 z_hairpin_toe_{tag}_2 """

itube = """
tube itube_all = {initiator_list}
"""

htube = """
tube htube = {toe_list} {initiator_list} {cross_list}
"""

dtube = """
tube dtube_{tag}_1 = detect_{tag}_1
tube dtube_{tag}_2 = detect_{tag}_2
"""

itube_concs = """
itube_all.initiator_{tag}_1.conc[uM] = 1
itube_all.initiator_{tag}_2.conc[uM] = 1
"""

htube_concs = """
htube.z_hairpin_toe_{tag}_1.conc[uM] = 1
htube.z_hairpin_toe_{tag}_2.conc[uM] = 1
htube.initiator_{tag}_1.conc[uM] = 1
htube.initiator_{tag}_2.conc[uM] = 1

htube.cross_{tag}_{tag}_1_2.conc[uM] = 0.00
htube.cross_{tag}_{tag}_2_1.conc[uM] = 0.00
"""

htube_cross_concs = """
htube.cross_{tag1}_{tag2}_1_1.conc[uM] = 0.00
htube.cross_{tag1}_{tag2}_1_2.conc[uM] = 0.00
htube.cross_{tag1}_{tag2}_2_1.conc[uM] = 0.00
htube.cross_{tag1}_{tag2}_2_2.conc[uM] = 0.00
"""

dtube_concs = """
dtube_{tag}_1.detect_{tag}_1.conc[uM] = 0.01
dtube_{tag}_2.detect_{tag}_2.conc[uM] = 0.01
"""

itube_maxsize = """
itube_all.maxsize = 2
htube.maxsize = 0
"""

maxsize = """
dtube_{tag}_1.maxsize = 2
dtube_{tag}_2.maxsize = 2
"""

itube_stop_condition = """
itube_all.stop[%] = 20.0
htube.stop[%] = 20.0
"""

stop_conditions = """
dtube_{tag}_1.stop[%] = 10.0
dtube_{tag}_2.stop[%] = 10.0
hairpin_{tag}_1.stop[%] = 5.0
hairpin_{tag}_2.stop[%] = 5.0
"""

prevented_patterns = """
prevent = AAAA, CCCC, GGGG, TTTT, KKKKKK, MMMMMM, RRRRRR, SSSSSS, WWWWWW, YYYYYY
"""

def main(argv=None):
    if argv is None:
        argv = sys.argv
    parser = OptionParser(usage=usage_str,version=version_str)

    parser.add_option('-t', '--toehold', dest='toehold',
                        help="use toehold LENGTH", metavar="LENGTH",
                        type='int', default=12)
    parser.add_option('-s', '--stem', dest='stem',
                        help="use stem LENGTH", metavar="LENGTH",
                        type='int', default=24)
    parser.add_option('-o', '--output', dest='outfile',
                     help='write output to FILE', metavar='FILE',
                     type='string', default='hcr.np')

    (options,args) = parser.parse_args(argv[1:])

    # get input options
    if len(args) > 1:
        sys.stderr.write("Error: Only one argument allowed.\n")
        parser.print_help()
        return 2

    n_systems = 0
    if len(args) > 0:
        n_systems = int(args[0])
    else:
        n_systems = 1

    tags = string.ascii_uppercase[:n_systems]

    stemlength = options.stem
    toelength = options.toehold

    output_file = options.outfile

    of = open(output_file, 'w')

    of.write(header)

    for tag in tags:
        of.write(structures.format(tag=tag, stemlength=stemlength, toelength=toelength))
        of.write(self_cross.format(tag=tag, stemlength=stemlength, toelength=toelength))

    tagpairs = [(b1, b2) for b1 in tags for b2 in tags if b1 != b2]

    for tag1, tag2 in tagpairs:
        of.write(cross_structures.format(tag1=tag1, tag2=tag2,
                                         stemlength=stemlength, toelength=toelength))

    for tag in tags:
        of.write(domains.format(tag=tag, toelength=toelength, stemlength=stemlength))

    for tag in tags:
        of.write(strand_assign.format(tag=tag))

    for tag in tags:
        of.write(seq_assign.format(tag=tag))
        of.write(self_cross_seq_assign.format(tag=tag))

    for tag1, tag2 in tagpairs:
        of.write(cross_seq_assign.format(tag1=tag1, tag2=tag2))

    init_names = [initiator_names.format(tag=tag) for tag in tags]
    toe_names = [toenames.format(tag=tag) for tag in tags]
    c_names = [cross_names.format(tag1=tag1, tag2=tag2) for tag1, tag2 in tagpairs]

    c_names += [self_cross_names.format(tag=tag) for tag in tags]

    of.write(itube.format(initiator_list=' '.join(init_names)))

    of.write(htube.format(initiator_list=' '.join(init_names), toe_list=' '.join(toe_names),
                         cross_list=' '.join(c_names)))

    for tag in tags:
        of.write(dtube.format(tag=tag))

    for tag in tags:
        of.write(itube_concs.format(tag=tag))

    for tag1 in tags:
        of.write(htube_concs.format(tag=tag1))

        for tag2 in tags:
            if tag1 != tag2:
                of.write(htube_cross_concs.format(tag1=tag1, tag2=tag2))

    for tag in tags:
        of.write(dtube_concs.format(tag=tag))

    of.write(itube_maxsize)

    for tag in tags:
        of.write(maxsize.format(tag=tag))

    of.write(itube_stop_condition)
    for tag in tags:
        of.write(stop_conditions.format(tag=tag))

    of.write(prevented_patterns)

    of.close()


if __name__ == "__main__":
        sys.exit(main())

