#!/usr/bin/env python
import os
import os.path

qsub_contents = """
#!/bin/bash
#PBS -S /bin/bash
#PBS -l walltime=720:00:00
#PBS -l nodes=1:ppn=1
#PBS -l mem=1GB
#PBS -V
#PBS -q default
#PBS -d {work_dir}
#PBS -o {output_file}
#PBS -e {error_file}
#PBS -N {job_id}
cd $PBS_O_WORKDIR
./cpathway {des_file}
"""

def main():
    filenames = []

    min_gates = 1
    max_gates = 5
    n_trials = 10

    default = dict(weights=1.0, wobble=False, gccontent=(0,1.0), nuccontent=(0, 1.0), patterns="")
    kw_arg_list = [
                dict(modifier='normal'),
                dict(modifier='patterns_four', patterns='AAAA,CCCC,GGGG,UUUU'),
                dict(modifier='patterns_all', patterns='AAAA,CCCC,GGGG,UUUU,MMMMMM,KKKKKK,WWWWWW,SSSSSS,RRRRRR,YYYYYY'),
                dict(modifier='weights_50', weights=0.50),
                dict(modifier='weights_25', weights=0.25),
                dict(modifier='content4060', gccontent=(0.4, 0.6)),
                dict(modifier='content0035', nuccontent=(0.0, 0.35)),
            ]

    for kwargs in kw_arg_list:
        for dk, dv in default.iteritems():
            if not dk in kwargs:
                kwargs[dk] = dv
        for i in range(min_gates, max_gates + 1):
            filenames += make_zhang(i, n_trials, **kwargs)
            filenames += make_hcr(i, n_trials, **kwargs)
            filenames += make_LM_mech2(i, n_trials, **kwargs)
            filenames += make_peng3arm(i, n_trials, **kwargs)
            filenames += make_seelig_gates(i, n_trials, **kwargs)

    zsub_fn = "z_job_submit.sh"
    zsub_file = open(zsub_fn, 'w')

    cwd = os.getcwd()
    out_dir = os.path.join(cwd, 'output')

    qsub_dir = 'qsub_files'
    try:
        os.makedirs(qsub_dir)
    except Exception as e:
        pass
    try:
        os.makedirs(out_dir)
    except Exception as e:
        pass

    for i, fn in enumerate(filenames):
        cont = qsub_contents.format(
                work_dir=cwd,
                output_file=os.path.join(out_dir, 'run{:04d}.stdout'.format(i)),
                error_file=os.path.join(out_dir, 'run{:04d}.stderr'.format(i)),
                job_id='{:04d}'.format(i) + '_' + fn.replace('/', '-'),
                des_file=fn)

        qsub_fn = "{}/run_{:04d}.sh".format(qsub_dir, i)
        qsub_file = open(qsub_fn, 'w')
        qsub_file.write(cont)
        zsub_file.write('qsub {}\n'.format(qsub_fn))

def make_chained_seelig_gates(n_gates=1):
    filename = 'seelig_chain_{}.np'.format(n_gates)

    header = '\n'.join([
        '# Generated by testgen.py',
        '# Design based on Seelig et al. Science, 314:1585-1588, 2006',
        '# DOI: 10.1126/science.1132493',
        '# Tag: {} chained AND-gate assembly'.format(n_gates),
        'material = dna1998',
        'temperature[C] = 25.0'])

def make_filename(dtype, n_gates, i_trial, **kwargs):
    return '{}/{}/{:02d}/{:02d}.np'.format(kwargs['modifier'], dtype, \
            n_gates, i_trial)

def make_seelig_gates(n_gates=1, n_trials=1, **kwargs):
    des_type = 'seelig'

    header = '\n'.join(['# Generated by testgen.py',
        '# Design based on Seelig et al. Science, 314:1585-1588, 2006',
        '# DOI: 10.1126/science.1132493',
        '# Tag: {} orthogonal AND-gate assembly'.format(n_gates),
        'material = dna1998',
        'temperature[C] = 25.0'])

    domains = {
            'Gt': 'N6',
            'Ft': 'N6',
            'Et': 'N6',
            'Gb': 'N24',
            'Fx': 'N18',
            'Fb': 'N24',
            'Eb1': 'N12',
            'Eb2': 'N12',
            'Ex': 'N6',
            }

    strands = {
            'sG': 'Gb* Gt*',
            'sEout': 'Eb2 Eb1 Et Ex',
            'sF': 'Fx Ft* Fb*',
            'sGin': 'Gt Gb',
            'sFin': 'Fb Ft',
            }

    strucs = {
            'E_G_F': 'D12(U12 D12(+D24(U6+)))',
            'E_F': 'D12(U12 D12(+U24))',
            'G_Gin': 'D30+',
            'F_Fin': 'U18 D30+',
            'Fin': 'U30',
            'Gin': 'U30',
            'Eout': 'U36',
            }

    seqs = {
            'E_G_F': 'sEout sG sF',
            'E_F': 'sEout sF',
            'G_Gin': 'sG sGin',
            'F_Fin': 'sF sFin',
            'Fin': 'sFin',
            'Gin': 'sGin',
            'Eout': 'sEout',
            }

    on_tubes = {
            'initial_gate': 'E_G_F',
            'initial_inputs': 'Fin Gin',
            'step': 'E_F G_Gin',
            'final': 'G_Gin F_Fin Eout'
            }

    other_lines = [
            'identical {gate}Gb = {gate}Fx {gate}Ft*',
            'identical {gate}Fb = {gate}Eb2 {gate}Et {gate}Ex',
            ]

    cross_tubes = {
            'X_active_first': ['E_G_F', 'Fin Gin'],
            'X_active_second': ['E_F', 'Fin Gin'],
            }

    nodimer_tubes = {
                'dimer_gates': 'E_G_F',
                'dimer_inputs': 'Fin Gin',
            }

    kwargs['weightlist'] = [
                ('Gin', 'Gb'),
                ('Fin', 'Fb'),
                ('E_F', 'Fx'),
                ('F_Fin', 'Fx'),
                ('Eout', 'Ex'),
                ('Eout', 'Ex'),
                ('Eout', 'Eb1'),
                ('Eout', 'Eb2'),
            ]

    concs = {
                'initial_gate': 1e-9,
                'initial_inputs': 1e-7,
                'step': 1e-9,
                'final': 1e-9,
                'X_active_first': 1e-7,
                'X_active_second': 1e-7,
                'dimer_gates': 1e-7,
                'dimer_inputs': 1e-7,
            }

    sizes = {
                'initial_gate': 3,
                'initial_inputs': 2,
                'step': 3,
                'final': 3,
                'X_active_first': 3,
                'X_active_second': 3,
                'dimer_gates': 2,
                'dimer_inputs': 2,
            }

    return write_files(des_type, header, domains, strands, strucs, seqs,
            on_tubes, other_lines, cross_tubes, nodimer_tubes, concs, sizes,
            n_gates, n_trials, **kwargs)

def write_files(des_type, header, domains, strands, strucs, seqs, on_tubes,
        other_lines, cross_tubes, nodimer_tubes, concs, sizes, n_gates,
        n_trials, **kwargs):
    fns = []
    for i in range(n_trials):
        fn = make_filename(des_type, n_gates, i, **kwargs)
        fns.append(fn)
        dirname = os.path.dirname(fn)
        try:
            os.makedirs(dirname)
        except:
            pass
        write_file(fn, header, domains, strands, strucs, seqs, on_tubes,
                other_lines, cross_tubes, nodimer_tubes, concs, sizes,
                n_gates, **kwargs)
    return fns


def write_file(filename, header, domains, strands, strucs, seqs, on_tubes,
        other_lines, cross_tubes, nodimer_tubes, concs, sizes, n_gates, **kwargs):

    f = open(filename, 'w')
    f.write(header)
    f.write('\n')

    if kwargs['wobble']:
        f.write('allowwobble = true\n')

    for g_id in range(n_gates):
        gatename = 'g{}_'.format(g_id)
        lines = []
        f.write('\n\n# Gate {}\n\n'.format(gatename[:-1]))
        f.write('# domains for {}\n'.format(gatename[:-1]))
        for name, seq in domains.items():
            f.write('domain {}{} = {}\n'.format(gatename, name, seq))

        if kwargs['wobble']:
            for name, seq in domains.items():
                f.write('domain {}{}_wob = {}\n'.format(gatename, name, seq))
                f.write('complement[wobble] {}{} = {}{}_wob*\n'.format(gatename, name, gatename, name))

        gccontent = kwargs['gccontent']

        if gccontent[0] > 0.00001 or gccontent[1] < 0.9999:
            f.write('\n# GC content constraints\n')
            for name, seq in domains.items():
                f.write('match {}{}_m = {}\n'.format(gatename, name,
                    seq.replace('N', 'S')))
                f.write('{}{}_m.matchrange[frac] {}{} = [{}, {}]\n'.format(
                    gatename, name, gatename, name, gccontent[0],
                    gccontent[1]))

        nuccontent = kwargs['nuccontent']

        if nuccontent[0] > 0.00001 or nuccontent[1] < 0.9999:
            f.write('\n# nucleotide content constraints for {}\n'.format(gatename[:-1]))
            for name, seq in domains.items():
                if (seq[0] == 'N'):
                    seqlength = int(seq[1:])
                if seqlength > 2:
                    for nuc in 'ACGU':
                        f.write('match {}{}_m{} = {}\n'.format(gatename, name,
                            nuc, seq.replace('N', nuc)))
                        f.write('{}{}_m{}.matchrange[frac] {}{} = [{}, {}]\n'.format(
                            gatename, name, nuc, gatename, name,
                            nuccontent[0], nuccontent[1]))

        f.write('\n# strands for {}\n'.format(gatename[:-1]))
        for name, seq in strands.items():
            strdef = ' '.join([gatename + s for s in seq.split()])
            if kwargs['wobble']:
                strdef = strdef.replace('*', '_wob*')
            f.write('strand {}{} = {}\n'.format(gatename, name, strdef))

        f.write('\n# structures for {}\n'.format(gatename[:-1]))
        for name, struc in strucs.items():
            f.write('structure {}{} = {}\n'.format(gatename, name, struc))

        f.write('\n# sequence links for {}\n'.format(gatename[:-1]))
        for name, seq in seqs.items():
            seqdef = ' '.join([gatename + s for s in seq.split()])
            f.write('{}{}.seq = {}\n'.format(gatename, name, seqdef))


        if kwargs['weights'] < 0.9999:
            f.write('\n# structural defect weights for {}\n'.format(gatename[:-1]))
            for item in kwargs['weightlist']:
                names = [gatename + s for s in item]
                f.write('{}.weight[frac] = {}\n'.format('.'.join(names), kwargs['weights']))

        f.write('\n# on-target tubes for {}\n'.format(gatename[:-1]))
        for name, strs in on_tubes.items():
            strsdef = ' '.join([gatename + s for s in strs.split()])
            tubename = gatename + name
            f.write('tube {} = {}\n'.format(tubename, strsdef))
            for strucname in strsdef.split():
                f.write('{}.{}.conc[M] = {}\n'.format(tubename, strucname, concs[name]))
            f.write('{}.maxsize = {}\n'.format(tubename, sizes[name]))
            f.write('{}.stop[%] = 5\n'.format(tubename))
            f.write('\n')

        f.write('\n# crosstalk tubes for {}\n'.format(gatename[:-1]))
        for name, strs in cross_tubes.items():
            cur_strs = [gatename + s for s in strs[0].split()]
            has_off = False
            for g_id2 in range(n_gates):
                if g_id == g_id2:
                    continue
                has_off = True
                gatename2 = 'g{}_'.format(g_id2)
                cur_strs += [gatename2 + s for s in strs[1].split()]

            strsdef = ' '.join(cur_strs)

            tubename = gatename + name
            f.write('tube {} = {}\n'.format(tubename, strsdef))
            for strucname in strsdef.split():
                f.write('{}.{}.conc[M] = {}\n'.format(tubename, strucname, concs[name]))
            f.write('{}.maxsize = {}\n'.format(tubename, sizes[name]))
            f.write('{}.stop[%] = 5\n'.format(tubename))
            f.write('\n')

        f.write('\n# additional lines for {}\n'.format(gatename[:-1]))

        for l in other_lines:
            f.write(l.format(gate=gatename))
            f.write('\n')

        f.write('\n')

    patterns = kwargs['patterns']
    if len(patterns) > 0:
        f.write('# Prevented patterns\n')
        f.write('prevent = {}\n'.format(patterns))


    f.write('\n# dimer negative design tubes\n')
    for name, strs in nodimer_tubes.items():
        cur_strs = strs.split()
        str_names = []
        for g_id in range(n_gates):
            gatename = 'g{}_'.format(g_id)
            str_names += [gatename + s for s in cur_strs]
        f.write('tube {} = {}\n'.format(name, ' '.join(str_names)))
        for sname in str_names:
            f.write('{}.{}.conc[M] = {}\n'.format(name, sname, concs[name]))
        f.write('{}.maxsize = {}\n'.format(name, sizes[name]))
        f.write('{}.stop[%] = 5\n'.format(name))

    f.close()



def make_peng3arm(n_gates=1, n_trials=1, **kwargs):
    des_type = 'yin'
    filename = 'yin_{}.np'.format(n_gates)

    header = ['# Generated by testgen.py',
              '# Design based on Yin et al. Nature, 451:318-322, 2008',
              '# DOI: 10.1038/nature06451',
              '# Tag: {} orthogonal 3-arm junction assembly'.format(n_gates),
              'material = dna1998',
              'temperature[C] = 25.0']

    header = '\n'.join(header)

    domains = {
            'a': 'N6',
            'b': 'N6',
            'c': 'N6',
            'x': 'N6',
            'y': 'N6',
            'z': 'N6',
            }

    strands = {
            'I_s': 'y* b* x* a*',
            'A_s': 'a x b y z* c* y* b* x*',
            'At_s': 'z* c* y* b* x*',
            'B_s': 'b y c z x* a* z* c* y*',
            'Bt_s': 'x* a* z* c* y*',
            'C_s': 'c z a x y* b* x* a* z*',
            'Ct_s': 'y* b* x* a* z*',
            }

    strucs = {
            'I': 'U24',
            'At': 'U30',
            'Bt': 'U30',
            'Ct': 'U30',
            'A': 'U6 D18 U12',
            'B': 'U6 D18 U12',
            'C': 'U6 D18 U12',
            'I_A': 'D24 (U30 +)',
            'At_B': 'D24(U30 +) U6',
            'Bt_C': 'D24(U30 +) U6',
            'A_B_C': 'D24( D24(U6+) D24(U6+) ) U6',
            # 'I_A_B': 'D24 (D24(U6+)U30+)',
            # 'I_A_B_C': 'D24(D24(U6+) D24(U6+) U30 +)',
            }

    seqs = {
            'I': 'I_s',
            'At': 'At_s',
            'Bt': 'Bt_s',
            'Ct': 'Ct_s',
            'A': 'A_s',
            'B': 'B_s',
            'C': 'C_s',
            'I_A': 'A_s I_s',
            'At_B': 'B_s At_s',
            'Bt_C': 'C_s Bt_s',
            'A_B_C': 'A_s B_s C_s',
            # 'I_A_B': 'A_s B_s I_s',
            # 'I_A_B_C': 'A_s B_s C_s I_s',
            }

    kwargs['weightlist'] = [
                ('I', 'x*'),
                ('I', 'b*'),
                ('I', 'y*'),
                ('I_A', 'A_s', 'x*'),
                ('I_A', 'A_s', 'b*'),
                ('I_A', 'A_s', 'y*'),
                ('I_A', 'A_s', 'c*'),
                ('At_B', 'B_s', 'y*'),
                ('At_B', 'B_s', 'c*'),
                ('At_B', 'B_s', 'z*'),
                ('At_B', 'B_s', 'a*'),
                ('Bt_C', 'C_s', 'z*'),
                ('Bt_C', 'C_s', 'a*'),
                ('Bt_C', 'C_s', 'x*'),
                ('Bt_C', 'C_s', 'b*'),
            ]

    on_tubes = {
            'no_mis1': 'A B',
            'no_mis2': 'B C',
            'no_mis3': 'C A',
            'step1_tube': 'I_A',
            'step2_tube': 'At_B',
            'step3_tube': 'Bt_C',
            'fullassembly': 'I A_B_C'
            }

    other_lines = [
            ]

    cross_tubes = {
            'X_active_A': ['A', 'I At Bt'],
            'X_active_B': ['B', 'I At Bt'],
            'X_active_C': ['C', 'I At Bt'],
            }

    nodimer_tubes = {
            'nodimer_init': 'I At Bt Ct',
            }

    concs = {
                'no_mis1': 1e-7,
                'no_mis2': 1e-7,
                'no_mis3': 1e-7,
                'step1_tube': 1e-6,
                'step2_tube': 1e-6,
                'step3_tube': 1e-6,
                'fullassembly': 1e-9,
                'X_active_A': 1e-8,
                'X_active_B': 1e-8,
                'X_active_C': 1e-8,
                'nodimer_init': 1e-7,
            }

    sizes = {
                'no_mis1': 2,
                'no_mis2': 2,
                'no_mis3': 2,
                'step1_tube': 2,
                'step2_tube': 2,
                'step3_tube': 2,
                'fullassembly': 3,
                'X_active_A': 2,
                'X_active_B': 2,
                'X_active_C': 2,
                'nodimer_init': 2,
            }


    return write_files(des_type, header, domains, strands, strucs, seqs,
            on_tubes, other_lines, cross_tubes, nodimer_tubes, concs, sizes,
            n_gates, n_trials, **kwargs)

def make_LM_mech2(n_gates=1, n_trials=1, **kwargs):
    des_type = 'hochrein'
    header = ['# Generated by testgen.py',
              '# Design based on Hochrein et al. JACS 2013, 135, 17322-17330',
              '# DOI: 10.1021/ja404676x',
              '# Tag: {} orthogonal conditional dicer transducers'.format(n_gates),
              'material = rna1999',
              'temperature[C] = 37.0']
    header = '\n'.join(header)

    domains = {
            'a': 'N6',
            'b': 'N4',
            'c': 'N8',
            's': 'N5',
            'w': 'N2',
            'x': 'N12',
            'y': 'N4',
            'z': 'N3',
            }

    strands = {
            'Xs_s': 'a b c',
            'A_s': 'c* b* a* z* y*',
            'B_s': 'x y z a b',
            'C_s': 'w x y s a* z* y* x* w*',
            }

    strucs = {
            'Xs': 'U18',
            'A_B': 'U8 D4 D6 D3 D4(+ U12)',
            'C': 'D2 D12 D4( U5 U6 U3 )',
            'Xs_A': 'D6 D4 D8(+) U3 U4',
            'B': 'U12 U4 U3 U6 U4',
            'B_C': 'D12 D4 D3 D6 (U4 + U2 U12 U4 U5) U2'
            }

    seqs = {
            'Xs': 'Xs_s',
            'A_B': 'A_s B_s',
            'C': 'C_s',
            'Xs_A': 'Xs_s A_s',
            'B': 'B_s',
            'B_C': 'B_s C_s',
            }

    kwargs['weightlist'] = [
            ('Xs', 'b'),
            ('Xs', 'a'),
            ('B', 'y'),
            ('B', 'x'),
            ('B', 'b'),
            ('B_C', 'B_s', 'b'),
            ('B_C', 'C_s', 'w'),
            ('B_C', 'C_s', 'x'),
            ('B_C', 'C_s', 'y'),
            ('B_C', 'C_s', 's'),
            ]

    on_tubes = {
            'reactants_tube': 'A_B C',
            'step1_tube': 'Xs_A B',
            'step2_tube': 'Xs_A B_C',
            'no_B_tube': 'Xs C',
            }

    other_lines = [
                "source {gate}GFP_seq =\\\natggtgagcaagggcgaggagctgttcaccggggtggtgcccatcctggtcgagctggac\\\nggcgacgtaaacggccacaagttcagcgtgtccggcgagggcgagggcgatgccacctac\\\nggcaagctgaccctgaagttcatctgcaccaccggcaagctgcccgtgccctggcccacc\\\nctcgtgaccaccctgacctacggcgtgcagtgcttcagccgctaccccgaccacatgaag\\\ncagcacgacttcttcaagtccgccatgcccgaaggctacgtccaggagcgcaccatcttc\\\nttcaaggacgacggcaactacaagacccgcgccgaggtgaagttcgagggcgacaccctg\\\ngtgaaccgcatcgagctgaagggcatcgacttcaaggaggacggcaacatcctggggcac\\\naagctggagtacaactacaacagccacaacgtctatatcatggccgacaagcagaagaac\\\nggcatcaaggtgaacttcaagatccgccacaacatcgaggacggcagcgtgcagctcgcc\\\ngaccactaccagcagaacacccccatcggcgacggccccgtgctgctgcccgacaaccac\\\ntacctgagcacccagtccgccctgagcaaagaccccaacgagaagcgcgatcacatggtc\\\nctgctggagttcgtgaccgccgccgggatcactctcggcatggacgagctgtacaagaag\\\ncttagccatggcttcccgccggaggtggaggagcaggatgatggcacgctgcccatgtct\\\ntgtgcccaggagagcgggatggaccgtcaccctgcagcctgtgcttctgctaggatcaat\\\ngtgtag\n"
                "source {gate}dsRed = \\\nggtcagatccgctagcgctaccggtcgccaccatggcctcctccgagaacgtcatcaccg\\\nagttcatgcgcttcaaggtgcgcatggagggcaccgtgaacggccacgagttcgagatcg\\\nagggcgagggcgagggccgcccctacgagggccacaacaccgtgaagctgaaggtgacca\\\nagggcggccccctgcccttcgcctgggacatcctgtccccccagttccagtacggctcca\\\naggtgtacgtgaagcaccccgccgacatccccgactacaagaagctgtccttccccgagg\\\ngcttcaagtgggagcgcgtgatgaacttcgaggacggcggcgtggcgaccgtgacccagg\\\nactcctccctgcaggacggctgcttcatctacaaggtgaagttcatcggcgtgaacttcc\\\ncctccgacggccccgtgatgcagaagaagaccatgggctgggaggcctccaccgagcgcc\\\ntgtacccccgcgacggcgtgctgaagggcgagacccacaaggccctgaagctgaaggacg\\\ngcggccactacctggtggagttcaagtccatctacatggccaagaagcccgtgcagctgc\\\nccggctactactacgtggacgccaagctggacatcacctcccacaacgaggactacacca\\\ntcgtggagcagtacgagcgcaccgagggccgccaccacctgttcctgagatctcgagctc\\\naagcttcgaattctgcagtcgacggtaccgcgggcccgggatccaccggatctagataac\\\ntgatcataatcagccataccacatttgtagaggttttacttgctttaaaaaacctcccac\\\nacctccccctgaacctgaaacataaaatgaatgcaattgttgttgttaacttgtttattg\\\ncagcttataatggttacaaataaagcaatagcatcacaaatttcacaaataaagcatttt\\\ntttcactgcaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\n",
                "window {gate}GFP_window = {gate}a {gate}b {gate}c",
                "window {gate}dsRed_window = {gate}w {gate}x {gate}y {gate}z",
                "{gate}GFP_window.source = {gate}GFP_seq",
                "{gate}dsRed_window.source = {gate}dsRed",
            ]

    cross_tubes = {
            'X_active_tube': ['A_B C', 'Xs'],
            }

    nodimer_tubes = {
            }

    concs = {
                'reactants_tube': 1e-7,
                'step1_tube': 1e-9,
                'step2_tube': 1e-9,
                'no_B_tube': 1e-7,
                'X_active_tube': 1e-7,
            }

    sizes = {
                'reactants_tube': 3,
                'step1_tube': 3,
                'step2_tube': 3,
                'no_B_tube': 3,
                'X_active_tube': 3,
            }

    return write_files(des_type, header, domains, strands, strucs, seqs,
            on_tubes, other_lines, cross_tubes, nodimer_tubes, concs, sizes,
            n_gates, n_trials, **kwargs)

def make_hcr(n_gates=1, n_trials=1, **kwargs):
    des_type = 'dirks'

    header = ['# Generated by testgen.py',
              '# Design based on Dirks, R. M. PNAS 101, 15275-78 (2004)',
              '# DOI: 10.1073/pnas.0407024101'
              '# Tag: {} orthogonal HCR systems'.format(n_gates),
              'material = dna',
              'temperature[C] = 25.0']
    header = '\n'.join(header)
    domains = {
            'a': 'N10',
            'b': 'N24',
            'c': 'N10',
            }

    strands = {
            'i1': 'b* a*',
            'i2': 'c* b*',
            'h1': 'a b c* b*',
            'h2': 'b* a* b c',
            }

    strucs = {
            'H1': 'U10 D24 U10',
            'H2': 'D24 U10 U10',
            'I1': 'U34',
            'I2': 'U34',
            'D1': 'D34(U34+)',
            'D2': 'D34(+U34)',
            }

    seqs = {
            'H1': 'h1',
            'H2': 'h2',
            'I1': 'i1',
            'I2': 'i2',
            'D1': 'h1 i1',
            'D2': 'i2 h2',
            }

    on_tubes = {
            'D1_tube': 'D1',
            'D2_tube': 'D2',
            'I_tube': 'I1 I2',
            'H_tube': 'H1 H2',
            }

    kwargs['weightlist'] = [
               ('I1', 'b*'),
               ('I2', 'b*'),
               ('D1', 'h1', 'b*'),
               ('D2', 'h2', 'b*'),
             ]

    other_lines = [
            ]

    cross_tubes = {
            'X_active_tube': ['H1 H2', 'I1 I2'],
            }

    nodimer_tubes = {
            }

    concs = {
                'D1_tube': 1e-8,
                'D2_tube': 1e-8,
                'I_tube': 1e-8,
                'H_tube': 1e-8,
                'X_active_tube': 1e-8,
            }

    sizes = {
                'D1_tube': 2,
                'D2_tube': 2,
                'I_tube':  2,
                'H_tube':  2,
                'X_active_tube': 2,
            }

    return write_files(des_type, header, domains, strands, strucs, seqs,
            on_tubes, other_lines, cross_tubes, nodimer_tubes, concs, sizes,
            n_gates, n_trials, **kwargs)

def make_zhang(n_gates=1, n_trials=1, **kwargs):
    des_type = 'zhang'
    header = ['# Generated by testgen.py',
              '# Design based on Zhang, DY JACS 2011, 133, 1077-1086',
              '# DOI: 10.1021/ja109089q',
              '# Tag: {} orthogonal cooperative gates'.format(n_gates),
              'material = dna',
              'temperature[C] = 25.0']
    header = '\n'.join(header)

    domains = {
            'a': 'N7',
            'b': 'N18',
            'c': 'N18',
            'd': 'N7',
            'b1': 'N7',
            'b2': 'N7',
            'b3': 'N4',
            'c1': 'N14',
            'c2': 'N4',
            }

    strands = {
            'T1_s': 'a b',
            'T2_s': 'c d',
            'P_s': 'b1 b2 b3 c1 c2',
            'L_s': 'd* c* b* a*',
            'F_s': 'b3 c1',
            'RL_s': 'c1* b3* b2*',
            }

    strucs = {
            'T1': 'U25',
            'T2': 'U25',
            'D': 'U7 D18 D18(U7 +)',
            'H': 'D25(D25(+)+)',
            'P': 'U36',
            'R': 'D14 D4 (U7 +)',
            'F': 'U4 U14',
            'PR': 'D14 D4 D7 ( + U7) U4',
            }

    seqs = {
            'T1': 'T1_s',
            'T2': 'T2_s',
            # 'I1': 'T1_s P_s L_s',
            # 'I2': 'P_s T2_s L_s',
            'D': 'L_s P_s',
            'H': 'L_s T1_s T2_s',
            'P': 'P_s',
            'R': 'RL_s F_s',
            'F': 'F_s',
            'PR': 'RL_s P_s',
            }

    on_tubes = {
            'DR_tube': 'D R',
            'H_tube': 'H P',
            'I_tube': 'D T1 R',
            'J_tube': 'D T2 R',
            'T_tube': 'T1 T2 R',
            'F_tube': 'H PR F',
            }

    kwargs['weightlist'] = [
                ('T1', 'b'),
                ('T2', 'c'),
                ('P', 'b1'),
                ('P', 'b3'),
                ('P', 'c1'),
                ('P', 'c2'),
                ('F', 'F_s'),
            ]

    other_lines = [
            ]

    cross_tubes = {
            'X_active_tube': ['D R', 'T1 T2'],
            'X_report_tube': ['R', 'P'],
            }

    nodimer_tubes = {
            }

    concs = {
                'DR_tube': 1e-8,
                'H_tube': 1e-8,
                'I_tube': 1e-8,
                'J_tube': 1e-8,
                'T_tube': 1e-8,
                'F_tube': 1e-8,
                'X_active_tube': 1e-8,
                'X_report_tube': 1e-8,
                'T_dimers': 1e-8,
                'P_dimers': 1e-8,
            }

    sizes = {
                'DR_tube': 3,
                'H_tube':  3,
                'I_tube':  3,
                'J_tube':  3,
                'T_tube':  3,
                'F_tube':  3,
                'X_active_tube': 3,
                'X_report_tube': 3,
                'T_dimers': 2,
                'P_dimers': 2,
            }

    return write_files(des_type, header, domains, strands, strucs, seqs,
            on_tubes, other_lines, cross_tubes, nodimer_tubes, concs, sizes,
            n_gates, n_trials, **kwargs)

if __name__ == '__main__':
    main()
